package ru.yandex.search.salo;

import java.io.IOException;
import java.util.List;
import java.util.Map;

import ru.yandex.dbfields.OracleFields;
import ru.yandex.dbfields.PgFields;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;

public class PgEnvelope extends DatabaseEnvelope {
    public static final EnvelopeFactory FACTORY = new EnvelopeFactory() {
        @Override
        public void process(
            final EnvelopesContext context,
            final JsonMap json,
            final List<Envelope> storage)
            throws IOException, JsonException
        {
            PgEnvelope.process(context, json, storage);
        }
    };

    private final long uid;
    private final int midsCount;
    private final String mids;

    public PgEnvelope(
        final EnvelopesContext context,
        final JsonMap json,
        final int offset)
        throws IOException, JsonException
    {
        super(context, json);
        uid = json.get(PgFields.UID).asLong();
        uri.append("&uid=");
        uri.append(uid);
        uri.append("&change-type=");
        uri.append(json.get(PgFields.CHANGE_TYPE).asString());
        StringBuilder mids = new StringBuilder(", mids = [");
        JsonList changed = json.getListOrNull(PgFields.CHANGED);
        int midsCount = 0;
        if (changed != null) {
            uri.append("&changed-size=");
            uri.append(changed.size());
            for (JsonObject changeObject: changed) {
                if (changeObject.type() == JsonObject.Type.MAP) {
                    JsonMap change = changeObject.asMap();
                    String mid = change.get(PgFields.MID).asStringOrNull();
                    if (mid != null) {
                        if (midsCount != 0) {
                            mids.append(',');
                        }
                        ++midsCount;
                        mids.append(mid);
                    }
                }
            }
        }
        if (offset >= 0) {
            uri.append("&split-offset=");
            uri.append(offset);
        }
        uri.trimToSize();
        this.midsCount = midsCount;
        if (midsCount != 0) {
            this.mids = "";
        } else {
            mids.append(']');
            mids.trimToSize();
            this.mids = mids.toString();
        }
    }

    private static void process(
        final EnvelopesContext context,
        final JsonMap json,
        final List<Envelope> storage)
        throws IOException, JsonException
    {
        JsonList changed = json.getListOrNull(PgFields.CHANGED);
        if (changed == null || changed.size() < context.midsLimit()) {
            //pass throught
            storage.add(new PgEnvelope(context, json, -1));
        } else {
            //deep copy all except .changed[]
            JsonMap root = new JsonMap(json.containerFactory());
            for (Map.Entry<String, JsonObject> entry: json.entrySet()) {
                final String key = entry.getKey();
                if (!key.equals(PgFields.CHANGED)) {
                    root.put(
                        key,
                        entry.getValue().deepCopy(json.containerFactory()));
                }
            }
            int offset = 0;
            int left = changed.size();
            while (left > 0) {
                final int batchSize = Math.min(context.midsLimit(), left);
                JsonList batchArray =
                    changed.subList(offset, offset + batchSize);
                JsonMap batch = root.deepCopy();
                batch.put(PgFields.CHANGED, batchArray);
                batch.put(OracleFields.OPERATION_PART, new JsonLong(offset));
                storage.add(new PgEnvelope(context, batch, offset));
                left -= batchSize;
                offset += batchSize;
            }
        }
    }

    @Override
    public int shard() {
        return (int) (uid % Mdb.SHARDS);
    }

    @Override
    public int midsCount() {
        return midsCount;
    }

    @Override
    protected void writeAdditionalInfo(final StringBuilder sb) {
        sb.append("uid = ");
        sb.append(uid);
        sb.append(mids);
    }
}
