package ru.yandex.search.salo;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.message.BasicHttpRequest;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpRequestHandler;

import ru.yandex.collection.Pattern;
import ru.yandex.http.util.HttpExceptionConverter;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.uri.ScanningCgiParams;
import ru.yandex.search.salo.config.ImmutableMailSaloConfig;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;

public class Server extends Salo implements HttpRequestHandler {
    public static final String PG = "pg";

    private final Mdb[] mdbs;

    public Server(final ImmutableMailSaloConfig config) throws IOException {
        super(config);

        ThreadGroup threadGroup = new ThreadGroup(getThreadGroup(), "Mdbs");

        List<Mdb> mdbs = new ArrayList<>();
        MdbsContext ssdMdbsContext =
            new MdbsContext(this, threadGroup, "ora-ssd");
        MdbsContext sataMdbsContext =
            new MdbsContext(this, threadGroup, "ora-sata");
        if (config.mdblist() != null) {
            Set<String> mdbsDedup = new HashSet<>();
            for (String line: Files.readAllLines(
                    config.mdblist().toPath(),
                    StandardCharsets.UTF_8))
            {
                String mdb = line.trim();
                if (!mdb.isEmpty() && mdb.charAt(0) != '#') {
                    if (mdbsDedup.add(mdb)) {
                        MdbsContext mdbsContext;
                        if (config.ssdMdbs().matcher(mdb).matches()) {
                            mdbsContext = ssdMdbsContext;
                        } else {
                            mdbsContext = sataMdbsContext;
                        }
                        mdbs.add(
                            new Mdb(
                                mdbsContext,
                                new DatabaseContext(
                                    mdb,
                                    "mdb=" + mdb,
                                    OracleEnvelope.FACTORY),
                                config.zoolooserService()));
                    } else {
                        logger().severe("Duplicate mdb in list: " + mdb);
                    }
                }
            }
        }

        MdbsContext mdbsContext = new MdbsContext(this, threadGroup, PG);
        if (config.pgshards() != null) {
            Set<String> shards = new HashSet<>();
            for (String line: Files.readAllLines(
                    config.pgshards().toPath(),
                    StandardCharsets.UTF_8))
            {
                String shard = line.trim();
                if (!shard.isEmpty() && shard.charAt(0) != '#') {
                    if (shards.add(shard)) {
                        mdbs.add(
                            new Mdb(
                                mdbsContext,
                                new DatabaseContext(
                                    PG + shard,
                                    "mdb=pg&pgshard=" + shard,
                                    PgEnvelope.FACTORY),
                                config.pgZoolooserService()));
                    } else {
                        logger().severe("Duplicate pgshard in list: " + shard);
                    }
                }
            }
        }

        this.mdbs = mdbs.toArray(new Mdb[mdbs.size()]);

        register(
            new Pattern<>("/drop-position", false),
            this,
            RequestHandlerMapper.GET);
        registerStater(new MdbsStater(this.mdbs));
    }

    @Override
    public void start() throws IOException {
        for (Mdb mdb: mdbs) {
            mdb.start();
        }
        super.start();
    }

    @Override
    @SuppressWarnings("try")
    public void close() throws IOException {
        super.close();
        try (CloseableHttpClient msalClient = this.msalClient();
                CloseableHttpClient zoolooserClient = this.zoolooserClient())
        {
            for (Mdb mdb: mdbs) {
                mdb.close();
            }
            for (Mdb mdb: mdbs) {
                try {
                    mdb.join();
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
            }
        }
    }

    @Override
    public Map<String, Object> status(final boolean verbose) {
        Map<String, Object> status = super.status(verbose);
        if (verbose) {
            Map<String, Object> mdbsStatus = new TreeMap<>();
            for (Mdb mdb: mdbs) {
                mdbsStatus.put(mdb.name(), mdb.status());
            }
            status.put("mdbs", mdbsStatus);
        }
        return status;
    }

    @Override
    public void handle(
        final HttpRequest request,
        final HttpResponse response,
        final HttpContext context)
        throws HttpException
    {
        ScanningCgiParams params = new ScanningCgiParams(request);
        String name = params.get(
            "pgshard",
            null,
            NonEmptyValidator.INSTANCE);
        if (name == null) {
            name = params.get(
                "mdb",
                NonEmptyValidator.INSTANCE);
        } else {
            name = PG + name;
        }
        Mdb mdb = null;
        for (Mdb currentMdb: mdbs) {
            if (currentMdb.name().equals(name)) {
                mdb = currentMdb;
                break;
            }
        }
        if (mdb == null) {
            response.setStatusCode(HttpStatus.SC_NOT_FOUND);
        } else {
            HttpRequest producerRequest = new BasicHttpRequest(
                HttpGet.METHOD_NAME,
                "/_producer_drop_position?service=" + mdb.service()
                + "&producer-name=" + name
                + "&positions-count=" + mdb.workersCount()
                + "&session-timeout=" + mdb.config().sessionTimeout());
            try (CloseableHttpResponse producerResponse =
                    zoolooserClient().execute(
                        mdb.config().zoolooserConfig().host(),
                        producerRequest))
            {
                int status = producerResponse.getStatusLine().getStatusCode();
                if (status == HttpStatus.SC_OK) {
                    mdb.interrupt();
                } else {
                    throw HttpExceptionConverter.toHttpException(
                        producerRequest,
                        producerResponse);
                }
            } catch (IOException e) {
                throw HttpExceptionConverter.toHttpException(e);
            }
        }
    }

    private static class MdbsStater implements Stater {
        private final Mdb[] mdbs;

        MdbsStater(final Mdb[] mdbs) {
            this.mdbs = mdbs;
        }

        @Override
        public <E extends Exception> void stats(
            final StatsConsumer<? extends E> statsConsumer)
            throws E
        {
            int pgLockedShards = 0;
            int pgPinholes = 0;
            int mdbLockedShards = 0;
            int mdbPinholes = 0;
            for (Mdb mdb: mdbs) {
                if (mdb.name().startsWith(PG)) {
                    if (mdb.locked()) {
                        ++pgLockedShards;
                    }
                    if (mdb.pinhole()) {
                        ++pgPinholes;
                    }
                } else {
                    if (mdb.locked()) {
                        ++mdbLockedShards;
                    }
                    if (mdb.pinhole()) {
                        ++mdbPinholes;
                    }
                }
            }
            statsConsumer.stat("pg-locked-shards_ammv", pgLockedShards);
            statsConsumer.stat("pg-pinholes_ammv", pgPinholes);
            statsConsumer.stat("mdb-locked-shards_ammv", mdbLockedShards);
            statsConsumer.stat("mdb-pinholes_ammv", mdbPinholes);
        }
    }
}

