package ru.yandex.search.salo.config;

import ru.yandex.http.config.HttpHostConfig;
import ru.yandex.http.config.HttpHostConfigBuilder;
import ru.yandex.http.util.server.AbstractHttpProxyConfigBuilder;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.DurationParser;
import ru.yandex.parser.string.NonNegativeIntegerValidator;
import ru.yandex.parser.string.PositiveIntegerValidator;
import ru.yandex.stater.StaterConfig;
import ru.yandex.stater.StaterConfigBuilder;

public abstract class AbstractSaloConfigBuilder
    <T extends AbstractSaloConfigBuilder<T>>
    extends AbstractHttpProxyConfigBuilder<T>
    implements SaloConfig
{
    private HttpHostConfigBuilder msalConfig;
    private HttpHostConfigBuilder zoolooserConfig;
    private int selectLength;
    private int workerQueueLength;
    private int workersPerMdb;
    private int workersLookahead;
    private int requestsBatchSize;
    private int midsLimit;
    private long envelopesCheckInterval;
    private long lockCheckInterval;
    private long sessionTimeout;

    private StaterConfigBuilder msalResponsesStaterConfig;
    private StaterConfigBuilder minTransactionStaterConfig;
    private StaterConfigBuilder transferLagStaterConfig;
    private long activeShardsTimeFrame;

    protected AbstractSaloConfigBuilder(final SaloConfig config) {
        super(config);
        msalConfig(config.msalConfig());
        zoolooserConfig(config.zoolooserConfig());

        selectLength(config.selectLength());
        workerQueueLength(config.workerQueueLength());
        workersPerMdb(config.workersPerMdb());
        workersLookahead(config.workersLookahead());
        requestsBatchSize(config.requestsBatchSize());
        midsLimit(config.midsLimit());
        envelopesCheckInterval(config.envelopesCheckInterval());
        lockCheckInterval(config.lockCheckInterval());
        sessionTimeout(config.sessionTimeout());

        msalResponsesStaterConfig(config.msalResponsesStaterConfig());
        minTransactionStaterConfig(config.minTransactionStaterConfig());
        transferLagStaterConfig(config.transferLagStaterConfig());
        activeShardsTimeFrame(config.activeShardsTimeFrame());
    }

    protected AbstractSaloConfigBuilder(
        final IniConfig config,
        final SaloConfig defaults)
        throws ConfigException
    {
        super(config, defaults);
        msalConfig = new HttpHostConfigBuilder(
            config.section("msal"),
            defaults.msalConfig());
        IniConfig zoolooserSection = config.section("zoolooser");
        zoolooserConfig = new HttpHostConfigBuilder(
            zoolooserSection,
            defaults.zoolooserConfig());

        IniConfig salo = config.section("salo");
        selectLength = salo.get(
            "select-length",
            defaults.selectLength(),
            PositiveIntegerValidator.INSTANCE);
        workerQueueLength = salo.get(
            "worker-queue-length",
            selectLength,
            PositiveIntegerValidator.INSTANCE);
        workersPerMdb = salo.get(
            "workers-per-mdb",
            defaults.workersPerMdb(),
            PositiveIntegerValidator.INSTANCE);
        workersLookahead = salo.get(
            "workers-lookahead",
            defaults.workersLookahead(),
            NonNegativeIntegerValidator.INSTANCE);
        requestsBatchSize = salo.get(
            "requests-batch-size",
            defaults.requestsBatchSize(),
            PositiveIntegerValidator.INSTANCE);
        midsLimit = salo.get(
            "mids-limit",
            defaults.midsLimit(),
            PositiveIntegerValidator.INSTANCE);
        envelopesCheckInterval = salo.getLongDuration(
            "envelopes-check-interval",
            defaults.envelopesCheckInterval());
        lockCheckInterval = salo.getLongDuration(
            "lock-check-interval",
            defaults.lockCheckInterval());
        sessionTimeout = salo.get(
            "session-timeout",
            defaults.sessionTimeout(),
            DurationParser.POSITIVE_LONG);

        msalResponsesStaterConfig = new StaterConfigBuilder(
            config.section("msal-responses-stat"),
            defaults.msalResponsesStaterConfig());
        minTransactionStaterConfig = new StaterConfigBuilder(
            config.section("min-transaction-stat"),
            defaults.minTransactionStaterConfig());
        transferLagStaterConfig = new StaterConfigBuilder(
            config.section("transfer-lag-stat"),
            defaults.transferLagStaterConfig());
        activeShardsTimeFrame = salo.get(
            "active-shards-time-frame",
            defaults.activeShardsTimeFrame(),
            DurationParser.POSITIVE_LONG);
    }

    @Override
    public HttpHostConfigBuilder msalConfig() {
        return msalConfig;
    }

    public T msalConfig(final HttpHostConfig msalConfig) {
        this.msalConfig = new HttpHostConfigBuilder(msalConfig);
        return self();
    }

    @Override
    public HttpHostConfigBuilder zoolooserConfig() {
        return zoolooserConfig;
    }

    public T zoolooserConfig(final HttpHostConfig zoolooserConfig) {
        this.zoolooserConfig = new HttpHostConfigBuilder(zoolooserConfig);
        return self();
    }

    @Override
    public int selectLength() {
        return selectLength;
    }

    public T selectLength(final int selectLength) {
        this.selectLength = selectLength;
        return self();
    }

    @Override
    public int workerQueueLength() {
        return workerQueueLength;
    }

    public T workerQueueLength(final int workerQueueLength) {
        this.workerQueueLength = workerQueueLength;
        return self();
    }

    @Override
    public int workersPerMdb() {
        return workersPerMdb;
    }

    public T workersPerMdb(final int workersPerMdb) {
        this.workersPerMdb = workersPerMdb;
        return self();
    }

    @Override
    public int workersLookahead() {
        return workersLookahead;
    }

    public T workersLookahead(final int workersLookahead) {
        this.workersLookahead = workersLookahead;
        return self();
    }

    @Override
    public int requestsBatchSize() {
        return requestsBatchSize;
    }

    public T requestsBatchSize(final int requestsBatchSize) {
        this.requestsBatchSize = requestsBatchSize;
        return self();
    }

    @Override
    public int midsLimit() {
        return midsLimit;
    }

    public T midsLimit(final int midsLimit) {
        this.midsLimit = midsLimit;
        return self();
    }

    @Override
    public long envelopesCheckInterval() {
        return envelopesCheckInterval;
    }

    public T envelopesCheckInterval(final long envelopesCheckInterval) {
        this.envelopesCheckInterval = envelopesCheckInterval;
        return self();
    }

    @Override
    public long lockCheckInterval() {
        return lockCheckInterval;
    }

    public T lockCheckInterval(final long lockCheckInterval) {
        this.lockCheckInterval = lockCheckInterval;
        return self();
    }

    @Override
    public long sessionTimeout() {
        return sessionTimeout;
    }

    public T sessionTimeout(final long sessionTimeout) {
        this.sessionTimeout = sessionTimeout;
        return self();
    }

    @Override
    public StaterConfigBuilder msalResponsesStaterConfig() {
        return msalResponsesStaterConfig;
    }

    public T msalResponsesStaterConfig(
        final StaterConfig msalResponsesStaterConfig)
    {
        this.msalResponsesStaterConfig =
            new StaterConfigBuilder(msalResponsesStaterConfig);
        return self();
    }

    @Override
    public StaterConfigBuilder minTransactionStaterConfig() {
        return minTransactionStaterConfig;
    }

    public T minTransactionStaterConfig(
        final StaterConfig minTransactionStaterConfig)
    {
        this.minTransactionStaterConfig =
            new StaterConfigBuilder(minTransactionStaterConfig);
        return self();
    }

    @Override
    public StaterConfigBuilder transferLagStaterConfig() {
        return transferLagStaterConfig;
    }

    public T transferLagStaterConfig(
        final StaterConfig transferLagStaterConfig)
    {
        this.transferLagStaterConfig =
            new StaterConfigBuilder(transferLagStaterConfig);
        return self();
    }

    @Override
    public long activeShardsTimeFrame() {
        return activeShardsTimeFrame;
    }

    public T activeShardsTimeFrame(final long activeShardsTimeFrame) {
        this.activeShardsTimeFrame = activeShardsTimeFrame;
        return self();
    }
}

