package ru.yandex.search.sharpei;

import java.io.IOException;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.Future;
import java.util.function.Function;
import java.util.function.Supplier;

import org.apache.http.HttpHost;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import org.apache.http.nio.protocol.HttpAsyncRequestProducer;

import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.util.HttpStatusPredicates;
import ru.yandex.http.util.RequestErrorType;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;
import ru.yandex.json.parser.JsonException;

public class SharpeiClient extends AbstractAsyncClient<SharpeiClient> {
    public static final Function<Exception, RequestErrorType>
        ERROR_CLASSIFIER =
        new Function<Exception, RequestErrorType>() {
            @Override
            public RequestErrorType apply(final Exception e) {
                RequestErrorType type;
                if (e instanceof SharpeiException) {
                    if (e instanceof SharpeiNotFoundException) {
                        type = RequestErrorType.NON_RETRIABLE;
                    } else {
                        type = RequestErrorType.HOST_NON_RETRIABLE;
                    }
                } else if (e instanceof IOException
                    && e.getCause() instanceof JsonException)
                {
                    type = RequestErrorType.HOST_NON_RETRIABLE;
                } else {
                    type = RequestErrorType.ERROR_CLASSIFIER.apply(e);
                }
                return type;
            }
        };

    private static final String MODE = "&mode=";
    private static final StatusCheckAsyncResponseConsumerFactory<ConnInfo>
        CONNINFO_CONSUMER_FACTORY =
            new StatusCheckAsyncResponseConsumerFactory<>(
                HttpStatusPredicates.OK,
                ConnInfoConsumerFactory.INSTANCE);

    private final HttpHost host;
    private final Supplier<HttpAsyncRequestProducer> statRequest;

    public SharpeiClient(
        final SharedConnectingIOReactor reactor,
        final ImmutableHttpHostConfig sharpeiConfig)
    {
        super(reactor, sharpeiConfig, ERROR_CLASSIFIER);
        host = sharpeiConfig.host();
        BasicAsyncRequestProducerGenerator producerGenerator =
            new BasicAsyncRequestProducerGenerator("/v2/stat");
        statRequest = () -> producerGenerator.apply(host);
    }

    public SharpeiClient(
        final CloseableHttpAsyncClient client,
        final SharpeiClient sample)
    {
        super(client, sample);

        this.host = sample.host;
        this.statRequest = sample.statRequest;
    }

    public Future<ConnInfo> connInfo(
        final long uid,
        final SharpeiMode mode,
        final FutureCallback<? super ConnInfo> callback)
    {
        return connInfo(uid, mode, httpClientContextGenerator(), callback);
    }

    // CSOFF: ParameterNumber
    public Future<ConnInfo> connInfo(
        final long uid,
        final SharpeiMode mode,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super ConnInfo> callback)
    {
        StringBuilder sb = new StringBuilder("/conninfo?format=json&uid=");
        sb.append(uid);
        sb.append(MODE);
        sb.append(mode.name().toLowerCase(Locale.ROOT));
        return execute(
            host,
            new BasicAsyncRequestProducerGenerator(new String(sb)),
            CONNINFO_CONSUMER_FACTORY,
            contextGenerator,
            callback);
    }

    // CSOFF: ParameterNumber
    public Future<ConnInfo> orgInfo(
        final long orgId,
        final SharpeiMode mode,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super ConnInfo> callback)
    {
        StringBuilder sb =
            new StringBuilder("/org_conninfo?format=json&org_id=");
        sb.append(orgId);
        sb.append(MODE);
        sb.append(mode.name().toLowerCase(Locale.ROOT));
        return execute(
            host,
            new BasicAsyncRequestProducerGenerator(new String(sb)),
            CONNINFO_CONSUMER_FACTORY,
            contextGenerator,
            callback);
    }
    // CSON: ParameterNumber

    public Future<ShardInfo> shardInfo(
        final String shardId,
        final FutureCallback<? super ShardInfo> callback)
    {
        return shardInfo(shardId, httpClientContextGenerator(), callback);
    }

    public Future<ShardInfo> shardInfo(
        final String shardId,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super ShardInfo> callback)
    {
        // PS-3692
        BasicAsyncRequestProducerGenerator generator =
            new BasicAsyncRequestProducerGenerator("/v2/stat?shard_id=" + shardId);
        return execute(
            () -> generator.apply(host),
            new StatusCheckAsyncResponseConsumerFactory<>(
                HttpStatusPredicates.OK,
                new ShardInfoConsumerFactory(shardId)),
            contextGenerator,
            callback);
    }

    public Future<List<String>> shardsIds(
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super List<String>> callback)
    {
        return execute(
            statRequest,
            new StatusCheckAsyncResponseConsumerFactory<>(
                HttpStatusPredicates.OK,
                new ShardsInfoConsumerFactory()),
            contextGenerator,
            callback);
    }

    @Override
    protected SharpeiClient adjust(final CloseableHttpAsyncClient client) {
        return new SharpeiClient(client, this);
    }
}

