#!/usr/bin/python

import math
from optparse import OptionParser

import requests

INSTANCES_URL = 'http://api.gencfg.yandex-team.ru/trunk/groups/{group}/instances'
SEARCHMAP_ENTRY = '{service} host:{hostname},search_port_ng:{port},shards:{min_shard}-{max_shard}'


def group_hosts(group, mtn=True):
    r = requests.get(INSTANCES_URL.format(group=group), timeout=20)
    if r.status_code != 200:
        raise RuntimeError("Failed to get group instances " + group + " " + unicode(r))

    instances_json = r.json().get('instances')
    if not instances_json:
        raise RuntimeError("Empty intsances list for " + group)

    instances = []
    for instance in instances_json:
        hostname = instance['hostname']
        port = instance['port']
        if mtn:
            hostname = str(hostname).split('.')[0]
            hostname += '-' + str(group).lower().replace('_', '-')
            hostname += '-' + str(port) + '.gencfg-c.yandex.net'

        instances.append((hostname, port))

    print 'For group', group, 'instances', instances, 'Dc', instance['dc']
    return instance['dc'], instances


def generate_entry(hosts, service, min_shard, max_shard):
    for host in hosts:
        yield SEARCHMAP_ENTRY.format(
            service=service,
            hostname=host[0],
            port=str(int(host[1]) + 1),
            min_shard=min_shard,
            max_shard=max_shard)


def generate_shards_hosts(dc_hosts, replicas, min_shard, max_shard):
    used_dcs = []
    used_hosts = []

    for replica in range(replicas):
        dc_with_max_left_hosts = None
        for dc, hosts in dc_hosts.iteritems():
            if dc in used_dcs or not hosts:
                continue
            if not dc_with_max_left_hosts:
                dc_with_max_left_hosts = dc
            elif len(dc_hosts[dc_with_max_left_hosts]) < len(hosts):
                dc_with_max_left_hosts = dc

        if not dc_with_max_left_hosts:
            raise RuntimeError(
                'Not enough hosts left for [' + str(min_shard) + ',' + str(max_shard) + ')'
                + str(dc_hosts) + ' replica ' + str(replica))

        used_hosts.append(dc_hosts[dc_with_max_left_hosts].pop())
        used_dcs.append(dc_with_max_left_hosts)

    return used_hosts, dc_hosts


def generate_searchmap(service, replicas, groups, shards=65534):
    hosts_count = 0
    dcs_hosts = {}
    for group in groups:
        dc, hosts = group_hosts(group)
        dcs_hosts[dc] = hosts
        hosts_count += len(hosts)

    shards_per_replica = int(math.ceil(float(shards) / (hosts_count / replicas)))
    for i in range(0, shards, shards_per_replica):
        min_shard = i
        max_shard = min(shards - 1, i + shards_per_replica - 1)
        hosts, dcs_hosts = generate_shards_hosts(dcs_hosts, replicas, min_shard, max_shard)
        for line in generate_entry(hosts, service, min_shard, max_shard):
            yield line


if __name__ == '__main__':
    parser = OptionParser()
    parser.add_option("-s", "--service", dest="service", help="Service name")
    parser.add_option("-r", "--replicas", help="Number of replicas per shard", type=int, dest="replicas", default=2)
    parser.add_option("-g", "--groups", help="Comma separated gencfg groups", dest="groups")
    parser.add_option("-f", "--file", help="Output file", dest="output")

    (options, args) = parser.parse_args()
    if options.output:
        with open(options.output, 'w') as f:
            for line in generate_searchmap(options.service, options.replicas, str(options.groups).split(',')):
                f.write(line + '\n')
    else:
        for line in generate_searchmap(options.service, options.replicas, str(options.groups).split(',')):
            print line

