package ru.yandex.search.mail.tupita;

import java.io.IOException;
import java.text.ParseException;
import java.util.Collection;
import java.util.Collections;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.util.FilterFutureCallback;
import ru.yandex.util.string.StringUtils;

public class CleaningCallback<T> implements FutureCallback<T> {
    private final AtomicBoolean done = new AtomicBoolean(false);

    private final TupitaIndexationContext context;
    private final FutureCallback<? super T> callback;

    private Collection<String> urls = Collections.emptyList();

    public CleaningCallback(
        final TupitaIndexationContext context,
        final FutureCallback<? super T> callback)
    {
        this.context = context;
        this.callback = callback;
    }

    public FutureCallback<? super Collection<String>> adjustIndexCallback(
        final FutureCallback<? super Collection<String>> callback)
    {
        return new CleaningIndexCallback(callback);
    }

    protected void checkCleanup() {
        if (done.get()) {
            Collection<String> urls;
            synchronized (this) {
                urls = this.urls;
            }

            if (urls.isEmpty()) {
                return;
            }

            context.session().logger().info("Cleaning up");
            StringBuilder deleteQuery = new StringBuilder("/delete?text=url:(");
            deleteQuery.append(StringUtils.join(urls, "+OR+"));
            deleteQuery.append(")");
            deleteQuery.append("&prefix=");
            deleteQuery.append(context.prefix());

            try {
                context.tupita().lucene().delete(
                    deleteQuery.toString(), context);
            } catch (IOException | ParseException e) {
                context.session().logger()
                    .log(Level.WARNING, "Failed to cleanup", e);
            } finally {
                context.session().logger().info("Cleaning up finished");
            }
        }
    }

    public void indexed(final Collection<String> urls) {
        synchronized (this) {
            this.urls = urls;
        }

        checkCleanup();
    }

    @Override
    public void cancelled() {
        if (done.compareAndSet(false, true)) {
            callback.cancelled();
        }

        checkCleanup();
    }

    @Override
    public void completed(final T result) {
        if (done.compareAndSet(false, true)) {
            callback.completed(result);
        }

        checkCleanup();
    }

    @Override
    public void failed(final Exception e) {
        if (done.compareAndSet(false, true)) {
            callback.failed(e);
        }

        checkCleanup();
    }

    private final class CleaningIndexCallback
        extends FilterFutureCallback<Collection<String>>
    {
        private CleaningIndexCallback(
            final FutureCallback<? super Collection<String>> callback)
        {
            super(callback);
        }

        @Override
        public void completed(final Collection<String> urls) {
            indexed(urls);

            super.completed(urls);
        }
    }
}
