package ru.yandex.search.mail.tupita;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.nio.charset.Charset;
import java.nio.charset.CodingErrorAction;
import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.TimeUnit;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NByteArrayEntity;

import ru.yandex.charset.StringDecoder;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.nio.NByteArrayEntityFactory;
import ru.yandex.io.DecodableByteArrayOutputStream;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;

public class QueryCheckCallback
    extends AbstractProxySessionCallback<List<Collection<String>>>
{
    private static final String UID = "uid";
    private final JsonType jsonType;
    private final Charset acceptedCharset;
    private final List<Long> users;
    private final boolean debug;
    public QueryCheckCallback(
        final TupitaProxySession session,
        final String stid,
        final List<Long> users)
        throws HttpException
    {
        super(session);

        jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
        acceptedCharset = CharsetUtils.acceptedCharset(session.request());
        this.users = users;
        this.debug = session.params().getBoolean("debug", false);
    }

    @Override
    public void completed(final List<Collection<String>> queries) {
        DecodableByteArrayOutputStream out =
            new DecodableByteArrayOutputStream();
        if (users.size() != queries.size()) {
            failed(
                new Exception(
                    "Users count is " + users.size()
                        + " responses got " + queries.size()));
            return;
        }

        Long lastUid = 0L;
        try (OutputStreamWriter outWriter =
                 new OutputStreamWriter(
                     out,
                     acceptedCharset.newEncoder()
                         .onMalformedInput(CodingErrorAction.REPLACE)
                         .onUnmappableCharacter(CodingErrorAction.REPLACE));
             JsonWriter writer = jsonType.create(outWriter))
        {
            writer.startObject();
            writer.key("result");
            writer.startArray();
            for (int i = 0; i < users.size(); i++) {
                lastUid = users.get(i);
                writer.startObject();
                writer.key(UID);
                writer.value(users.get(i));
                writer.key("matched_queries");
                writer.value(queries.get(i));
                writer.endObject();
                if (debug) {
                    session.logger().info(
                        users.get(i).toString() + ' '
                            + queries.get(i).toString());
                }
            }
            writer.endArray();
            writer.endObject();
        } catch (IOException e) {
            failed(e);
            return;
        }

        long mts = System.currentTimeMillis();
        long ts = TimeUnit.MILLISECONDS.toSeconds(mts);

        session.connection().setSessionInfo(
            "timestamp",
            Long.toString(ts));
        session.connection().setSessionInfo(
            "module",
            "tupita-filter-test");
        session.connection().setSessionInfo(
            "uid",
            Long.toString(lastUid));
        StringDecoder decoder;
        if (StandardCharsets.UTF_8.equals(acceptedCharset)) {
            decoder = StringDecoder.UTF_8.get();
        } else {
            decoder = new StringDecoder(1024, acceptedCharset);
        }
        session.connection().setSessionInfo(
            "result",
            out.processWith(decoder));

        NByteArrayEntity entity =
            out.processWith(NByteArrayEntityFactory.INSTANCE);
        entity.setContentType(
            ContentType.APPLICATION_JSON.withCharset(acceptedCharset)
                .toString());
        session.response(HttpStatus.SC_OK, entity);
    }
}
