package ru.yandex.search.mail.tupita;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NByteArrayEntity;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.protocol.HttpContext;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.NByteArrayEntityFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.DecodableByteArrayOutputStream;
import ru.yandex.json.dom.JsonBoolean;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.search.document.mail.MailMetaInfo;
import ru.yandex.search.mail.tupita.light.NewFormatSequentialParserHandler;
import ru.yandex.search.mail.tupita.light.UserQueryChecker;

public class QueryCheckHandler extends AbstractQueryCheckHandler {
    private static final int FAT_QUERIES_THRSH = 80;

    // new format
    private final NewFormatSequentialParserHandler newSequentialParseHandler;

    public QueryCheckHandler(final Tupita tupita) {
        super(tupita);

        this.newSequentialParseHandler =
            new NewFormatSequentialParserHandler(tupita);
    }

    protected HttpEntity prepareFatRequest(
        final JsonMap message,
        final JsonList queries)
        throws IOException
    {
        DecodableByteArrayOutputStream out =
            new DecodableByteArrayOutputStream();
        JsonWriterBase writer = JsonType.DOLLAR.create(out);
        writer.startObject();
        writer.key(MESSAGE);
        message.writeValue(writer);
        writer.key(QUERIES);
        queries.writeValue(writer);
        writer.endObject();
        writer.close();

        NByteArrayEntity entity =
            out.processWith(NByteArrayEntityFactory.INSTANCE);
        entity.setContentType(
            ContentType.APPLICATION_JSON.withCharset(StandardCharsets.UTF_8)
                .toString());
        return entity;
    }

    //CSOFF: ParameterNumber
    protected void sendFats(
        final List<JsonMap> users,
        final JsonMap message,
        final ProxySession session,
        final FutureCallback<List<Collection<String>>> callback)
        throws JsonException, IOException
    {
        MultiFutureCallback<Collection<String>> mfcb =
            new MultiFutureCallback<>(callback);
        AsyncClient fatClient =
            tupita.fatCheckClient().adjust(
                session.context());
        Supplier<? extends HttpClientContext> contextGenerator =
            session.listener().createContextGeneratorFor(
                fatClient);

        for (JsonMap user : users) {
            JsonList queries = user.getList(QUERIES);
            session.logger().info(
                "Going fat with  queries " + queries.size());
            long uid = user.getLong(UID);

            String uri =
                "/fat-check?uid=" + uid
                    + "&stid=" + message.getString(MailMetaInfo.STID);
            fatClient.execute(
                tupita.fatCheckHost(),
                new BasicAsyncRequestProducerGenerator(
                    uri,
                    prepareFatRequest(message, queries)),
                StringCollectionConsumerFactory.DEFAULT_OK,
                contextGenerator,
                mfcb.newCallback());
        }

        mfcb.done();
    }

    // CSOFF: ReturnCount
    private void handleUsers(
        final ProxySession session,
        final JsonMap message,
        final List<JsonMap> users,
        final FutureCallback<List<Collection<String>>> callback)
        throws JsonException, HttpException, IOException
    {
        Header contentLengthHeader =
            session.request().getFirstHeader(HttpHeaders.CONTENT_LENGTH);
        if (contentLengthHeader != null) {
            long requestLength =
                Long.parseLong(contentLengthHeader.getValue());
            long lengthPerUser = 0;
            if (users.size() > 0) {
                lengthPerUser = requestLength / users.size();
            }

            long sum = 0;
            for (JsonMap user : users) {
                JsonList queries = user.getList(QUERIES);
                sum += queries.size();
            }

            if (sum > FAT_QUERIES_THRSH || lengthPerUser
                > tupita.config().fatRequestLengthThreshold())
            {
                session.logger().info(
                    "Fat request, ContentLength is " + requestLength
                        + " Queries is " + sum);
                sendFats(users, message, session, callback);
                return;
            }
        }

        long prefix = tupita.nextPrefix();

        TupitaMailMetaInfo meta =
            parseMessage(
                session,
                message,
                prefix);

        TupitaIndexationContext indexationContext =
            new TupitaIndexationContext(tupita, session, meta);

        long qrscnt = 0;
        for (JsonMap user : users) {
            JsonList queries = user.getList(QUERIES);
            qrscnt += queries.size();
        }

        if (!tupita.getOrFail(qrscnt)) {
            session.response(
                YandexHttpStatus.SC_TOO_MANY_REQUESTS,
                "Too many queries to parse");
            return;
        }

        CleaningCallback<List<Collection<String>>> cleanCallback
            = new CleaningCallback<>(indexationContext, callback);

        //we are limiting number of queries for parsing
        LimitReleasingCallback<List<Collection<String>>> limiteReleaser =
            new LimitReleasingCallback<>(
                tupita,
                cleanCallback,
                qrscnt);

        MultiFutureCallback<Collection<String>> usersCallbacks =
            new MultiFutureCallback<>(limiteReleaser);

        List<FutureCallback<? super Collection<String>>> indexWaiters =
            new ArrayList<>(users.size());

        boolean needTikaite = false;
        for (JsonMap user : users) {
            JsonList queries = user.getList(QUERIES);

            Collection<? extends TupitaQuery> tupitaQueries;
            try {
                tupitaQueries =
                    newSequentialParseHandler.handle(
                        indexationContext,
                        queries);
            } catch (Exception e) {
                usersCallbacks.failed(e);
                return;
            }

            UserQueryChecker checker =
                new UserQueryChecker(
                    indexationContext,
                    usersCallbacks.newCallback(),
                    tupitaQueries);

            needTikaite |=
                newSequentialParseHandler.needTikaite(
                    tupitaQueries);
            indexWaiters.add(checker);
        }

        //all queries sequential parsed, releasing queries limiter
        limiteReleaser.checkAndRelease();

        if (!indexWaiters.isEmpty()) {
            TupitaIndexer indexer;
            if (needTikaite) {
                indexer = new TikaiteTupitaIndexer(tupita);
            } else {
                indexer = new TikaiteLessTupitaIndexer(tupita);
            }

            indexer.index(
                indexationContext,
                cleanCallback.adjustIndexCallback(
                    new MultiIndexationCallback(indexWaiters)));
        }

        usersCallbacks.done();
    }
    //CSON: ParameterNumber

    @Override
    public void handle(
        final JsonObject data,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException, IOException
    {
        TupitaProxySession session =
            new TupitaProxySession(tupita, exchange, context);

        long mts = System.currentTimeMillis();
        long ts = TimeUnit.MILLISECONDS.toSeconds(mts);

        session.connection().setSessionInfo(
            "timestamp",
            Long.toString(ts));
        session.connection().setSessionInfo(
            "module",
            "tupita-subscriptions");
        session.connection().setSessionInfo(
            "uid",
            "0");
        try {
            JsonMap root = data.asMap();
            JsonMap message = root.getMap(MESSAGE);
            String stid = message.getString(MailMetaInfo.STID);
            boolean messageSpam = message.getBoolean(SPAM, false);
            JsonList users = root.getList(USERS);
            if (users.size() == 0) {
                session.logger().warning("No users in request");
                QueryCheckCallback callback =
                    new QueryCheckCallback(
                        session,
                        stid,
                        Collections.emptyList());
                callback.completed(Collections.emptyList());
                return;
            }

            //new format
            List<JsonMap> personalSpamUsers = new ArrayList<>(users.size());
            List<JsonMap> otherUsers = new ArrayList<>(users.size());

            for (JsonObject entry : users) {
                JsonMap entryMap = entry.asMap();
                if (entryMap.containsKey(SPAM)) {
                    if (entryMap.getBoolean(SPAM) != messageSpam) {
                        personalSpamUsers.add(entryMap);
                        continue;
                    }
                }

                otherUsers.add(entryMap);
            }

            if (personalSpamUsers.size() == 0) {
                List<Long> uids = new ArrayList<>(users.size());
                for (JsonMap user: otherUsers) {
                    uids.add(user.getLong(UID));
                }

                handleUsers(
                    session,
                    message,
                    otherUsers,
                    new QueryCheckCallback(session, stid, uids));
            } else if (otherUsers.size() == 0) {
                session.logger().info("Only users with personal spam");
                List<Long> uids = new ArrayList<>(users.size());
                for (JsonMap user: personalSpamUsers) {
                    uids.add(user.getLong(UID));
                }

                //invert spam flag
                if (!messageSpam) {
                    message.replace(SPAM, JsonBoolean.TRUE);
                } else {
                    message.replace(SPAM, JsonBoolean.FALSE);
                }

                handleUsers(
                    session,
                    message,
                    personalSpamUsers,
                    new QueryCheckCallback(session, stid, uids));
            } else {
                List<Long> uids = new ArrayList<>(users.size());
                for (JsonMap user: otherUsers) {
                    uids.add(user.getLong(UID));
                }

                session.logger().info(
                    "personal spam users in query, normal are " + uids);

                for (JsonMap user: personalSpamUsers) {
                    uids.add(user.getLong(UID));
                }

                QueryCheckCallback callback =
                    new QueryCheckCallback(session, stid, uids);

                DoubleFutureCallback<
                    List<Collection<String>>,
                        List<Collection<String>>> aggregateCb =
                    new DoubleFutureCallback<>(
                        new WithPersonalSpamCallback(callback));

                handleUsers(
                    session,
                    message,
                    otherUsers,
                    aggregateCb.first());

                //invert spam flag
                if (!messageSpam) {
                    message.replace(SPAM, JsonBoolean.TRUE);
                } else {
                    message.replace(SPAM, JsonBoolean.FALSE);
                }

                handleUsers(
                    session,
                    message,
                    personalSpamUsers,
                    aggregateCb.second());
            }
        } catch (JsonException e) {
            throw new BadRequestException(
                "Unable to parse message "
                    + JsonType.HUMAN_READABLE.toString(data),
                e);
        }
    }
    // CSON: ReturnCount

    private static final class WithPersonalSpamCallback
        implements FutureCallback<Map.Entry<List<Collection<String>>,
        List<Collection<String>>>>
    {
        private final QueryCheckCallback callback;

        private WithPersonalSpamCallback(final QueryCheckCallback callback) {
            this.callback = callback;
        }

        @Override
        public void completed(
            final Map.Entry<List<Collection<String>>,
                List<Collection<String>>> entry)
        {
            List<Collection<String>> result = new ArrayList<>(
                entry.getKey().size() + entry.getValue().size());
            result.addAll(entry.getKey());
            result.addAll(entry.getValue());
            callback.completed(result);
        }

        @Override
        public void failed(final Exception ex) {
            callback.failed(ex);
        }

        @Override
        public void cancelled() {
            callback.cancelled();
        }
    }
}
