package ru.yandex.search.mail.tupita;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.nio.charset.StandardCharsets;
import java.text.ParseException;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.logging.Level;

import org.apache.http.Header;
import org.apache.http.HeaderIterator;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.DecodableByteArrayOutputStream;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.stater.RequestInfo;
import ru.yandex.util.timesource.TimeSource;

public class TikaiteTupitaIndexer implements TupitaIndexer {
    private static final AllDataInTwoPartsDocumentProvider DOCUMENT_PROVIDER =
        new AllDataInTwoPartsDocumentProvider();

    protected final Tupita tupita;

    public TikaiteTupitaIndexer(final Tupita tupita) {
        this.tupita = tupita;
    }

    protected void logDebugRequest(
        final PrefixedLogger logger,
        final byte[] data)
    {
        String dataStr = new String(data, StandardCharsets.UTF_8);
        try {
            logger.info(
                JsonType.HUMAN_READABLE.toString(
                    TypesafeValueContentHandler.parse(dataStr)));
        } catch (JsonException je) {
            logger.log(
                Level.WARNING,
                "Bad indexing data " + dataStr,
                je);
        }
    }

    protected void index(
        final TupitaIndexationContext context,
        final byte[] data)
        throws IOException, ParseException
    {
        long startTime = TimeSource.INSTANCE.currentTimeMillis();

        if (context.debugRequest()) {
            logDebugRequest(context.session().logger(), data);
        }

        tupita.lucene().index(data, context);
        context.tupita().indexStater().accept(
            new RequestInfo(
                TimeSource.INSTANCE.currentTimeMillis(),
                HttpStatus.SC_OK,
                startTime,
                startTime,
                0L,
                data.length));
        context.session().logger().info("Request message indexed");
    }

    // CSOFF: ParameterNumber
    protected void index(
        final TupitaIndexationContext context,
        final List<String> urls,
        final byte[] data,
        final FutureCallback<? super Collection<String>> callback)
    {
        try {
            index(context, data);
            callback.completed(urls);
        } catch (IOException | ParseException e) {
            context.session().logger().log(
                Level.WARNING,
                "Failed to index message ",
                e);
            callback.failed(e);
        } catch (Exception e) {
            context.session().logger().log(
                Level.WARNING,
                "Uncaught index exception",
                e);

            context.session().logger().info(
                "Bad data " + new String(data, StandardCharsets.UTF_8));
            callback.failed(new BadRequestException(e));
        }
    }
    // CSON: ParameterNumber

    @Override
    public void index(
        final TupitaIndexationContext context,
        final FutureCallback<? super Collection<String>> callback)
    {
        if (context.stid() == null) {
            callback.failed(new BadRequestException("No stid supplied"));
            return;
        }

        AsyncClient client =
            tupita.tikaiteClient().adjust(context.session().context());

        BasicAsyncRequestProducerGenerator producerGenerator =
            new BasicAsyncRequestProducerGenerator(
                "/mail/handler?json-type=dollar&fast-mode&stid="
                    + context.stid());
        context.logger().info("stid: " + context.stid());

        String tvm2Ticket = Objects.toString(
            context.session().headers().getLastOrNull(
                YandexHeaders.X_YA_SERVICE_TICKET),
            tupita.tvm2Ticket());

        producerGenerator.addHeader(
            YandexHeaders.X_YA_SERVICE_TICKET,
            tvm2Ticket);
        producerGenerator.addHeader(YandexHeaders.X_SRW_NAMESPACE, "MAIL");
        producerGenerator.addHeader(YandexHeaders.X_SRW_KEY_TYPE, "STID");
        producerGenerator.addHeader(YandexHeaders.X_SRW_KEY, context.stid());
        client.execute(
            tupita.tikaiteHost(),
            producerGenerator,
            KeepHeadersJsonAsyncDomConsumerFactory.OK,
            context.session().listener().createContextGeneratorFor(client),
            new TupitaTikaiteCallback(context, callback));
    }

    private final class TupitaTikaiteCallback
        extends AbstractFilterFutureCallback<
        Map.Entry<Object, HeaderIterator>, Collection<String>>
    {
        private final TupitaIndexationContext context;

        private TupitaTikaiteCallback(
            final TupitaIndexationContext context,
            final FutureCallback<? super Collection<String>> callback)
        {
            super(callback);

            this.context = context;
        }

        private String extractRequestId(final HeaderIterator headerIterator) {
            while (headerIterator.hasNext()) {
                Header header = headerIterator.nextHeader();
                if (YandexHeaders.X_REQUEST_ID.equalsIgnoreCase(
                    header.getName()))
                {
                    return header.getValue();
                }
            }

            return "None";
        }

        @Override
        public void completed(
            final Map.Entry<Object, HeaderIterator> tikaiteResult)
        {
            context.session().logger().info(
                "Tikaite request id "
                    + extractRequestId(tikaiteResult.getValue()));

            DecodableByteArrayOutputStream out =
                new DecodableByteArrayOutputStream();
            List<String> urls;
            try (JsonWriter writer = JsonType.NORMAL.create(
                new OutputStreamWriter(out, tupita.lucene().charset())))
            {
                urls =
                    DOCUMENT_PROVIDER.apply(
                        tikaiteResult.getKey(),
                        writer,
                        context);
            } catch (IOException | JsonUnexpectedTokenException e) {
                callback.failed(e);
                return;
            }

            index(context, urls, out.toByteArray(), callback);
        }
    }
}
