package ru.yandex.search.mail.tupita;

import java.io.IOException;
import java.net.URISyntaxException;
import java.security.GeneralSecurityException;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.message.BasicHttpRequest;
import org.apache.lucene.search.Query;

import ru.yandex.client.tvm2.Tvm2TicketRenewalTask;
import ru.yandex.collection.Pattern;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.json.parser.JsonException;
import ru.yandex.ljinx.Ljinx;
import ru.yandex.mail.search.subscriptions.SubscriptionsStatusHandler;
import ru.yandex.multistarter.MultiStarter;
import ru.yandex.search.mail.tupita.admin360.ConditionsConverterHandler;
import ru.yandex.search.mail.tupita.config.ImmutableTupitaConfig;
import ru.yandex.search.mail.tupita.fat.FatQueryCheckHandler;
import ru.yandex.search.proxy.universal.UniversalSearchProxy;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;
import ru.yandex.stater.RequestsStater;

public class Tupita extends UniversalSearchProxy<ImmutableTupitaConfig> {
    private static final long MILLIS_MULTIPLIER = 10;

    private final AsyncClient tikaiteClient;
    private final AsyncClient fatCheckClient;

    private final HttpHost tikaiteHost;

    private final Tvm2TicketRenewalTask tvm2RenewalTask;
    private final TupitaLucene lucene;
    private final RequestsStater queryParseStater;
    private final RequestsStater searchStater;
    private final RequestsStater indexStater;
    private final TimeFrameQueue<Integer> sequentialQueriesParsed;
    private final TimeFrameQueue<Integer> batchedParsingRequests;
    private final TimeFrameQueue<Integer> queryParseError;

    private final AtomicLong prefixGenerator =
        new AtomicLong(System.currentTimeMillis() * MILLIS_MULTIPLIER);

    private final QueriesLimiter qpLimiter;

    public Tupita(
        final ImmutableTupitaConfig config)
        throws GeneralSecurityException,
        HttpException,
        IOException,
        JsonException,
        URISyntaxException
    {
        super(config);

        this.tikaiteClient =
            client("Tikaite", config.tikaiteConfig())
                .adjustStater(
                    config.staters(),
                    new RequestInfo(
                        new BasicHttpRequest(
                            RequestHandlerMapper.GET,
                            "/tikaite")));

        this.fatCheckClient =
            client("FatCheck", config.fatcheckerConfig());

        this.tikaiteHost = config.tikaiteConfig().host();

        try {
            this.lucene = new TupitaLucene(config);
        } catch (Exception e) {
            throw new IOException("Failed to start lucene ", e);
        }

        tvm2RenewalTask = new Tvm2TicketRenewalTask(
            logger().addPrefix("tvm2"),
            serviceContextRenewalTask,
            config.tvm2ClientConfig());

        queryParseStater =
            config.staters().preparedStaters().get(
                new RequestInfo(
                    new BasicHttpRequest(
                        RequestHandlerMapper.GET,
                        "/queryParse")));
        registerStater(queryParseStater);

        searchStater =
            config.staters().preparedStaters().get(
                new RequestInfo(
                    new BasicHttpRequest(
                        RequestHandlerMapper.GET,
                        "/luceneSearch")));
        registerStater(searchStater);

        indexStater =
            config.staters().preparedStaters().get(
                new RequestInfo(
                    new BasicHttpRequest(
                        RequestHandlerMapper.GET,
                        "/luceneIndex")));
        registerStater(indexStater);

        sequentialQueriesParsed =
            new TimeFrameQueue<>(config.metricsTimeFrame());

        registerStater(
            new PassiveStaterAdapter<>(
                sequentialQueriesParsed,
                new NamedStatsAggregatorFactory<>(
                    "queries-sequential-parsed_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));

        this.batchedParsingRequests =
            new TimeFrameQueue<>(config.metricsTimeFrame());

        registerStater(
            new PassiveStaterAdapter<>(
                batchedParsingRequests,
                new NamedStatsAggregatorFactory<>(
                    "batched-parsing-requests_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));

        queryParseError = new TimeFrameQueue<>(config.metricsTimeFrame());
        registerStater(
            new PassiveStaterAdapter<>(
                queryParseError,
                new NamedStatsAggregatorFactory<>(
                    "query-parse-errors_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));

        this.register(
            new Pattern<>("/check", true),
            new QueryCheckHandler(this),
            RequestHandlerMapper.POST);

        this.register(
            new Pattern<>("/fat-check", true),
            new FatQueryCheckHandler(this),
            RequestHandlerMapper.POST);

        this.register(
            new Pattern<>("/api/async/mail/subscriptions/status", false),
            new SubscriptionsStatusHandler(this, config.subscriptionsPumpkin()),
            RequestHandlerMapper.GET,
            RequestHandlerMapper.POST);
        this.register(
            new Pattern<>("/api/mail/conditions/convert", false),
            new ConditionsConverterHandler(this),
            RequestHandlerMapper.POST);

        this.registerStater(lucene);
        logger().info(
            "We depending on ljinx " + Ljinx.class.getSimpleName());
        logger().info(
            "We depending on multistarter"
                + MultiStarter.class.getSimpleName());

        //qpLimiter = new BasicQueriesLimiter(LIMIT);

        qpLimiter = new FakeQueriesLimiter();
    }

    public TupitaLucene lucene() {
        return lucene;
    }

    public String tvm2Ticket() {
        return tvm2RenewalTask.ticket();
    }

    public AsyncClient tikaiteClient() {
        return tikaiteClient;
    }

    public HttpHost tikaiteHost() {
        return tikaiteHost;
    }

    public AsyncClient fatCheckClient() {
        return fatCheckClient;
    }

    public HttpHost fatCheckHost() {
        return config().fatcheckerConfig().host();
    }

    public RequestsStater queryParseStater() {
        return queryParseStater;
    }

    public RequestsStater searchStater() {
        return searchStater;
    }

    public RequestsStater indexStater() {
        return indexStater;
    }

    public TimeFrameQueue<Integer> sequentialParsedQueries() {
        return sequentialQueriesParsed;
    }

    public void queryParseError() {
        queryParseError.accept(1);
    }

    public void batchedParsing() {
        batchedParsingRequests.accept(1);
    }

    public long nextPrefix() {
        return prefixGenerator.incrementAndGet();
    }

    @Override
    public void start() throws IOException {
        super.start();

        tvm2RenewalTask.start();
    }

    @Override
    public void close() throws IOException {
        super.close();

        tvm2RenewalTask.cancel();
        lucene.close();
    }

    @Override
    public Map<String, Object> status(final boolean verbose) {
        Map<String, Object> status = super.status(verbose);
        status.put("queries-limiter", qpLimiter.get());
        return status;
    }

    public Query parseQueryWithPolicy(
        final TupitaIndexationContext context,
        final String id,
        final String text)
        throws Exception
    {
        Query query = null;
        try {
            query = lucene.parseQuery(context.session(),
                text,
                context.longPrefix());
            if (context.debugRequest()) {
                context.session().logger().info("Raw query: " + text);
                context.session().logger().info(
                    "Parsed query: " + query.toString());
            } else {
                query.toString();
            }
        } catch (Exception e) {
            StringBuilder logsb = new StringBuilder(
                "Failed to parse query ");
            logsb.append(context.prefix());
            logsb.append(' ');
            logsb.append(id);
            logsb.append(' ');
            logsb.append(text);
            context.logger().log(Level.WARNING, logsb.toString(), e);
            if (QueryParseErrorPolicy.STRICT
                == context.queryParseErrorPolicy())
            {
                Exception finalException = e;
                if (query != null && e instanceof NullPointerException) {
                    finalException =
                        new BadRequestException(logsb.toString(), e);
                }

                throw finalException;
            } else {
                query = null;
                queryParseError();
            }
        }

        return query;
    }

    public boolean getOrFail(final long values) {
        if (qpLimiter.acquire(values)) {
            logger().info("Limit acquired " + qpLimiter.get());
            return true;
        }

        logger().warning("Limit queries exceeded " + values);
        return false;
    }

    public void releaseLimit(final long values) {
        qpLimiter.release(values);

        logger().info("Released limiter, " + qpLimiter.get());
    }
}
