package ru.yandex.search.mail.tupita;

import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.logging.Level;

import org.apache.http.message.BasicHttpRequest;
import org.apache.lucene.queryParser.QueryParser;
import org.apache.lucene.search.Query;
import org.apache.lucene.util.Version;

import ru.yandex.concurrent.NamedThreadFactory;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.logger.ImmutableLoggersConfig;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.msearch.Config;
import ru.yandex.msearch.DatabaseManager;
import ru.yandex.msearch.Index;
import ru.yandex.msearch.IndexDispatcher;
import ru.yandex.msearch.JsonDeleteMessage;
import ru.yandex.msearch.JsonUpdateMessage;
import ru.yandex.msearch.Message;
import ru.yandex.msearch.MessageContext;
import ru.yandex.msearch.MessageQueueId;
import ru.yandex.msearch.NewSearchRequest;
import ru.yandex.msearch.PrefixingAnalyzerWrapper;
import ru.yandex.msearch.QueueShard;
import ru.yandex.msearch.SearchRequestBase;
import ru.yandex.msearch.SearchResultsConsumer;
import ru.yandex.msearch.Searcher;
import ru.yandex.msearch.collector.CollectorFactory;
import ru.yandex.msearch.collector.DocCollector;
import ru.yandex.msearch.collector.SortedCollectorFactory;
import ru.yandex.msearch.collector.YaDoc3;
import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.msearch.util.Compress;
import ru.yandex.msearch.util.IOStats;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.queryParser.QueryParserFactory;
import ru.yandex.search.NullScorerFactory;
import ru.yandex.search.YandexScorerFactory;
import ru.yandex.search.mail.tupita.config.ImmutableTupitaConfig;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;

public class TupitaLucene implements GenericAutoCloseable<IOException>, Stater {
    private static final Set<String> GET_FIELDS = Collections.singleton("url");
    private static final String TYPE = "type";

    private static final long MB = 1024L;

    private static final QueueShard SHARD = new QueueShard("tupita", -1);
    private static final MessageQueueId QUEUE_ID =
        new MessageQueueId(
            QueueShard.MAGIC_QUEUEID,
            QueueShard.MAGIC_QUEUEID,
            true);

    protected final Charset luceneCharset = StandardCharsets.UTF_8;
    protected final IndexDispatcher dispatcher;

    private final Index index;
    private final Config daemonConfig;
    private final CollectorFactory docCollectorFactory;
    private final EmptyResultConsumer consumer = new EmptyResultConsumer();
    private final QueryParserFactory parserFactory;
    private final YandexScorerFactory scorerFactory;
    private final ThreadPoolExecutor queryParserExecutor;
    private final TimeFrameQueue<Long> parsedQueries;
    private final long maxMemInDoc;
    private final List<Stater> staters = new ArrayList<>();

    public TupitaLucene(
        final ImmutableTupitaConfig tupitaConfig)
        throws Exception
    {
        final IniConfig iniConfig = patchConfig(tupitaConfig.luceneConfig());
        final ImmutableLoggersConfig loggers = tupitaConfig.loggers();
        PrefixedLogger fullLogger =
            loggers.preparedLoggers().get(
                new RequestInfo(
                    new BasicHttpRequest(
                        RequestHandlerMapper.GET,
                        "/lucene/full")));
        PrefixedLogger indexLogger =
            loggers.preparedLoggers().get(
                new RequestInfo(
                    new BasicHttpRequest(
                        RequestHandlerMapper.GET,
                        "/lucene/index")));

        daemonConfig = new Config(iniConfig);
        DatabaseConfig config =
            daemonConfig.databasesConfigs().get(DatabaseManager.DEFAULT_DATABASE);

        maxMemInDoc = config.maxMemDocs() * MB;
        index = new Index(
            config.indexPath(),
            daemonConfig,
            config,
            fullLogger,
            indexLogger,
            null);
        docCollectorFactory = new SortedCollectorFactory();
        parserFactory = new NullScorerFactory(config);
        scorerFactory = new NullScorerFactory(config);
        this.dispatcher = new IndexDispatcher(index, false, fullLogger);
        fullLogger.info(
            "BatchQueueSize " + tupitaConfig.batchParseQueueSize());
        fullLogger.info(
            "BatchCoreThreads " + tupitaConfig.coreQueryParsingThreads());
        fullLogger.info(
            "BatchMaxThreads " + tupitaConfig.maxQueryParsingThreads());
        fullLogger.info(
            "BatchSize " + tupitaConfig.fatRequestLengthThreshold());
        this.queryParserExecutor =
            new ThreadPoolExecutor(
                tupitaConfig.coreQueryParsingThreads(),
                tupitaConfig.maxQueryParsingThreads(),
        1,
                TimeUnit.MINUTES,
                new LinkedBlockingQueue<>(tupitaConfig.batchParseQueueSize()),
                new NamedThreadFactory(
                    new ThreadGroup("QueryParser"),
            true),
                new ThreadPoolExecutor.CallerRunsPolicy());
        this.queryParserExecutor.prestartAllCoreThreads();

        parsedQueries = new TimeFrameQueue<>(tupitaConfig.metricsTimeFrame());

        staters.add(index.stater());
        staters.add(
            new PassiveStaterAdapter<>(
                parsedQueries,
                new NamedStatsAggregatorFactory<>(
                    "queries_parsed_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));
    }

    public DatabaseConfig config() {
        return index.config();
    }

    private static IniConfig patchConfig(
        final IniConfig config)
        throws ConfigException
    {
        final String analyze = "analyze";
        // dirty hack i suppose
        IniConfig uidSection = config.section("field.uid");
        uidSection.put("tokenizer", "boolean");
        uidSection.put(analyze, Boolean.toString(true));

        for (IniConfig section: config.section("field").sections().values()) {
            if (section.getBoolean(analyze, false)) {
                section.put("prefixed", Boolean.toString(true));
            }

            if (section.getString(TYPE, null) != null) {
                section.put(TYPE, null);
            }
        }

        return config;
    }

    public Charset charset() {
        return luceneCharset;
    }

    public Query parseQuery(
        final ProxySession session,
        final String text,
        final LongPrefix prefix)
        throws Exception
    {
        PrefixingAnalyzerWrapper analyzer =
                index.analyzerProvider().searchAnalyzer(prefix);

        QueryParser parser = parserFactory
            .create(Version.LUCENE_40, null, analyzer);
        NewSearchRequest.adjustParser(parser, index.config(), session.params());
        parser.setLowercaseExpandedTerms(true);
        parser.setReplaceEExpandedTerms(true);
        parser.useOldWildcatdQuery(true);

        Query query = parser.parse(text);
        parsedQueries.accept(1L);
        return query;
    }

    @Override
    public <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer)
        throws E
    {
        statsConsumer.stat(
            "batched_query_active_tasks_axxx",
            queryParserExecutor.getActiveCount());
        statsConsumer.stat(
            "batched_query_parse_queue_axxx",
            queryParserExecutor.getQueue().size());

        long docsInMem = 0L;
        try {
            for (int i = 0; i < index.shardsCount(); i++) {
                docsInMem += index.getShard(i).getDocsInMem();
            }
        } catch (IOException ioe) {
            index.logger().log(
                Level.WARNING,
                "Failed to compute docs in mem",
                ioe);
        }

        long overflow = (docsInMem - maxMemInDoc) / (MB * MB);

        statsConsumer.stat("index_docs_in_mem_axxx", docsInMem);
        statsConsumer.stat("index_docs_in_mem_overflow_axxx", overflow);

        for (Stater stater: staters) {
            stater.stats(statsConsumer);
        }
    }

    public void scheduleQueryParse(final Runnable parseTask) {
        this.queryParserExecutor.execute(parseTask);
    }

    protected void index(
        final byte[] data,
        final MessageContext context)
        throws IOException, ParseException
    {
        context.logger().info("lucene index message size: " + data.length);
        dispatcher.dispatch(
            new JsonUpdateMessage(
                context,
                data,
                luceneCharset,
                Message.Priority.ONLINE,
                SHARD,
                QUEUE_ID,
                index.config(),
                false,
                index.config().orderIndependentUpdate()));
    }

    public void delete(
        final String query,
        final MessageContext context)
        throws IOException, ParseException
    {
        dispatcher.dispatch(new JsonDeleteMessage(
            context,
            query,
            Message.Priority.ONLINE,
            SHARD,
            QUEUE_ID,
            index.config(),
            false));
    }

    // CSOFF: ReturnCount
    public void search(
        final TupitaIndexationContext context,
        final Collection<? extends TupitaQuery> queries,
        final Consumer<TupitaQuery> matchedConsumer)
        throws IOException
    {
        if (queries.size() <= 0) {
            return;
        }

        SearchRequestBase request = new TupitaSearchRequest(context);
        SearchResultsConsumer consumer = this.consumer;
        Searcher searcher = index.getSearcher(context.longPrefix(), true);
        Compress.resetStats();
        int matched = 0;
        try {
            StringBuilder log = new StringBuilder("Matched: ");
            for (TupitaQuery query: queries) {
                if (query.luceneQuery() == null) {
                    continue;
                }

                if (context.debugRequest()) {
                    context.session().logger().info(
                        "Testing " + query.luceneQuery().toString());
                    request = new TupitaDebugSearchRequest(context);
                    consumer =
                        new PrintingResultConsumer(context.session().logger());
                }

                DocCollector col =
                    docCollectorFactory.createCollector(request, consumer);

                col.setPrefix(context.longPrefix());
                searcher.searcher().search(query.luceneQuery(), col, true);
                col.flush();
                col.close();
                if (col.getTotalCount() > 0) {
                    if (context.debugRequest()) {
                        context.session().logger().info(col.toString());
                    }
                    matched++;
                    matchedConsumer.accept(query);
                    log.append(query.id());
                    if (query.stop()) {
                        log.append("(stop)");
                    }
                    log.append(", ");
                }
            }

            context.session().logger().fine(log.toString());

            context.session().logger().info(
                "Matched queries count: " + matched);

            IOStats stats = Compress.stats();
            if (stats.reads() > 0) {
                context.session().logger().info(
                    "IOStats: " + stats);
                StringBuilder sb = new StringBuilder("Shards sizes ");
                long total = 0;
                for (int shard = 0; shard < index.shardsCount(); shard++) {
                    sb.append(',');
                    long size = index.getShard(shard).indexSizeMb();
                    sb.append(size);
                    total += size;
                }

                sb.append(" Total ");
                sb.append(total);
                context.session().logger().info(sb.toString());
            }
        } finally {
            if (searcher != null) {
                searcher.free();
            }
        }
    }
    // CSON: ReturnCount

    @Override
    public void close() throws IOException {
        this.queryParserExecutor.shutdown();
    }

    private final class TupitaDebugSearchRequest extends SearchRequestBase {
        private TupitaDebugSearchRequest(
            final TupitaIndexationContext context)
            throws IOException
        {
            super(
                context,
                TupitaLucene.this.index,
                TupitaLucene.this.index.config());
            try {
                getFields = NewSearchRequest.extractGetFieldsConfig(
                    index.config(),
                    Collections.emptySet(),
                    new CgiParams("&get=*,url")).fields();
            } catch (BadRequestException bre) {
                throw new IOException(bre);
            }
        }

        @Override
        public YandexScorerFactory scorerFactory() {
            return TupitaLucene.this.scorerFactory;
        }
    }

    private final class TupitaSearchRequest extends SearchRequestBase {
        private TupitaSearchRequest(final TupitaIndexationContext context) {
            super(
                context,
                TupitaLucene.this.index,
                TupitaLucene.this.index.config());
        }

        @Override
        public Set<String> getFields() {
            return GET_FIELDS;
        }

        @Override
        public YandexScorerFactory scorerFactory() {
            return TupitaLucene.this.scorerFactory;
        }
    }

    private static final class PrintingResultConsumer
        extends EmptyResultConsumer
    {
        private final PrefixedLogger logger;

        private PrintingResultConsumer(final PrefixedLogger logger) {
            this.logger = logger;
        }

        @Override
        public void document(
            final YaDoc3 doc,
            final Set<String> fields)
            throws IOException
        {
            StringBuilder sb = new StringBuilder();
            for (String field: fields) {
                String value = doc.getString(field);
                if (value == null) {
                    continue;
                }

                sb.append(field);
                sb.append('=');
                sb.append(value);
                sb.append("; ");
            }

            logger.info("Found doc " + sb.toString());
        }

        @Override
        public void totalHitsCount(final int count) throws IOException {
            logger.info("Total search result cnt " + count);
        }

        @Override
        public void uniqHitsCount(final int count) throws IOException {
            logger.info("Uniq search result cnt " + count);
        }
    }

    private static class EmptyResultConsumer
        implements SearchResultsConsumer
    {
        @Override
        public void checkAlive() {
        }

        @Override
        public void totalHitsCount(final int count) throws IOException {
        }

        @Override
        public void uniqHitsCount(final int count) throws IOException {
        }

        @Override
        public void startResults() throws IOException {
        }

        @Override
        public void startHits() throws IOException {
        }

        @Override
        public void endHits() throws IOException {
        }

        @Override
        public void endResults() throws IOException {
        }

        @Override
        public void document(
            final YaDoc3 doc,
            final Set<String> fields)
            throws IOException
        {
        }
    }
}
