package ru.yandex.search.mail.tupita.admin360;

import java.nio.charset.StandardCharsets;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import ru.yandex.dbfields.MailIndexFields;
import ru.yandex.function.GenericFunction;
import ru.yandex.function.StringBuilderable;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.parser.string.EnumParser;
import ru.yandex.search.document.mail.MailMetaInfo;
import ru.yandex.search.request.util.SearchRequestText;

public class RuleParser {
//    private static final int COND_ELEMENT = 1;
//    private static final int COND_LOGIC_OP = 1;
    //private static final Map<String, ConditionType> CONDITION_TYPES;

    private final EnumParser<FieldGroup> fieldNameParser = new EnumParser<>(FieldGroup.class);
    private static final Map<String, CheckPredicate> PREDICATES =
        Collections.unmodifiableMap(
            Arrays
                .stream(CheckPredicate.values())
                .collect(
                    Collectors.toMap(CheckPredicate::value, Function.identity())));
    private static final Map<String, SubPredicate> SUBPREDICATES =
        Collections.unmodifiableMap(
            Arrays
                .stream(SubPredicate.values())
                .collect(
                    Collectors.toMap(SubPredicate::value, Function.identity())));

//
//    public ConditionQuery parse(final JsonMap map, final ConditionType ctype) throws JsonException {
//        ConditionType conditionType = ConditionType.AND;
//        // only condition type or field queryies
//
//        List<QueryAtom> queries = new ArrayList<>();
//        for (String key: map.keySet()) {
//            ConditionType ctype = getConditionOrNull(key);
//            if (ctype != null) {
//                queries.add(parse(map.getMap(key), ctype));
//                continue;
//            }
//
//            queries.add(parseFieldQuery(key, map.getMap(key)));
//        }
//    }

//    private ConditionType tryAnyConditionType(final JsonMap map) throws JsonException {
//        for (ConditionType ctype: ConditionType.values()) {
//            JsonList list = map.getListOrNull('$' + ctype.name().toLowerCase(Locale.ENGLISH));
//            if (list != null) {
//                return ctype;
//            }
//        }
//
//        return null;
//    }

//    private QueryAtom parseFieldQuery(final String key, final JsonMap value) throws JsonException {
//        int colon = key.indexOf(':');
//        if (colon < 0) {
//
//        }
//    }

    public Condition parse(final JsonMap map, final ParseContext context) throws JsonException, ParseException {
        //start we have logic_or, or conditions itself

        JsonList orList = map.getListOrNull('$' + LogicOperation.OR.value());
        JsonList andList = map.getListOrNull('$' + LogicOperation.AND.value());
        if (orList != null || andList != null) {
            if (map.size() > 1) {
                throw new ParseException(
                    "Extra fields with logic conditions " + String.join(",", map.keySet()), 0);
            }

            LogicOperation logicOperation = orList == null ? LogicOperation.AND : LogicOperation.OR;
            JsonList list = orList == null ? andList : orList;
            List<Condition> conditions = new ArrayList<>(list.size());
            for (JsonObject condObj: list) {
                if (condObj.type() != JsonObject.Type.MAP) {
                    throw new ParseException(
                        "Condition should be represented as object inseadof " + condObj.type().name(),
                        0);
                }
                conditions.add(parse(condObj.asMap(), context));
            }

            return new LogicCondition(logicOperation, conditions);
        }

        // now we in have field:matcher packed list of conditions

        LogicOperation logicOperation = LogicOperation.AND;
        List<Condition> conditions = new ArrayList<>(map.size());
        for (Map.Entry<String, JsonObject> entry: map.entrySet()) {
            String fieldKey = entry.getKey();
            int colonIndex = fieldKey.indexOf(':');
            String fieldName = fieldKey;
            String subKey = null;
            if (colonIndex >= 0) {
                fieldName = fieldKey.substring(0, colonIndex);
                subKey = fieldKey.substring(colonIndex + 1);
            }

            FieldGroup fieldGroup = fieldNameParser.apply(fieldName);
            if (!fieldGroup.subKeyPossible() && subKey != null) {
                throw new ParseException(fieldGroup.value() + " is not supposed to have subkey, but got " + subKey, 0);
            }

            Object subKeyValue = null;
            if (subKey != null) {
                subKeyValue = fieldGroup.subkeyParser().apply(subKey);
            }

            // now we parsing field value

            PredicateMatcher predicateMatcher;
            switch (entry.getValue().type()) {
                case LIST:
                case NULL:
                    throw new ParseException(
                        entry.getValue().type().name()
                            + " is not allowed as field value for "
                            + fieldGroup.value(), 0);
                case MAP:
                    // parse object
                    predicateMatcher = parsePredicateOrValue(entry.getValue().asMap());
                    break;
                default:
                    predicateMatcher = new PredicateMatcher(
                        CheckPredicate.EQUAL,
                        null,
                        new IdentityValueMatcher(entry.getValue().asString()));
            }

            conditions.add(
                new FieldCondition(
                    fieldGroup,
                    subKeyValue,
                    predicateMatcher,
                    context));
        }

        return new LogicCondition(logicOperation, conditions);
    }

    private List<String> jsonToSearchField(final FieldGroup group, final Object subField, final ParseContext context) throws ParseException {
        switch (group) {
            case ADDRESS:
                if (subField instanceof FieldGroup) {
                    return addressToSearchField((FieldGroup) subField, context);
                } else {
                    throw new ParseException("Invalid subfield for " + group.value(), 0);
                }
            case TOCC:
            case FROM:
            case TO:
            case CC:
                return addressToSearchField(group, context);
            case BODY:
                context.requiredFields.add(MailMetaInfo.STID);
                return Collections.singletonList(MailIndexFields.BODY_TEXT);
            case SUBJECT:
                context.requiredFields.add(MailMetaInfo.SUBJECT);
                return Collections.singletonList(MailMetaInfo.HDR + MailMetaInfo.SUBJECT + MailMetaInfo.NORMALIZED);
            case HEADER:
                context.requiredFields.add(MailMetaInfo.STID);
                return Collections.singletonList(MailMetaInfo.HEADERS);
            case ATTACH:
                context.requiredFields.add(MailMetaInfo.STID);
                return Collections.singletonList(MailMetaInfo.ATTACHNAME);
            default:
                throw new ParseException("Unsupported field " + group, 0);
        }
    }

    private List<String> addressToSearchField(final FieldGroup group, final ParseContext context) throws ParseException {
        if (group == null) {
            throw new ParseException("Empty address subfield", 0);
        }
        context.requiredFields.add(MailMetaInfo.FROM);
        context.requiredFields.add(MailMetaInfo.TO);
        context.requiredFields.add(MailMetaInfo.CC);
        context.requiredFields.add(MailMetaInfo.BCC);
        context.requiredFields.add(MailMetaInfo.REPLY_TO_FIELD);
        switch (group) {
            case CC:
                return Arrays.asList(
                    MailMetaInfo.HDR + MailMetaInfo.CC  +MailMetaInfo.EMAIL,
                    MailMetaInfo.HDR + MailMetaInfo.CC  +MailMetaInfo.DISPLAY_NAME);
            case TO:
                return Arrays.asList(
                    MailMetaInfo.HDR + MailMetaInfo.TO + MailMetaInfo.EMAIL,
                    MailMetaInfo.HDR + MailMetaInfo.TO  +MailMetaInfo.DISPLAY_NAME);
            case TOCC:
                return Arrays.asList(
                    MailMetaInfo.HDR + MailMetaInfo.TO + MailMetaInfo.EMAIL,
                    MailMetaInfo.HDR + MailMetaInfo.TO  +MailMetaInfo.DISPLAY_NAME,
                    MailMetaInfo.HDR + MailMetaInfo.CC  +MailMetaInfo.EMAIL,
                    MailMetaInfo.HDR + MailMetaInfo.CC  +MailMetaInfo.DISPLAY_NAME);
            case FROM:
                return Arrays.asList(MailMetaInfo.HDR + MailMetaInfo.FROM + MailMetaInfo.EMAIL, MailMetaInfo.HDR + MailMetaInfo.FROM  +MailMetaInfo.DISPLAY_NAME);
            default:
                throw new ParseException("Not address subfield " + group.value, 0);
        }
    }

    private PredicateMatcher parsePredicateOrValue(final JsonMap map) throws JsonException, ParseException {
        if (map.size() != 1) {
            throw new ParseException(
                "Expecting one predicate, but got  " + JsonType.NORMAL.toString(map),
                0);
        }

        Map.Entry<String, JsonObject> entry = map.entrySet().iterator().next();
        String trimmed = entry.getKey().toLowerCase(Locale.ENGLISH).trim();
        CheckPredicate checkPredicate = PREDICATES.get(trimmed);
        if (checkPredicate == null) {
            // ok trying to parse value instead
            ValueMatcher valueMatcher = parseValue(map);
            return new PredicateMatcher(CheckPredicate.EQUAL, null, valueMatcher);
        }

        SubPredicate subPredicate = null;
        ValueMatcher valueMatcher;
        switch (entry.getValue().type()) {
            case LIST:
            case NULL:
                throw new ParseException(
                    entry.getValue().type().name()
                        + " is not allowed as field value with predicate "
                        + entry.getKey(), 0);
            case MAP:
                if (checkPredicate.subPredicates() == null || checkPredicate.subPredicates().isEmpty()) {
                    valueMatcher = parseValue(entry.getValue().asMap());
                } else {
                    // parse subpredicate
                    return parseSubconditionOrValue(checkPredicate, entry.getValue().asMap());
                }
                break;
            default:
                if (checkPredicate == CheckPredicate.EXISTS) {
                    boolean existsValue = entry.getValue().asBoolean();
                    if (!existsValue) {
                        checkPredicate = CheckPredicate.NOT_EXISTS;
                    }
                }
                valueMatcher = new IdentityValueMatcher(entry.getValue().asString());
                break;
        }

        return new PredicateMatcher(
            checkPredicate,
            subPredicate,
            valueMatcher);
    }

    private ValueMatcher parseValue(final JsonMap map) throws ParseException, JsonException {
        String value = map.getString("$base64", null);
        if (value == null) {
            throw new ParseException("expecting base64 value, but got nothing", 0);
        }
        return new Base64ValueMatcher(value);
    }

    private PredicateMatcher parseSubconditionOrValue(
        final CheckPredicate checkPredicate,
        final JsonMap map)
        throws JsonException, ParseException
    {
        if (map.size() != 1) {
            throw new ParseException("Expecting \"all\" or \"any\" but got " + JsonType.NORMAL.toString(map), 0);
        }

        Map.Entry<String, JsonObject> entry = map.entrySet().iterator().next();
        String key = entry.getKey().toLowerCase(Locale.ENGLISH);
        SubPredicate subPredicate = SUBPREDICATES.get(key);
        if (subPredicate == null) {
            throw new ParseException("Expecting \"$all\" or \"$any\" but got " + key, 0);
        }

        if (entry.getValue().type() != JsonObject.Type.LIST) {
            throw new ParseException("Expecting array for key " + key, 0);
        }

        JsonList list = entry.getValue().asList();
        List<ValueMatcher> matchers = new ArrayList<>(list.size());
        for (JsonObject jo: list) {
            matchers.add(parseValue(jo.asMap()));
        }

        LogicOperation operation;
        if (subPredicate == SubPredicate.ANY) {
            operation = LogicOperation.OR;
        } else {
            operation = LogicOperation.AND;
        }
        return new PredicateMatcher(checkPredicate, subPredicate, new ConditionValueMatcher(operation, matchers));
    }

    public static class ParseContext {
        private final Set<String> requiredFields = new LinkedHashSet<>();

        public ParseContext() {
        }

        public Set<String> requiredFields() {
            return requiredFields;
        }
    }

    private enum FieldGroup {
        FROM,
        TO,
        CC,
        TOCC,
        BODY,
        HEADER(SubkeyAnyNonEmptyParser.INSTANCE),
        FILENAME,
        ATTACH(SubkeyFieldValidator.of(FILENAME)),
        ADDRESS(SubkeyFieldValidator.of(FROM, TO, CC, TOCC)),
        SUBJECT;

        private final boolean subKeyPossible;
        private final String value;
        private final GenericFunction<String, ? extends Object, ParseException> subkeyParser;

        FieldGroup() {
            this.subkeyParser = null;
            this.subKeyPossible = false;
            this.value = name().toLowerCase(Locale.ENGLISH);
        }

        FieldGroup(final GenericFunction<String, ? extends Object, ParseException> subkeyParser) {
            this.subkeyParser = subkeyParser;
            this.subKeyPossible = true;
            this.value = name().toLowerCase(Locale.ENGLISH);
        }

        public String value() {
            return value;
        }

        public boolean subKeyPossible() {
            return subKeyPossible;
        }

        public GenericFunction<String, ? extends Object, ParseException> subkeyParser() {
            return subkeyParser;
        }
    }

    public enum CheckPredicate {
        EQUAL("$eq"),
        NOT_EQUAL("$ne", true),
        CONTAINS(SubPredicate.ALL, SubPredicate.ANY),
        NOT_CONTAINS("$not-contains", true, SubPredicate.ALL, SubPredicate.ANY),
        EXISTS("$exists", false),
        NOT_EXISTS("$not-exists", true);

        private final String value;
        private final Set<SubPredicate> subPredicates;
        private final boolean negation;

        CheckPredicate(final SubPredicate... subs) {
            this.value = '$' + name().toLowerCase(Locale.ENGLISH);
            this.subPredicates = Collections.unmodifiableSet(new LinkedHashSet<>(Arrays.asList(subs)));
            this.negation = false;
        }

        CheckPredicate(final String value, final SubPredicate... subs) {
            this.value = value;
            this.subPredicates = Collections.unmodifiableSet(new LinkedHashSet<>(Arrays.asList(subs)));
            this.negation = false;
        }

        CheckPredicate(
            final String value,
            final boolean negation,
            final SubPredicate... subs)
        {
            this.value = value;
            this.subPredicates = Collections.unmodifiableSet(new LinkedHashSet<>(Arrays.asList(subs)));
            this.negation = negation;
        }

        public boolean negation() {
            return negation;
        }

        public String value() {
            return value;
        }

        public Set<SubPredicate> subPredicates() {
            return subPredicates;
        }
    }

    public enum SubPredicate {
        ANY,
        ALL;

        private final String value;

        SubPredicate() {
            this.value = '$' + name().toLowerCase(Locale.ENGLISH);
        }

        public String value() {
            return value;
        }
    }


    private enum SubkeyAnyNonEmptyParser implements GenericFunction<String, String, ParseException> {
        INSTANCE;

        @Override
        public String apply(final String s) throws ParseException {
            if (s == null || s.isBlank()) {
                throw new ParseException("Subkey is not supposed to be empty", 0);
            }

            return s.trim();
        }
    }
    private static class SubkeyFieldValidator implements GenericFunction<String, FieldGroup, ParseException> {
        private final Set<FieldGroup> possible;
        private final EnumParser<FieldGroup> parser = new EnumParser<>(FieldGroup.class);

        public SubkeyFieldValidator(final FieldGroup... fields) {
            this.possible = Collections.unmodifiableSet(new LinkedHashSet<>(Arrays.asList(fields)));
        }

        @Override
        public FieldGroup apply(final String o) throws ParseException {
            FieldGroup group = parser.apply(o);
            if (possible.contains(group)) {
                return group;
            }

            throw new ParseException(
                group.value() + " is not allowed here, possible are "
                    + possible.stream().map(FieldGroup::value).collect(Collectors.joining()),
                0);
        }

        public static SubkeyFieldValidator of(final FieldGroup... fields) {
            return new SubkeyFieldValidator(fields);
        }
    }

    private enum LogicOperation implements StringBuilderable {
        OR,
        AND;

        private final String value;

        LogicOperation() {
            this.value = name().toLowerCase(Locale.ENGLISH);
        }

        public String value() {
            return value;
        }

        @Override
        public void toStringBuilder(@Nonnull final StringBuilder sb) {
            sb.append(' ');
            sb.append(value);
            sb.append(' ');
        }
    }

    public interface Condition extends StringBuilderable {

    }

    private static final class LogicCondition implements Condition {
        private LogicOperation operation;
        private List<Condition> conditions;

        public LogicCondition(final LogicOperation operation, final List<Condition> conditions) {
            this.operation = operation;
            this.conditions = conditions;
        }

        @Override
        public void toStringBuilder(@Nonnull final StringBuilder sb) {
            sb.append('(');
            for (int i = 0; i < conditions.size(); i++) {
                if (i != 0) {
                    operation.toStringBuilder(sb);
                }
                conditions.get(i).toStringBuilder(sb);
            }
            sb.append(')');
        }
    }

    private final class FieldCondition implements Condition {
        private final FieldGroup field;
        private final Object subField;
        private final FieldMatcher matcher;
        private final List<String> luceneFields;

        public FieldCondition(
            final FieldGroup field,
            final Object subField,
            final FieldMatcher matcher,
            final ParseContext context)
            throws ParseException
        {
            this.field = field;
            this.subField = subField;
            this.matcher = matcher;
            this.luceneFields = jsonToSearchField(field, subField, context);
        }

        @Override
        public void toStringBuilder(@Nonnull final StringBuilder sb) {
            if (matcher.negation()) {
                sb.append("mid_p:* AND NOT ");
            }

            if (luceneFields.size() == 1) {
                sb.append(luceneFields.get(0));
                sb.append(':');
                if (field == FieldGroup.HEADER) {
                    sb.append("\"" + SearchRequestText.fullEscape(String.valueOf(subField), true) + ":\"");
                }

                matcher.toStringBuilder(sb);
            } else {
                sb.append('(');
                for (String lf: luceneFields) {
                    sb.append(lf);
                    sb.append(':');
                    if (field == FieldGroup.HEADER) {
                        sb.append("\"" + SearchRequestText.fullEscape(String.valueOf(subField), true) + ":\"");
                    }

                    matcher.toStringBuilder(sb);
                    sb.append(" OR ");
                }
                sb.setLength(sb.length() - 4);
                sb.append(')');
            }
        }
    }

    private interface FieldMatcher extends StringBuilderable {
        default boolean negation() {
            return false;
        }
    }

    private static class ConditionValueMatcher implements ValueMatcher {
        private final LogicOperation operation;
        private final List<ValueMatcher> matchers;

        public ConditionValueMatcher(final LogicOperation operation, final List<ValueMatcher> matchers) {
            this.operation = operation;
            this.matchers = matchers;
        }

        @Override
        public void toStringBuilder(@Nonnull final StringBuilder sb) {
            sb.append('(');
            for (int i = 0; i < matchers.size(); i++) {
                if (i != 0) {
                    operation.toStringBuilder(sb);
                }

                matchers.get(i).toStringBuilder(sb);
            }
            sb.append(')');
        }
    }

    private static class PredicateMatcher implements FieldMatcher{
        private final CheckPredicate predicate;
        //private final SubPredicate subPredicate;
        private final ValueMatcher valueMatcher;

        public PredicateMatcher(
            final CheckPredicate predicate,
            final SubPredicate subPredicate,
            final ValueMatcher valueMatcher)
        {
            this.predicate = predicate;
            //this.subPredicate = subPredicate;
            this.valueMatcher = valueMatcher;
        }

        @Override
        public boolean negation() {
            return predicate.negation();
        }

        @Override
        public void toStringBuilder(@Nonnull final StringBuilder sb) {
            switch (predicate) {
                case EQUAL:
                    sb.append('(');
                    sb.append("\"");
                    valueMatcher.toStringBuilder(sb);
                    sb.append("\"");
                    sb.append(')');
                    break;
                case CONTAINS:
                case NOT_CONTAINS:
                    if (valueMatcher instanceof ConditionValueMatcher) {
                        valueMatcher.toStringBuilder(sb);
                    } else {
                        sb.append('(');
                        sb.append("*");
                        valueMatcher.toStringBuilder(sb);
                        sb.append("*");
                        sb.append(')');
                    }

                    break;
                case EXISTS:
                case NOT_EXISTS:
                    sb.append('*');
                    break;
                case NOT_EQUAL:
                    valueMatcher.toStringBuilder(sb);
                    break;
            }

        }
    }

    private interface ValueMatcher extends StringBuilderable {

    }

    private static class IdentityValueMatcher implements ValueMatcher {
        private final String value;

        public IdentityValueMatcher(final String value) {
            this.value = value;
        }

        @Override
        public void toStringBuilder(@Nonnull final StringBuilder sb) {
            sb.append(SearchRequestText.fullEscape(value, true));
        }
    }

    private static class Base64ValueMatcher implements ValueMatcher {
        private final String value;

        public Base64ValueMatcher(final String value) throws ParseException {
            this.value = new String(Base64.getDecoder().decode(value), StandardCharsets.UTF_8);
        }

        @Override
        public void toStringBuilder(@Nonnull final StringBuilder sb) {
            sb.append(SearchRequestText.fullEscape(value, true));
        }
    }
}
