package ru.yandex.search.mail.tupita.admin360;

import java.nio.charset.StandardCharsets;
import java.text.ParseException;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import ru.yandex.dbfields.MailIndexFields;
import ru.yandex.function.GenericFunction;
import ru.yandex.function.StringBuilderable;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.search.document.mail.MailMetaInfo;

public class RuleParser2 {
    private final WeirdEnumParser<FieldGroup> fieldNameParser = new WeirdEnumParser<>(FieldGroup.class);
    private static final Map<String, CheckPredicate> PREDICATES =
        Collections.unmodifiableMap(
            Arrays
                .stream(CheckPredicate.values())
                .collect(
                    Collectors.toMap(CheckPredicate::value, Function.identity())));
    private static final Map<String, SubPredicate> SUBPREDICATES =
        Collections.unmodifiableMap(
            Arrays
                .stream(SubPredicate.values())
                .collect(
                    Collectors.toMap(SubPredicate::value, Function.identity())));

    public static <T extends Enum<T>> T parseEnum(final Class<T> clazz, final String value) throws ParseException {
        WeirdEnumParser<T> parser = new WeirdEnumParser<>(clazz);
        try {
            return parser.apply(value);
        } catch (Exception e) {
            throw new ParseException("Not valid value, " + value, -1);
        }
    }

    private List<String> luceneFieldsForAddress(final FieldGroup group, final FieldMatchOperator operator) throws ParseException {
        switch (operator) {
            case EQUAL_ANY:
            case EQUAL_ONE:
            case NOT_EQUAL_ONE:
            case NOT_EQUAL_ANY:
                return luceneFieldsForAddressEquals(group);
            default:
                return luceneFieldsForAddressContains(group);
        }

    }

    private List<String> luceneFieldsForAddressEquals(final FieldGroup group) throws ParseException {
        List<String> luceneFields = new ArrayList<>();
        switch (group) {
            case TOCC:
                luceneFields.add(MailMetaInfo.HDR + MailMetaInfo.TO + MailMetaInfo.EMAIL);
                luceneFields.add(MailMetaInfo.HDR + MailMetaInfo.TO + MailMetaInfo.DISPLAY_NAME);
                luceneFields.add(MailMetaInfo.HDR + MailMetaInfo.CC + MailMetaInfo.EMAIL);
                luceneFields.add(MailMetaInfo.HDR + MailMetaInfo.CC + MailMetaInfo.DISPLAY_NAME);
                break;
            case FROM:
                luceneFields.add(MailMetaInfo.HDR + MailMetaInfo.FROM  +MailMetaInfo.EMAIL);
                luceneFields.add(MailMetaInfo.HDR + MailMetaInfo.FROM  +MailMetaInfo.DISPLAY_NAME);
                break;
            case TO:
                luceneFields.add(MailMetaInfo.HDR + MailMetaInfo.TO  +MailMetaInfo.EMAIL);
                luceneFields.add(MailMetaInfo.HDR + MailMetaInfo.TO  +MailMetaInfo.DISPLAY_NAME);
                break;
            case CC:
                luceneFields.add(MailMetaInfo.HDR + MailMetaInfo.CC  +MailMetaInfo.EMAIL);
                luceneFields.add(MailMetaInfo.HDR + MailMetaInfo.CC  +MailMetaInfo.DISPLAY_NAME);
                break;
        }

        return luceneFields;
    }

    private List<String> luceneFieldsForAddressContains(final FieldGroup group) throws ParseException {
        List<String> luceneFields = new ArrayList<>();
        switch (group) {
            case TOCC:
                luceneFields.add(MailMetaInfo.HDR + MailMetaInfo.TO + "_keyword");
                luceneFields.add(MailMetaInfo.HDR + MailMetaInfo.CC + "_keyword");
                break;
            case FROM:
                luceneFields.add(MailMetaInfo.HDR + MailMetaInfo.FROM + "_keyword");
                break;
            case TO:
                luceneFields.add(MailMetaInfo.HDR + MailMetaInfo.TO + "_keyword");
                break;
            case CC:
                luceneFields.add(MailMetaInfo.HDR + MailMetaInfo.CC + "_keyword");
                break;
        }

        return luceneFields;
    }

    private List<String> luceneFieldsForAddress(final FieldGroup group) throws ParseException {
        List<String> luceneFields = new ArrayList<>();
        switch (group) {
            case TOCC:
                luceneFields.add(MailMetaInfo.HDR + MailMetaInfo.TO + MailMetaInfo.EMAIL);
                luceneFields.add(MailMetaInfo.HDR + MailMetaInfo.TO + MailMetaInfo.DISPLAY_NAME);
                luceneFields.add(MailMetaInfo.HDR + MailMetaInfo.CC + MailMetaInfo.EMAIL);
                luceneFields.add(MailMetaInfo.HDR + MailMetaInfo.CC + MailMetaInfo.DISPLAY_NAME);
                break;
            case FROM:
                luceneFields.add(MailMetaInfo.HDR + MailMetaInfo.FROM  +MailMetaInfo.EMAIL);
                luceneFields.add(MailMetaInfo.HDR + MailMetaInfo.FROM  +MailMetaInfo.DISPLAY_NAME);
                break;
            case TO:
                luceneFields.add(MailMetaInfo.HDR + MailMetaInfo.TO  +MailMetaInfo.EMAIL);
                luceneFields.add(MailMetaInfo.HDR + MailMetaInfo.TO  +MailMetaInfo.DISPLAY_NAME);
                break;
            case CC:
                luceneFields.add(MailMetaInfo.HDR + MailMetaInfo.CC  +MailMetaInfo.EMAIL);
                luceneFields.add(MailMetaInfo.HDR + MailMetaInfo.CC  +MailMetaInfo.DISPLAY_NAME);
                break;
        }

        return luceneFields;
    }

    public Condition parseLogicList(
        final JsonList list,
        final LogicOperation operation,
        final ParseContext context)
        throws Exception
    {
        List<Condition> conditions = new ArrayList<>(list.size());
        for (JsonObject condObj: list) {
            if (condObj.type() != JsonObject.Type.MAP) {
                throw new ParseException(
                    "Condition should be represented as object inseadof " + condObj.type().name(),
                    0);
            }
            conditions.add(parse(condObj.asMap(), context));
        }

        if (conditions.size() == 0) {
            throw new ParseException("Zero condition list", -1);
        }
        return new LogicCondition(operation, conditions);
    }

    public Condition parse(final JsonMap map, final ParseContext context) throws Exception {
        //start we have logic_or, or conditions itself
        List<Condition> conditions = new ArrayList<>(map.size());

        JsonList orList = map.remove('$' + LogicOperation.OR.value()).asListOrNull();
        JsonList andList = map.remove('$' + LogicOperation.AND.value()).asListOrNull();
        // now we in have field:matcher packed list of conditions

        LogicOperation logicOperation = LogicOperation.AND;
        for (Map.Entry<String, JsonObject> entry: map.entrySet()) {
            String fieldKey = entry.getKey();
            int colonIndex = fieldKey.indexOf(':');
            String fieldName = fieldKey;
            String subKey = null;
            if (colonIndex >= 0) {
                fieldName = fieldKey.substring(0, colonIndex);
                subKey = fieldKey.substring(colonIndex + 1);
            }

            FieldGroup fieldGroup = fieldNameParser.apply(fieldName);
            if (!fieldGroup.subKeyPossible() && subKey != null) {
                throw new ParseException(fieldGroup.value() + " is not supposed to have subkey, but got " + subKey, 0);
            }

            // now we parsing field value

            FieldMatchOperator operator;
            List<String> values;
            switch (entry.getValue().type()) {
                case NULL:
                    throw new ParseException(
                        entry.getValue().type().name()
                            + " is not allowed as field value for "
                            + fieldGroup.value(), 0);
                case LIST:
                    // could be just list of values
                    values = parseValueList(entry.getValue().asList());

                    if (values.size() <= 1) {
                        operator = FieldMatchOperator.EQUAL_ONE;
                    } else {
                        operator = FieldMatchOperator.EQUAL_ANY;
                    }
                    break;
                case MAP:
                    // parse object
                    Map.Entry<FieldMatchOperator, List<String>> opValueEntry = parsePredicateOrValue(entry.getValue().asMap());
                    operator = opValueEntry.getKey();
                    values = opValueEntry.getValue();
                    break;
                default:
                    operator = FieldMatchOperator.EQUAL_ONE;
                    values = Collections.singletonList(entry.getValue().asString());
            }

            List<String> luceneFields = new ArrayList<>();
            boolean header = false;
            switch (fieldGroup) {
                case ADDRESS:
                    if (subKey == null) {
                        throw new ParseException("attach has no sub_field declared", -1);
                    }
                    FieldGroup subField = parseEnum(FieldGroup.class, subKey);
                    switch (subField) {
                        case CC:
                        case TO:
                        case FROM:
                        case TOCC:
                            luceneFields = luceneFieldsForAddress(subField, operator);
                            break;
                        default:
                            throw new ParseException("unsupport sub_field " + subField.name() + " for field " + fieldGroup.name(), -1);
                    }
                    break;
                case SUBJECT:
                    switch (operator) {
                        case EQUAL_ANY:
                        case EQUAL_ONE:
                        case NOT_EQUAL_ONE:
                        case NOT_EQUAL_ANY:
                            luceneFields.add(MailMetaInfo.HDR + MailMetaInfo.SUBJECT + "_keyword");
                            break;
                        default:
                            luceneFields.add(MailMetaInfo.HDR + MailMetaInfo.SUBJECT + "_keyword");
                            break;
                    }

                    break;
                case BODY:
                    luceneFields.add(MailIndexFields.BODY_TEXT);
                    luceneFields.add(MailIndexFields.PURE_BODY);
                    break;
                case HEADER:
                    luceneFields.add(MailMetaInfo.HEADERS);
                    if (subKey == null) {
                        throw new ParseException("Header has no sub_field declared", -1);
                    }
                    header = true;
                    break;
                case ATTACH:
                    luceneFields.add(MailMetaInfo.ATTACHNAME);
                    luceneFields.add(MailMetaInfo.ATTACHNAME + "_keyword");

                    if (subKey == null || parseEnum(AttachField.class, subKey) != AttachField.FILENAME) {
                        throw new ParseException("attach has no sub_field declared", -1);
                    }
                    break;
                case CC:
                case TO:
                case FROM:
                case TOCC:
                    luceneFields = luceneFieldsForAddress(fieldGroup);
                    break;
            }
            if (header) {
                List<String> headerValues;
                if (values.size() > 0) {
                    headerValues = new ArrayList<>(values.size());

                    for (String value: values) {
                        headerValues.add(subKey + ":" + value);
                    }
                } else {
                    headerValues = Collections.singletonList(subKey + ":");
                }

                values = headerValues;
            } else {
                if (operator == FieldMatchOperator.EXISTS || operator == FieldMatchOperator.NOT_EXISTS) {
                    throw new ParseException("exists operator is not suitable for field " + fieldGroup.value(), -1);
                }
            }
            conditions.add(
                new FieldCondition(
                    luceneFields,
                    operator,
                    values));
        }


        if (orList != null) {
            conditions.add(parseLogicList(orList,  LogicOperation.OR, context));
        }

        if (andList != null) {
            conditions.add(parseLogicList(andList,  LogicOperation.AND, context));
        }

        if (conditions.size() == 0) {
            conditions.add(AllMatchCondition.INSTANCE);
            //throw new ParseException("Zero condition list", -1);
        }
        return new LogicCondition(logicOperation, conditions);
    }

    private FieldMatchOperator operatorByPredicate(
        final CheckPredicate checkPredicate,
        final List<String> values)
        throws ParseException, JsonException
    {
        FieldMatchOperator operator;
        switch (checkPredicate) {
            case EXISTS:
                throw new ParseException("Exists not allowed for values", -1);
            case CONTAINS:
                if (values.size() <= 1) {
                    operator = FieldMatchOperator.CONTAINS_ONE;
                } else {
                    operator = FieldMatchOperator.CONTAINS_ANY;
                }

                break;
            case EQUAL:
                if (values.size() <= 1) {
                    operator = FieldMatchOperator.EQUAL_ONE;
                } else {
                    operator = FieldMatchOperator.EQUAL_ANY;
                }

                break;
            case NOT_EQUAL:
                if (values.size() <= 1) {
                    operator = FieldMatchOperator.NOT_EQUAL_ONE;
                } else {
                    operator = FieldMatchOperator.NOT_EQUAL_ANY;
                }
                break;
            case NOT_CONTAINS:
                if (values.size() <= 1) {
                    operator = FieldMatchOperator.NOT_CONTAINS_ONE;
                } else {
                    operator = FieldMatchOperator.NOT_CONTAINS_ANY;
                }
                break;
            default:
                throw new ParseException("Unsupported predicate " + checkPredicate.name(), -1);
        }
        return operator;
    }


    private Map.Entry<FieldMatchOperator, List<String>> parsePredicateOrValue(final JsonMap map) throws JsonException, ParseException {
        if (map.size() != 1) {
            throw new ParseException(
                "Expecting one predicate, but got  " + JsonType.NORMAL.toString(map),
                0);
        }

        Map.Entry<String, JsonObject> entry = map.entrySet().iterator().next();
        //String trimmed = entry.getKey().toLowerCase(Locale.ENGLISH).trim();
        CheckPredicate checkPredicate = PREDICATES.get(entry.getKey());
        if (checkPredicate == null) {
            return parseSubconditionOrValue(CheckPredicate.EQUAL, map);
            // here in could be just value or sub predicate;
            // ok trying to parse value instead
//            List<String> values = parseValue(map);
//            if (values.size() != 1) {
//                throw new ParseException("Only one value for condition is allowed here", -1);
//            }
//
//            return new AbstractMap.SimpleEntry<>(FieldMatchOperator.EQUAL_ONE, values);
        }

        FieldMatchOperator operator;
        List<String> values;
        switch (entry.getValue().type()) {
            case NULL:
                throw new ParseException(
                    entry.getValue().type().name()
                        + " is not allowed as field value with predicate "
                        + entry.getKey(), 0);
            case LIST:
                values = parseValueList(entry.getValue().asList());
                operator = operatorByPredicate(checkPredicate, values);
                break;
            case MAP:
                if (checkPredicate.subPredicates() == null || checkPredicate.subPredicates().isEmpty()) {
                    values = parseValue(entry.getValue().asMap());
                    operator = operatorByPredicate(checkPredicate, values);
                } else {
                    // parse subpredicate
                    return parseSubconditionOrValue(checkPredicate, entry.getValue().asMap());
                }
                break;
            default:
                switch (checkPredicate) {
                    case EXISTS:
                        if (entry.getValue().type() != JsonObject.Type.BOOLEAN) {
                            throw new ParseException("Exists requires boolean value, but got " + entry.getValue().type(), -1);
                        }
                        boolean existsValue = entry.getValue().asBoolean();
                        if (!existsValue) {
                            operator = FieldMatchOperator.NOT_EXISTS;
                        } else {
                            operator = FieldMatchOperator.EXISTS;
                        }
                        values = Collections.emptyList();
                        break;
                    case CONTAINS:
                        values = Collections.singletonList(entry.getValue().asString());
                        operator = FieldMatchOperator.CONTAINS_ONE;
                        break;
                    case EQUAL:
                        values = Collections.singletonList(entry.getValue().asString());
                        operator = FieldMatchOperator.EQUAL_ONE;
                        break;
                    case NOT_EQUAL:
                        values = Collections.singletonList(entry.getValue().asString());
                        operator = FieldMatchOperator.NOT_EQUAL_ONE;
                        break;
                    case NOT_CONTAINS:
                        values = Collections.singletonList(entry.getValue().asString());
                        operator = FieldMatchOperator.NOT_CONTAINS_ONE;
                        break;
                    default:
                        throw new ParseException("Unsupported predicate " + checkPredicate.name(), -1);
                }
                break;
        }

        return new AbstractMap.SimpleEntry<>(operator, values);
    }

    private List<String> parseValueList(final JsonList list) throws ParseException, JsonException {
        List<String> values = new ArrayList<>(list.size());
        for (JsonObject jo: list) {
            switch (jo.type()) {
                case MAP:
                    values.addAll(parseValue(jo.asMap()));
                    break;
                case LIST:
                    throw new ParseException("List is not allowed in values list", -1);
                default:
                    values.add(jo.asString());
            }
        }
        if (values.size() == 0) {
            throw new ParseException("Values list is empty", -1);
        }

        return values;
    }
    private List<String> parseValue(final JsonMap map) throws ParseException, JsonException {
        String value = map.getString("$base64", null);
        if (value == null) {
            throw new ParseException("expecting base64 value, but got nothing", 0);
        }
        return Collections.singletonList(new String(Base64.getDecoder().decode(value), StandardCharsets.UTF_8));
    }

    private Map.Entry<FieldMatchOperator, List<String>> parseSubconditionOrValue(
        final CheckPredicate checkPredicate,
        final JsonMap map)
        throws JsonException, ParseException
    {
        if (map.size() != 1) {
            throw new ParseException("Expecting \"all\" or \"any\" but got " + JsonType.NORMAL.toString(map), 0);
        }

        Map.Entry<String, JsonObject> entry = map.entrySet().iterator().next();
        String key = entry.getKey().toLowerCase(Locale.ENGLISH);
        SubPredicate subPredicate = SUBPREDICATES.get(key);
        if (subPredicate == null) {
            // here it could be just value
            List<String> values = parseValue(map);
            FieldMatchOperator operator = operatorByPredicate(checkPredicate, values);
            return new AbstractMap.SimpleEntry<>(operator, values);
            //throw new ParseException("Expecting \"$all\" or \"$any\" but got " + key, 0);
        }

        if (entry.getValue().type() != JsonObject.Type.LIST) {
            throw new ParseException("Expecting array for key " + key, 0);
        }

        JsonList list = entry.getValue().asList();
        List<String> values = parseValueList(list);

        FieldMatchOperator operator;
        switch (checkPredicate) {
            case EXISTS:
                throw new ParseException("Exists not allowed for values", -1);
            case CONTAINS:
                switch (subPredicate) {
                    case ALL:
                        operator = FieldMatchOperator.CONTAINS_ALL;
                        break;
                    case ANY:
                        operator = FieldMatchOperator.CONTAINS_ANY;
                        break;
                    default:
                        throw new ParseException("Unsupported sub predicate " + subPredicate.name(), -1);
                }
                break;
            case EQUAL:
                switch (subPredicate) {
                    case ANY:
                        operator = FieldMatchOperator.EQUAL_ANY;
                        break;
                    default:
                        throw new ParseException("Unsupported sub predicate " + subPredicate.name(), -1);
                }

                break;
            case NOT_EQUAL:
                switch (subPredicate) {
                    case ALL:
                        operator = FieldMatchOperator.NOT_EQUAL_ALL;
                        break;
                    case ANY:
                        operator = FieldMatchOperator.NOT_EQUAL_ANY;
                        break;
                    default:
                        throw new ParseException("Unsupported sub predicate " + subPredicate.name(), -1);
                }
                break;
            case NOT_CONTAINS:
                switch (subPredicate) {
                    case ALL:
                        operator = FieldMatchOperator.NOT_CONTAINS_ALL;
                        break;
                    case ANY:
                        operator = FieldMatchOperator.NOT_CONTAINS_ANY;
                        break;
                    default:
                        throw new ParseException("Unsupported sub predicate " + subPredicate.name(), -1);
                }
                break;
            default:
                throw new ParseException("Unsupported predicate " + checkPredicate.name(), -1);
        }
        return new AbstractMap.SimpleEntry<>(operator, values);
    }

    public static class ParseContext {
        private final Set<String> requiredFields = new LinkedHashSet<>();

        public ParseContext() {
        }

        public Set<String> requiredFields() {
            return requiredFields;
        }
    }

    private enum FieldGroup {
        FROM,
        TO,
        CC,
        TOCC,
        BODY,
        HEADER(SubkeyAnyNonEmptyParser.INSTANCE),
        FILENAME,
        ATTACH(SubkeyFieldValidator.of(FILENAME)),
        ADDRESS(SubkeyFieldValidator.of(FROM, TO, CC, TOCC)),
        SUBJECT;

        private final boolean subKeyPossible;
        private final String value;
        private final GenericFunction<String, ? extends Object, ParseException> subkeyParser;

        FieldGroup() {
            this.subkeyParser = null;
            this.subKeyPossible = false;
            this.value = name().toLowerCase(Locale.ENGLISH);
        }

        FieldGroup(final GenericFunction<String, ? extends Object, ParseException> subkeyParser) {
            this.subkeyParser = subkeyParser;
            this.subKeyPossible = true;
            this.value = name().toLowerCase(Locale.ENGLISH);
        }

        public String value() {
            return value;
        }

        public boolean subKeyPossible() {
            return subKeyPossible;
        }

        public GenericFunction<String, ? extends Object, ParseException> subkeyParser() {
            return subkeyParser;
        }
    }

    public enum CheckPredicate {
        EQUAL("$eq", SubPredicate.ANY),
        NOT_EQUAL("$ne", true, SubPredicate.ALL, SubPredicate.ANY),
        CONTAINS(SubPredicate.ALL, SubPredicate.ANY),
        NOT_CONTAINS("$not-contains", true, SubPredicate.ALL, SubPredicate.ANY),
        EXISTS("$exists", false),
        NOT_EXISTS("$not-exists", true);

        private final String value;
        private final Set<SubPredicate> subPredicates;
        private final boolean negation;

        CheckPredicate(final SubPredicate... subs) {
            this.value = '$' + name().toLowerCase(Locale.ENGLISH);
            this.subPredicates = Collections.unmodifiableSet(new LinkedHashSet<>(Arrays.asList(subs)));
            this.negation = false;
        }

        CheckPredicate(final String value, final SubPredicate... subs) {
            this.value = value;
            this.subPredicates = Collections.unmodifiableSet(new LinkedHashSet<>(Arrays.asList(subs)));
            this.negation = false;
        }

        CheckPredicate(
            final String value,
            final boolean negation,
            final SubPredicate... subs)
        {
            this.value = value;
            this.subPredicates = Collections.unmodifiableSet(new LinkedHashSet<>(Arrays.asList(subs)));
            this.negation = negation;
        }

        public boolean negation() {
            return negation;
        }

        public String value() {
            return value;
        }

        public Set<SubPredicate> subPredicates() {
            return subPredicates;
        }
    }

    public enum SubPredicate {
        ANY,
        ALL;

        private final String value;

        SubPredicate() {
            this.value = '$' + name().toLowerCase(Locale.ENGLISH);
        }

        public String value() {
            return value;
        }
    }


    private enum SubkeyAnyNonEmptyParser implements GenericFunction<String, String, ParseException> {
        INSTANCE;

        @Override
        public String apply(final String s) throws ParseException {
            if (s == null || s.isBlank()) {
                throw new ParseException("Subkey is not supposed to be empty", 0);
            }

            return s.trim();
        }
    }
    private static class SubkeyFieldValidator implements GenericFunction<String, FieldGroup, ParseException> {
        private final Set<FieldGroup> possible;
        private final WeirdEnumParser<FieldGroup> parser = new WeirdEnumParser<>(FieldGroup.class);

        public SubkeyFieldValidator(final FieldGroup... fields) {
            this.possible = Collections.unmodifiableSet(new LinkedHashSet<>(Arrays.asList(fields)));
        }

        @Override
        public FieldGroup apply(final String o) throws ParseException {
            FieldGroup group = parser.apply(o);
            if (possible.contains(group)) {
                return group;
            }

            throw new ParseException(
                group.value() + " is not allowed here, possible are "
                    + possible.stream().map(FieldGroup::value).collect(Collectors.joining()),
                0);
        }

        public static SubkeyFieldValidator of(final FieldGroup... fields) {
            return new SubkeyFieldValidator(fields);
        }
    }

    // according to grammar
    private static class WeirdEnumParser<E extends Enum<E>>
        implements GenericFunction<String, E, ParseException>
    {
        private final Class<E> e;

        public WeirdEnumParser(final Class<E> e) {
            this.e = e;
        }

        @Override
        public E apply(final String value) throws ParseException {
            if (!org.apache.commons.lang3.StringUtils.isAllLowerCase(value)) {
                throw new ParseException("unknown field " + value, -1);
            }

            return Enum.valueOf(
                e,
                value.toUpperCase(Locale.ROOT));
        }
    }

    private enum LogicOperation implements StringBuilderable {
        OR,
        AND;

        private final String value;
        private final String luceneValue;

        LogicOperation() {
            this.value = name().toLowerCase(Locale.ENGLISH);
            this.luceneValue = name().toUpperCase(Locale.ENGLISH);
        }

        public String value() {
            return value;
        }

        @Override
        public void toStringBuilder(@Nonnull final StringBuilder sb) {
            sb.append(' ');
            sb.append(luceneValue);
            sb.append(' ');
        }
    }

    public interface Condition extends StringBuilderable {

    }

    private static final class LogicCondition implements Condition {
        private LogicOperation operation;
        private List<Condition> conditions;

        public LogicCondition(final LogicOperation operation, final List<Condition> conditions) {
            this.operation = operation;
            this.conditions = conditions;
        }

        @Override
        public void toStringBuilder(@Nonnull final StringBuilder sb) {
            sb.append('(');
            for (int i = 0; i < conditions.size(); i++) {
                if (i != 0) {
                    operation.toStringBuilder(sb);
                }
                conditions.get(i).toStringBuilder(sb);
            }
            sb.append(')');
        }
    }
}

