package ru.yandex.search.mail.tupita.config;

import java.io.File;
import java.io.IOException;

import ru.yandex.client.tvm2.Tvm2ClientConfig;
import ru.yandex.client.tvm2.Tvm2ClientConfigBuilder;
import ru.yandex.http.config.HttpHostConfig;
import ru.yandex.http.config.HttpHostConfigBuilder;
import ru.yandex.mail.search.logbroker.logger.config.LogbrokerLoggerConfig;
import ru.yandex.mail.search.logbroker.logger.config.LogbrokerLoggerConfigBuilder;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.search.mail.tupita.QueryParseErrorPolicy;
import ru.yandex.search.proxy.universal.AbstractUniversalSearchProxyConfigBuilder;

public abstract class AbstractTupitaConfigBuilder
    <T extends AbstractTupitaConfigBuilder<T>>
     extends AbstractUniversalSearchProxyConfigBuilder<T>
     implements TupitaConfig
{
    private HttpHostConfigBuilder tikaiteConfig;
    private HttpHostConfigBuilder fatcheckerConfig;
    private IniConfig luceneConfig;
    private Tvm2ClientConfigBuilder tvm2ClientConfig;

    private int fatRequestLengthThreshold;
    private int coreQueryParsingThreads;
    private int maxQueryParsingThreads;
    private int batchParseQueueSize;
    private int charsInBatch;
    private QueryParseErrorPolicy defaultParseErrorPolicy;
    private LogbrokerLoggerConfigBuilder logbrokerLogger;
    private boolean subscriptionsPumpkin;

    protected AbstractTupitaConfigBuilder(final TupitaConfig config) {
        super(config);
        tikaiteConfig(config.tikaiteConfig());
        fatcheckerConfig(config.fatcheckerConfig());
        luceneConfig(config.luceneConfig());
        tvm2ClientConfig(config.tvm2ClientConfig());
        fatRequestLengthThreshold(config.fatRequestLengthThreshold());
        charsInBatch(config.charsInBatch());
        coreQueryParsingThreads(config.coreQueryParsingThreads());
        maxQueryParsingThreads(config.maxQueryParsingThreads());
        batchParseQueueSize(config.batchParseQueueSize());
        defaultParseErrorPolicy(config.defaultParseErrorPolicy());
        logbrokerLogger(config.logbrokerLogger());
        subscriptionsPumpkin(config.subscriptionsPumpkin());
    }

    protected AbstractTupitaConfigBuilder(
        final IniConfig config,
        final TupitaConfig defaults)
        throws ConfigException
    {
        super(config, defaults);

        tikaiteConfig =
            new HttpHostConfigBuilder(
                config.section("tikaite"),
                defaults.tikaiteConfig());

        fatcheckerConfig =
            new HttpHostConfigBuilder(
                config.section("fat"),
                defaults.tikaiteConfig());
        luceneConfig = config.sectionOrNull("lucene");
        if (luceneConfig != null) {
            String configPath =
                luceneConfig.getString("config-path", null);
            if (configPath != null) {
                try {
                    luceneConfig = new IniConfig(new File(configPath));
                } catch (IOException e) {
                    throw new ConfigException("Failed to load lucene conf", e);
                }
            }
        }
        IniConfig batchSection =
            config.sectionOrNull("batched-queries-parsing");
        if (batchSection != null) {
            fatRequestLengthThreshold =
                batchSection.getInt(
                    "fat-threshold-bytes",
                    defaults.fatRequestLengthThreshold());
            charsInBatch =
                batchSection.getInt(
                    "chars-in-batch",
                    defaults.fatRequestLengthThreshold());
            coreQueryParsingThreads =
                batchSection.getInt(
                    "core-threads",
                    defaults.coreQueryParsingThreads());
            maxQueryParsingThreads =
                batchSection.getInt(
                    "max-threads",
                    defaults.maxQueryParsingThreads());
            batchParseQueueSize =
                batchSection.getInt(
                    "queue-size",
                    defaults.batchParseQueueSize());
        } else {
            fatRequestLengthThreshold = defaults.fatRequestLengthThreshold();
            charsInBatch = defaults.charsInBatch();
            coreQueryParsingThreads = defaults.coreQueryParsingThreads();
            maxQueryParsingThreads = defaults.maxQueryParsingThreads();
            batchParseQueueSize = defaults.batchParseQueueSize();
        }

        tvm2ClientConfig =
            new Tvm2ClientConfigBuilder(
                config.section("tvm2"),
                defaults.tvm2ClientConfig());

        this.defaultParseErrorPolicy =
            config.getEnum(
                QueryParseErrorPolicy.class,
                "default-parse-error-policy",
                defaults.defaultParseErrorPolicy());

        logbrokerLogger =
            new LogbrokerLoggerConfigBuilder(
                config.section("results-logger"),
                defaults.logbrokerLogger());

        subscriptionsPumpkin =
            config.getBoolean(
                "subscriptions-pumpkin",
                defaults.subscriptionsPumpkin());
    }

    @Override
    public HttpHostConfigBuilder tikaiteConfig() {
        return tikaiteConfig;
    }

    public T tikaiteConfig(final HttpHostConfig tikaiteConfig) {
        this.tikaiteConfig = new HttpHostConfigBuilder(tikaiteConfig);
        return self();
    }

    @Override
    public HttpHostConfigBuilder fatcheckerConfig() {
        return fatcheckerConfig;
    }

    public T fatcheckerConfig(final HttpHostConfig config) {
        this.fatcheckerConfig = new HttpHostConfigBuilder(config);
        return self();
    }

    @Override
    public IniConfig luceneConfig() {
        return luceneConfig;
    }

    public T luceneConfig(final IniConfig luceneConfig) {
        this.luceneConfig = luceneConfig;
        return self();
    }

    @Override
    public Tvm2ClientConfigBuilder tvm2ClientConfig() {
        return tvm2ClientConfig;
    }

    public T tvm2ClientConfig(final Tvm2ClientConfig tvm2ClientConfig) {
        this.tvm2ClientConfig = new Tvm2ClientConfigBuilder(tvm2ClientConfig);
        return self();
    }

    @Override
    public int fatRequestLengthThreshold() {
        return fatRequestLengthThreshold;
    }

    public T fatRequestLengthThreshold(final int fatRequestLengthThreshold) {
        this.fatRequestLengthThreshold = fatRequestLengthThreshold;
        return self();
    }

    @Override
    public int charsInBatch() {
        return charsInBatch;
    }

    public T charsInBatch(final int charsInBatch) {
        this.charsInBatch = charsInBatch;
        return self();
    }

    @Override
    public int coreQueryParsingThreads() {
        return coreQueryParsingThreads;
    }

    public T coreQueryParsingThreads(final int coreQueryParsingThreads) {
        this.coreQueryParsingThreads = coreQueryParsingThreads;
        return self();
    }

    @Override
    public int maxQueryParsingThreads() {
        return maxQueryParsingThreads;
    }

    public T maxQueryParsingThreads(final int maxQueryParsingThreads) {
        this.maxQueryParsingThreads = maxQueryParsingThreads;
        return self();
    }

    @Override
    public int batchParseQueueSize() {
        return batchParseQueueSize;
    }

    public T batchParseQueueSize(final int batchParseQueueSize) {
        this.batchParseQueueSize = batchParseQueueSize;
        return self();
    }

    @Override()
    public QueryParseErrorPolicy defaultParseErrorPolicy() {
        return defaultParseErrorPolicy;
    }

    public T defaultParseErrorPolicy(final QueryParseErrorPolicy value) {
        this.defaultParseErrorPolicy = value;
        return self();
    }

    @Override
    public LogbrokerLoggerConfigBuilder logbrokerLogger() {
        return logbrokerLogger;
    }

    public T logbrokerLogger(
        final LogbrokerLoggerConfig logbrokerLogger)
    {
        this.logbrokerLogger =
            new LogbrokerLoggerConfigBuilder(logbrokerLogger);
        return self();
    }

    @Override
    public boolean subscriptionsPumpkin() {
        return subscriptionsPumpkin;
    }

    public T subscriptionsPumpkin(
        final boolean subscriptionsPumpkin)
    {
        this.subscriptionsPumpkin = subscriptionsPumpkin;
        return self();
    }
}
