package ru.yandex.search.mail.tupita.fat;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.apache.http.HttpException;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.protocol.HttpContext;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.search.mail.tupita.AbstractQueryCheckHandler;
import ru.yandex.search.mail.tupita.AsyncParseTupitaQuery;
import ru.yandex.search.mail.tupita.CleaningCallback;
import ru.yandex.search.mail.tupita.TikaiteTupitaIndexer;
import ru.yandex.search.mail.tupita.Tupita;
import ru.yandex.search.mail.tupita.TupitaIndexationContext;
import ru.yandex.search.mail.tupita.TupitaMailMetaInfo;
import ru.yandex.search.mail.tupita.TupitaProxySession;

public class FatQueryCheckHandler extends AbstractQueryCheckHandler {
    private static final int MAX_BATCH_SIZE = 50;

    public FatQueryCheckHandler(final Tupita tupita) {
        super(tupita);
    }

    @Override
    public void handle(
        final JsonObject data,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException, IOException
    {
        TupitaProxySession session =
            new TupitaProxySession(tupita, exchange, context);

        session.logger().info("Parsing request");

        try {
            JsonMap root = data.asMap();
            JsonMap message = root.getMap("message");

            FatQueryCheckCallback callback = new FatQueryCheckCallback(session);
            JsonList queries = root.getList("queries");
            if (queries.size() == 0) {
                session.logger().warning("No queries, skipping");
                callback.completed(Collections.emptyList());
                return;
            }

            TupitaMailMetaInfo meta =
                parseMessage(session, message, tupita.nextPrefix());

            TupitaIndexationContext indexationContext =
                new TupitaIndexationContext(tupita, session, meta);

            CleaningCallback<Collection<String>> cleaningCallback =
                new CleaningCallback<>(indexationContext, callback);

            List<AsyncParseTupitaQuery> batch = new ArrayList<>();
            int batchesScheduled = 0;

            List<List<AsyncParseTupitaQuery>> batches = new ArrayList<>();
            int length = 0;
            for (int i = 0; i < queries.size(); i++) {
                boolean last = i == (queries.size() - 1);

                AsyncParseTupitaQuery query =
                    parseQueryJson(indexationContext, queries.get(i), i);
                if (query != null) {
                    length += query.rawQuery().length();
                    batch.add(query);
                }

                if (last
                    || batch.size() > MAX_BATCH_SIZE
                    || length > tupita.config().fatRequestLengthThreshold())
                {
                    batches.add(batch);
                    length = 0;

                    if (!last) {
                        batch = new ArrayList<>();
                    }
                }
            }

            PrefixedLogger logger = session.logger();
            logger.info(
                "Batching parsing queries, total is "
                    + queries.size() + " btaches count " + batches.size());

            // point of sync for indexation/parsed queries
            final AsyncParseQueriesChecker checker =
                new AsyncParseQueriesChecker(
                    indexationContext,
                    batches.size(),
                    cleaningCallback);

            for (List<AsyncParseTupitaQuery> queriesToParse: batches) {
                tupita.lucene().scheduleQueryParse(
                    checker.newParseTask(queriesToParse));
            }

            // indexation scheduled
            new TikaiteTupitaIndexer(tupita).index(
                indexationContext,
                cleaningCallback.adjustIndexCallback(checker));

            logger.info("Batches scheduled " + batchesScheduled);
        } catch (JsonException e) {
            throw new BadRequestException(
                "Unable to parse message "
                    + JsonType.HUMAN_READABLE.toString(data),
                e);
        }
    }

    protected AsyncParseTupitaQuery parseQueryJson(
        final TupitaIndexationContext context,
        final JsonObject queryItem,
        final int index)
        throws JsonException
    {
        JsonMap queryMap = queryItem.asMap();
        String id = queryMap.getString("id");
        String queryStr = queryMap.getString("query");
        boolean stop = queryMap.getBoolean("stop", false);

        if (queryStr == null || queryStr.isEmpty()) {
            context.session().logger().warning(
                "Query is empty " + JsonType.NORMAL.toString(queryItem));
            return null;
        }

        return new AsyncParseTupitaQuery(id, queryStr, index, stop);
    }
}
