package ru.yandex.search.messenger.indexer;

import java.io.IOException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.Utf8JsonWriter;
import ru.yandex.util.string.StringUtils;

public class ChatInfo extends ChatMessage {
    protected final long version;
    protected final double createTimestamp;
    protected final String avatarId;
    protected List<String> members = null;
    protected final boolean publicChat;
    protected final boolean privateChat;
    protected List<String> usersPermissions = null;
    protected List<String> departmentsPermissions = null;
    protected List<String> groupsPermissions = null;
    protected List<String> adminRoles = null;
    protected final String inviteHash;
    protected final String avatar;
    protected final String name;
    protected final String alias;
    protected final String description;
    protected final String iconId;
    protected final double longitude;
    protected final double latitude;
    protected final boolean geo;
    protected final JsonMap json;
    protected final long orgId;
    protected final String geoType;
    protected final Long geoId;
    protected final String entityId;
    protected final Integer subservice;
    protected final Integer namespace;
    protected final String parentUrl;
    protected final boolean showOnMorda;
    protected final boolean channel;

    public ChatInfo(final JsonMap json)
        throws JsonException, ParseException {
        super(json.getOrNull(MessengerChatsHandler.CHAT_ID), true);
        this.json = json;
        version = json.getLong("version");
        createTimestamp = json.getDouble("create_timestamp");
        avatarId = json.getOrNull("raw_avatar_id");
        publicChat = json.getBoolean("public", false);
        privateChat = json.getBoolean("private", false);
        showOnMorda = json.getBoolean("show_on_morda", false);
        channel = json.getBoolean("channel", false);
        inviteHash = json.getOrNull("invite_hash");
        avatar = json.getOrNull("avatar");
        name = json.getOrNull("name");
        alias = json.getOrNull("alias");
        iconId = json.getOrNull("icon_id");
        description = json.getOrNull("description");
        orgId = json.getLong("org_id");
        geoType = json.getOrNull("geo_type");
        entityId = json.getOrNull("entity_id");
        subservice = json.getInt("subservice", null);
        namespace = json.getInt("namespace", 0);
        parentUrl = json.getOrNull("parent_url");

        JsonObject geoId = json.get("geo_id");
        if (geoId != null && geoId.type() != JsonObject.Type.NULL) {
            this.geoId = geoId.asLong();
        } else {
            this.geoId = null;
        }

        parseMembers(json);

        parsePermissions(json);

        parseRoles(json);

        JsonObject lon = json.get("longitude");
        JsonObject lat = json.get("latitude");
        if ((lon != null && lon.type() != JsonObject.Type.NULL)
                && (lat != null && lat.type() != JsonObject.Type.NULL)) {
            longitude = lon.asDouble();
            latitude = lat.asDouble();
            geo = true;
        } else {
            geo = false;
            longitude = 0.;
            latitude = 0.;
        }
    }

    private void parseMembers(final JsonMap json) throws JsonException {
        JsonList membersList = json.getListOrNull("members");
        if (membersList != null) {
            members = new ArrayList<>(membersList.size());
            for (JsonObject member : membersList) {
                members.add(member.asString());
            }
        } else {
            members = Collections.emptyList();
        }
    }

    private List<String> getUsers(final JsonList src) throws JsonException {
        List<String> dest;
        if (src == null) {
            dest = Collections.emptyList();
        } else {
            dest = new ArrayList<>();
            for (JsonObject user : src) {
                dest.add(user.asString());
            }
        }
        return dest;
    }

    private void parsePermissions(final JsonMap json) throws JsonException {
        JsonMap permissions = json.getMapOrNull("permissions");
        if (permissions != null) {
            usersPermissions = getUsers(permissions.getListOrNull("users"));
            groupsPermissions = getUsers(
                permissions.getListOrNull("groups"));
            departmentsPermissions = getUsers(
                permissions.getListOrNull("departments"));
        } else {
            usersPermissions = getUsers(
                json.getListOrNull("permission_users"));
            departmentsPermissions = getUsers(
                json.getListOrNull("permission_departments"));
            groupsPermissions = getUsers(
                json.getListOrNull("permission_groups"));
        }
    }

    private void parseRoles(final JsonMap json) throws JsonException {
        JsonMap roles = json.getMapOrNull("roles");
        if (roles != null) {
            adminRoles = getUsers(roles.getListOrNull("admin"));
        } else {
            adminRoles = getUsers(json.getListOrNull("role_users"));
        }
    }

    public List<String> members() {
        return members;
    }

    @Override
    public Collection<String> preserveFields() {
        return MessengerChatInfoUpdateHandler.PRESERVE_FIELDS;
    }

    @Override
    protected void writeDocumentFields(final Utf8JsonWriter writer)
        throws IOException {
        //leave this first, making this fields apears in index
        //first, so FieldsVisitor breaks early
        writer.key("chat_geo");
        writer.value(geo);
        if (geo) {
            writer.key("chat_latitude");
            writer.value(latitude);
            writer.key("chat_longitude");
            writer.value(longitude);
            writer.key("chat_latitude_raw");
            writer.value(latitude);
            writer.key("chat_longitude_raw");
            writer.value(longitude);
            writer.key("geo_shard");
            //CSOFF: MagicNumber
            int geoShard =
                (int) (((longitude + 180) * 180 + (latitude + 90)) * 10);
            writer.value(geoShard);
            //CSON: MagicNumber
        }
        writer.key(MessengerChatsHandler.CHAT_ID);
        writer.value(chatId);
        //leave
        writer.key("chat_version");
        writer.value(version);
        writer.key("chat_create_timestamp");
        writer.value(createTimestamp);
        writer.key("chat_avatar_id");
        writer.value(avatarId);
        writer.key("chat_private");
        writer.value(privateChat);
        writer.key("chat_public");
        writer.value(publicChat);
        writer.key("chat_invite_hash");
        writer.value(inviteHash);
        writer.key("chat_avatar");
        writer.value(avatar);
        writer.key("chat_name");
        writer.value(name);
        writer.key(ru.yandex.ps.search.messenger.ChatFields.ALIAS.stored());
        writer.value(alias);
        writer.key("chat_icon_id");
        writer.value(iconId);
        writer.key("chat_description");
        writer.value(description);

//            writer.key(CHAT_MEMBERS);
//            writer.value(StringUtils.join(members, '\n'));

        writer.key("chat_permissions_users");
        writer.value(StringUtils.join(usersPermissions, '\n'));
        writer.key("chat_permissions_groups");
        writer.value(StringUtils.join(groupsPermissions, '\n'));
        writer.key("chat_permissions_departments");
        writer.value(StringUtils.join(departmentsPermissions, '\n'));

        writer.key("chat_roles_admin");
        writer.value(StringUtils.join(adminRoles, '\n'));

        writer.key("chat_org_id");
        writer.value(orgId);

        if (geoId != null) {
            writer.key("chat_geo_id");
            writer.value(geoId);
        }

        writer.key("chat_geo_type");
        writer.value(geoType);

        writer.key("chat_entity_id");
        writer.value(entityId);

        writer.key("chat_subservice");
        writer.value(subservice);

        writer.key("chat_namespace");
        writer.value(namespace);

        writer.key("chat_parent_url");
        writer.value(parentUrl);

        writer.key("chat_show_on_morda");
        writer.value(showOnMorda);

        writer.key(ru.yandex.ps.search.messenger.ChatFields.CHANNEL.stored());
        writer.value(channel);

        //whole blob
        writer.key("chat_data");
        writer.value(JsonType.NORMAL.toString(json));
    }

    @Override
    protected void writeGetFields(
        final Utf8JsonWriter writer,
        final Set<String> fields)
        throws IOException {
        for (String field : fields) {
            switch (field) {
                case MessengerChatsHandler.CHAT_MEMBERS:
                    writer.key(MessengerChatsHandler.CHAT_MEMBERS);
                    writer.value(members);
                    break;
                default:
                    break;
            }
        }
    }

    @Override
    public String uri(final String args) {
        return "/update?chat-id=" + chatId + args;
    }
}
