package ru.yandex.search.messenger.indexer;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.text.ParseException;

import NMessengerProtocol.Message.EGenericResponseStatus;
import NMessengerProtocol.Message.TMessageInfoRequest;
import NMessengerProtocol.Message.TMessageInfoResponse;
import NMessengerProtocol.Message.TOutMessage;
import com.google.protobuf.CodedOutputStream;
import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NByteArrayEntity;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.cityhash.CityHashingArrayOutputStream;
import ru.yandex.collection.IntPair;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.HttpStatusPredicates;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.NByteArrayEntityAsyncConsumerFactory;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.StatusCodeAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;

@SuppressWarnings("FutureReturnValueIgnored")
public class GetMessageHandler implements ProxyRequestHandler {
    private static final int VERSION = 2;

    protected final ThreadLocal<ByteArrayOutputStream> baosTls =
        ThreadLocal.<ByteArrayOutputStream>withInitial(
            () -> new ByteArrayOutputStream());
    private final ThreadLocal<CityHashingArrayOutputStream> outTls =
        ThreadLocal.<CityHashingArrayOutputStream>withInitial(
            () -> new CityHashingArrayOutputStream());
    private final String uri;
    private final HttpHost host;
    private final Malo malo;

    public GetMessageHandler(final Malo malo) {
        this.malo = malo;
        uri = malo.config().messages().uri().getPath();
        host = malo.config().messages().host();
    }

    @Override
    public void handle(final ProxySession session)
        throws HttpException, IOException
    {
        final String chatId = session.params().getString("chat-id");
        final long timestamp = session.params().getLong("timestamp");
        final MessageIndexSession messageSession =
            new MessageIndexSession(
                new MaloRequest(session),
                chatId,
                timestamp,
                null);
        session.logger().info("input CGI mesage: chat-id: "
            + chatId + ", timestamp: " + timestamp);
        final JsonType jsonType =
            JsonTypeExtractor.NULL.extract(session.params());

        AsyncClient client = malo.messagesClient().adjust(session.context());

        TMessageInfoRequest messageRequest =
            TMessageInfoRequest.newBuilder()
                .setChatId(chatId)
                .setTimestamp(timestamp)
                .build();

        CityHashingArrayOutputStream out = outTls.get();
        out.reset();
        CodedOutputStream googleOut = CodedOutputStream.newInstance(out);

        messageRequest.writeTo(googleOut);
        googleOut.flush();

        byte[] postData = out.toByteArrayWithVersion(VERSION);
//        Base64Encoder encoder = new Base64Encoder();
//        encoder.process(postData);
//        session.logger().info("Coded message: " + encoder.toString());
//        session.logger().info("ych: " + out.yandexCityHash());
//        session.logger().info("ch: " + out.cityHash());

        final BasicAsyncRequestProducerGenerator post =
            new BasicAsyncRequestProducerGenerator(
                uri,
                postData,
                ContentType.DEFAULT_BINARY);
        client.execute(
            host,
            post,
            new StatusCheckAsyncResponseConsumerFactory<IntPair<HttpEntity>>(
                HttpStatusPredicates.NON_PROTO_FATAL,
                new StatusCodeAsyncConsumerFactory<HttpEntity>(
                    NByteArrayEntityAsyncConsumerFactory.INSTANCE)),
            session.listener().createContextGeneratorFor(client),
            new MessageResponseCallback(messageSession, jsonType));
    }

    private byte[] messageToByteArray(final TOutMessage message)
        throws IOException
    {
        final ByteArrayOutputStream baos = baosTls.get();
        baos.reset();
        CodedOutputStream googleOut = CodedOutputStream.newInstance(baos);

        message.writeTo(googleOut);
        googleOut.flush();

        return baos.toByteArray();
    }

    private class MessageResponseCallback
        extends AbstractProxySessionCallback<IntPair<HttpEntity>>
    {
        private final JsonType jsonType;
        private final MessageIndexSession messageSession;

        MessageResponseCallback(
            final MessageIndexSession messageSession,
            final JsonType jsonType)
        {
            super(messageSession.session());
            this.messageSession = messageSession;
            this.jsonType = jsonType;
        }

        @Override
        public void completed(final IntPair<HttpEntity> response) {
            HttpEntity entity = response.second();
            int code = response.first();
            try {
                switch (code) {
                    case HttpStatus.SC_OK:
                        parseResponse(entity);
                        break;
                    case HttpStatus.SC_NOT_FOUND:
                        session.logger().info("Message with chatId: "
                            + messageSession.chatId() + ','
                            + ", timestamp:  "
                            + messageSession.timestamp() + " not found");
                        session.response(HttpStatus.SC_NOT_FOUND);
                        break;
                    default:
                        malo.badResponse(
                            uri,
                            "Message info getting error: code=" + code);
                }
            } catch (HttpException | IOException | ParseException e) {
                failed(e);
            }
        }

        private void parseResponse(final HttpEntity entity)
            throws IOException, HttpException, ParseException
        {
            InputStream is = entity.getContent();
            long skipped =
                is.skip(CityHashingArrayOutputStream.THEADER_SIZE);
            if (skipped != CityHashingArrayOutputStream.THEADER_SIZE) {
                malo.badResponse(
                    uri,
                    "Header skip failed");
            }
            TMessageInfoResponse response =
                TMessageInfoResponse.parseFrom(is);
            if (response.getStatus() != EGenericResponseStatus.Success) {
                malo.badResponse(
                    uri,
                    response.getStatus(),
                    response.getErrorInfo());
            }
            TOutMessage message = response.getMessage();
            if (jsonType == null) {
                session.response(
                    HttpStatus.SC_OK,
                    new NByteArrayEntity(messageToByteArray(message)));
            } else {
                try {
                    StringBuilderWriter sbw = new StringBuilderWriter();
                    try (JsonWriter writer = jsonType.create(sbw)) {
                        JsonObject json = ProtoUtils.protoToJson(message);
                        json.writeValue(writer);
                    }
                    session.response(
                        HttpStatus.SC_OK,
                        new NStringEntity(
                            sbw.toString(),
                            ContentType.APPLICATION_JSON
                                .withCharset(
                                    session.acceptedCharset())));
                } catch (IOException e) {
                    failed(e);
                }
            }
        }
    }
}

