package ru.yandex.search.messenger.indexer;

import java.io.IOException;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.Header;
import org.apache.http.HttpException;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.parser.string.CollectionParser;

public class IndexSession {
    protected final MaloRequest request;
    protected final String topic;
    protected final String partition;
    protected final String offset;
    protected final boolean ignorePosition;
    protected final Set<String> get;
    protected final Long zooQueueId;
    protected final String zooQueue;
    protected final Integer zooShardId;
    protected final boolean hasZoo;

    public IndexSession(final MaloRequest request)
        throws HttpException, IOException
    {
        this.request = request;
        topic = request.params().getString("topic");
        partition = request.params().getString("partition");
        offset = request.params().getString("offset");
        Header zooQueueIdHeader = request.session().request().getFirstHeader(YandexHeaders.ZOO_QUEUE_ID);
        if (zooQueueIdHeader != null) {
            Long queueId = null;
            try {
                queueId = Long.parseLong(zooQueueIdHeader.getValue());
            } catch (Exception e) {
                session().logger().log(Level.WARNING, "Failed to parse zoo queue id", e);
            }
            this.zooQueueId = queueId;
        } else {
            this.zooQueueId = null;
        }

        Header zooQueueHeader = request.session().request().getFirstHeader(YandexHeaders.ZOO_QUEUE);
        if (zooQueueHeader != null) {
            zooQueue = zooQueueHeader.getValue();
        } else {
            zooQueue = null;
        }

        Header zooShardIdHeader = request.session().request().getFirstHeader(YandexHeaders.ZOO_SHARD_ID);
        if (zooShardIdHeader != null) {
            Integer shardId = null;
            try {
                shardId = Integer.parseInt(zooShardIdHeader.getValue());
            } catch (Exception e) {
                session().logger().log(Level.WARNING, "Failed to parse zoo queue id", e);
            }
            this.zooShardId = shardId;
        } else {
            this.zooShardId = null;
        }

        hasZoo = zooQueue != null && zooQueueId != null && zooShardId != null;
        boolean ignorePosition =
            request.session().params().getBoolean("ignore-position", false);
        String zooShardId = request.session().headers().getString(
            YandexHeaders.ZOO_SHARD_ID,
            null);
        if (zooShardId != null) {
            long queueId = request.session().headers().getLong(
                YandexHeaders.ZOO_QUEUE_ID,
                (long) Integer.MIN_VALUE);
            //Ingore position if this is a consumer bathed request
            //do not ignore only if queueId is present
            if (queueId == Integer.MIN_VALUE) {
                ignorePosition = true;
            }
        }
        this.ignorePosition = ignorePosition;
        get = session().params().get(
            "get",
            new LinkedHashSet<>(),
                new CollectionParser<>(
                    String::trim,
                    LinkedHashSet::new,
                    ','));
    }

    public ProxySession session() {
        return request.session();
    }

    public String topic() {
        return topic;
    }

    public String partition() {
        return partition;
    }

    public String producerName() {
        if (hasZoo) {
            return zooQueue + '@' + zooShardId;
        } else {
            return partition + '@' + topic;
        }
    }

    public String producerPosition() {
        if (hasZoo) {
            return String.valueOf(zooQueueId);
        } else {
            return offset;
        }
    }

    public String offset() {
        return offset;
    }

    public boolean ignorePosition() {
        return ignorePosition;
    }

    public Logger logger() {
        return request.session().logger();
    }

    public Set<String> get() {
        return get;
    }

    public void info(final String name, final String value) {
        request.session().connection().setSessionInfo(name, value);
    }

    public void info(final MaloYtField field, final String value) {
        request.session().connection().setSessionInfo(field.fieldName(), value);
    }
}
