package ru.yandex.search.messenger.indexer;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import ru.yandex.json.writer.Utf8JsonValue;
import ru.yandex.json.writer.Utf8JsonWriter;

public abstract class IndexableMessage implements Utf8JsonValue {
    private final boolean addIfNotExists;
    protected List<IndexableMessage> mergedMessages = null;

    public IndexableMessage(final boolean addIfNotExists) {
        this.addIfNotExists = addIfNotExists;
    }

    public boolean hasBody() {
        return true;
    }

    public boolean getRequest() {
        return false;
    }

    public boolean writeIdField() {
        return true;
    }

    public String service() {
        return null;
    }

    public Collection<String> preserveFields() {
        return Collections.emptySet();
    }

    public void addMessage(final IndexableMessage message) {
        if (mergedMessages == null) {
            mergedMessages = new ArrayList<>();
        }
        mergedMessages.add(message);
    }

    @Override
    public void writeValue(final Utf8JsonWriter writer) throws IOException {
        if (multiMessage()) {
            throw new IOException("writeValue called on main multi message");
        }
        writer.startObject();
        writer.key("prefix");
        writer.value(prefix());
        if (addIfNotExists) {
            writer.key("AddIfNotExists");
            writer.value(true);
        }
        Collection<String> preserveFields = preserveFields();
        if (preserveFields != null && preserveFields.size() > 0) {
            writer.key("PreserveFields");
            writer.startArray();
            for (String field: preserveFields) {
                writer.value(field);
            }
            writer.endArray();
        }
        String updateRequest = updateRequest();
        if (updateRequest != null) {
            writer.key("query");
            writer.value(updateRequest);
        }
        writer.key("docs");
        writer.startArray();
        writeDocument(writer);
        if (mergedMessages != null) {
            for (IndexableMessage message: mergedMessages) {
                message.writeDocument(writer);
            }
        }
        writer.endArray();
        writer.endObject();
    }

    protected void writeDocument(final Utf8JsonWriter writer)
        throws IOException
    {
        writer.startObject();
        if (writeIdField()) {
            writer.key("id");
            writer.value(id());
        }
        writeDocumentFields(writer);
        writer.key("type");
        writer.value(type());
        writer.endObject();
    }

    protected abstract void writeGetFields(
        final Utf8JsonWriter writer,
        final Set<String> fields)
        throws IOException;

    protected abstract void writeDocumentFields(final Utf8JsonWriter writer)
        throws IOException;

    public abstract String uri(final String args);

    public abstract String id();

    public abstract String type();

    public boolean multiMessage() {
        return false;
    }

    public boolean multiService() {
        if (!multiMessage() || subMessages() == null | subMessages().size() <= 1) {
            return false;
        }

        Iterator<IndexableMessage> iterator = subMessages().iterator();
        String service = iterator.next().service();
        while (iterator.hasNext()) {
            if (iterator.next().service() != service) {
                return true;
            }
        }

        return false;
    }

    public String updateRequest() {
        return null;
    }

    public Collection<IndexableMessage> subMessages() {
        return null;
    }

//    public abstract long ctime();
    public String prefixHash() {
        return "0";
    }

    public String prefix() {
        return "0";
    }
}
