package ru.yandex.search.messenger.indexer;

import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URISyntaxException;
import java.nio.charset.UnsupportedCharsetException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;

import NMessengerProtocol.Message.EGenericResponseStatus;
import NMessengerProtocol.Message.TErrorInfo;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.HttpVersion;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.message.BasicHttpResponse;
import org.apache.http.protocol.HTTP;
import org.apache.james.mime4j.stream.EntityState;
import org.apache.james.mime4j.stream.Field;
import org.apache.james.mime4j.stream.MimeTokenStream;

import ru.yandex.client.tvm2.Tvm2TicketRenewalTask;
import ru.yandex.collection.Pattern;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.NotImplementedException;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.YandexReasonPhraseCatalog;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.server.UpstreamStater;
import ru.yandex.json.parser.JsonException;
import ru.yandex.mail.mime.DefaultMimeConfig;
import ru.yandex.mail.mime.OverwritingBodyDescriptorBuilder;
import ru.yandex.mail.mime.Utf8FieldBuilder;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.UriParser;
import ru.yandex.search.messenger.indexer.user.settings.SettingsUpdateHandler;
import ru.yandex.search.messenger.indexer.v2org.DepartmentsHandler;
import ru.yandex.search.messenger.indexer.v2org.GroupsHandler;

public class Malo
    extends HttpProxy<ImmutableMaloConfig>
    implements ProxyRequestHandler
{
    private static final int TEMP_BUFFER_SIZE = 256;
    private static final Collection<String> BANNED_RCA_URLS = Arrays.asList("pxl.leads.su", "trkleads.ru");
    private final AsyncClient chatsClient;
    private final AsyncClient messagesClient;
    private final AsyncClient producerClient;
    private final AsyncClient mailProducerClient;
    private final AsyncClient moxyClient;
    private final AsyncClient maloClient;
    private final AsyncClient rcaClient;
    private final AsyncClient cmntApiClient;
    private final HashMap<String, MessengerIndexHandlerBase<?, ?>> handlers;
    private final Tvm2TicketRenewalTask cmntApiTvm2RenewalTask;
    private final Cache<String, long[]> chatOrgsCache;

    public Malo(final ImmutableMaloConfig config)
        throws IOException, HttpException, JsonException, URISyntaxException
    {
        super(config);
        chatsClient = client("Chats", config.chats());
        messagesClient = client("Messages", config.messages());
        producerClient = client("Producer", config.producer());
        moxyClient = client("Moxy", config.producer());
        if (config.mailProducer() != null) {
            mailProducerClient = client("MailProducer", config.mailProducer());
            maloClient = client("Malo", config.producer());
        } else {
            mailProducerClient = null;
            maloClient = null;
        }
        if (config.rca() != null) {
            rcaClient = client("Rca", config.rca());
        } else {
            rcaClient = null;
        }
        if (config.cmntApi() != null) {
            cmntApiClient = client("CmtApi", config.cmntApi());
            if (config.cmntApiTvm2ClientConfig() != null) {
                cmntApiTvm2RenewalTask = new Tvm2TicketRenewalTask(
                    logger().addPrefix("tvm2.cmnt-api"),
                    serviceContextRenewalTask,
                    config.cmntApiTvm2ClientConfig());
            } else {
                cmntApiTvm2RenewalTask = null;
            }
        } else {
            cmntApiClient = null;
            cmntApiTvm2RenewalTask = null;
        }

        chatOrgsCache = CacheBuilder.newBuilder()
            .maximumSize(1000000)
            .build();
        UpstreamStater producerStater =
            new UpstreamStater(config.metricsTimeFrame(), "producer");
        MessengerChatsHandler chatsHandler =
            new MessengerChatsHandler(this, producerStater);
        MessengerUsersHandler usersHandler =
            new MessengerUsersHandler(this, producerStater);
        SettingsUpdateHandler usersBucketHandler =
                new SettingsUpdateHandler(this, producerStater);
        MessengerMessageHandler messagesHandler =
            new MessengerMessageHandler(this, producerStater);
        MessengerContactsHandler contactsHandler =
            new MessengerContactsHandler(this, producerStater);
        MessengerChatMembersDiffHandler membersDiffHandler =
            new MessengerChatMembersDiffHandler(this, producerStater);
        GroupsHandler groupsHandler =
            new GroupsHandler(this, producerStater);
        DepartmentsHandler departmentsHandler =
            new DepartmentsHandler(this, producerStater);
        register(new Pattern<>("", true), this, "POST");
        register(
            new Pattern<>("/index-chat", true),
            chatsHandler,
            RequestHandlerMapper.GET, RequestHandlerMapper.POST);
        register(
            new Pattern<>("/index-chat-members", true),
            membersDiffHandler,
            RequestHandlerMapper.GET, RequestHandlerMapper.POST);
        register(
            new Pattern<>("/update-chat-info", true),
            new MessengerChatInfoUpdateHandler(this, producerStater));
        if (rcaClient != null) {
            MessengerMessageInfoUpdateHandler msgUpdateHandler =
                new MessengerMessageInfoUpdateHandler(this, producerStater);
            registerStater(
                msgUpdateHandler.upstreamStater(config.metricsTimeFrame()));
            register(
                new Pattern<>("/update-message-info", true),
                msgUpdateHandler);
        }
        if (mailProducerClient != null) {
            register(
                new Pattern<>("/index-per-user", true),
                new MessengerIndexPerUserHandler(this));
        }
        register(
            new Pattern<>("/index-user", true),
            usersHandler,
            RequestHandlerMapper.GET, RequestHandlerMapper.POST);
        register(
            new Pattern<>("/index-search-bucket", true),
            usersBucketHandler,
            RequestHandlerMapper.GET, RequestHandlerMapper.POST);
        register(
            new Pattern<>("/index-message", true),
            messagesHandler,
            RequestHandlerMapper.GET, RequestHandlerMapper.POST);
        register(
            new Pattern<>("/index-contacts", true),
            contactsHandler,
            RequestHandlerMapper.GET, RequestHandlerMapper.POST);

        register(
            new Pattern<>("/get-message", true),
            new GetMessageHandler(this),
            RequestHandlerMapper.GET);

        register(
            new Pattern<>("/chat-members", true),
            new GetChatMembersHandler(this, producerStater),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/index-department", true),
            new DepartmentsHandler(this, producerStater),
            RequestHandlerMapper.GET, RequestHandlerMapper.POST);
        register(
            new Pattern<>("/index-group", true),
            new GroupsHandler(this, producerStater),
            RequestHandlerMapper.GET, RequestHandlerMapper.POST);

        handlers = new HashMap<>();

        handlers.put(
            "rt3.messenger--mssngr--test-search-messages",
            messagesHandler);
        handlers.put(
            "rt3.messenger--mssngr--test-search-chats",
            chatsHandler);
        handlers.put(
            "rt3.messenger--mssngr--test-search-chat-members",
            membersDiffHandler);
        handlers.put(
            "rt3.messenger--mssngr--test-search-users",
            usersHandler);
        handlers.put(
            "rt3.messenger--mssngr--test-search-contacts",
            contactsHandler);
        handlers.put(
            "rt3.messenger--mssngr--test-search-buckets",
            usersBucketHandler);

        // New test logbroker lbk
        handlers.put(
            "rt3.kafka-bs--mssngr@test@search--messages",
            messagesHandler);
        handlers.put(
            "rt3.kafka-bs--mssngr@test@search--chats",
            chatsHandler);
        handlers.put(
            "rt3.kafka-bs--mssngr@test@search--chat-members",
            membersDiffHandler);
        handlers.put(
            "rt3.kafka-bs--mssngr@test@search--users",
            usersHandler);
        handlers.put(
            "rt3.kafka-bs--mssngr@test@search--contacts",
            contactsHandler);
        handlers.put(
            "rt3.kafka-bs--mssngr@test@search--buckets",
            usersBucketHandler);

        handlers.put(
            "rt3.messenger--mssngr--prod-search-messages",
            messagesHandler);
        handlers.put(
            "rt3.messenger--mssngr--prod-search-chats",
            chatsHandler);
        handlers.put(
            "rt3.messenger--mssngr--prod-search-chat-members",
            membersDiffHandler);
        handlers.put(
            "rt3.messenger--mssngr--prod-search-users",
            usersHandler);
        handlers.put(
            "rt3.messenger--mssngr--prod-search-buckets",
            usersBucketHandler);
        handlers.put(
            "rt3.messenger--mssngr--prod-search-contacts",
            contactsHandler);

        // New prod logbroker lbk
        handlers.put(
            "rt3.kafka-bs--mssngr@prod@search--messages",
            messagesHandler);
        handlers.put(
            "rt3.kafka-bs--mssngr@prod@search--chats",
            chatsHandler);
        handlers.put(
            "rt3.kafka-bs--mssngr@prod@search--chat-members",
            membersDiffHandler);
        handlers.put(
            "rt3.kafka-bs--mssngr@prod@search--users",
            usersHandler);
        handlers.put(
            "rt3.kafka-bs--mssngr@prod@search--contacts",
            contactsHandler);
        handlers.put(
            "rt3.kafka-bs--mssngr@prod@search--buckets",
            usersBucketHandler);
        handlers.put(
            "rt3.kafka-bs--mssngr@prod@search--groups",
            groupsHandler);
        handlers.put(
            "rt3.kafka-bs--mssngr@prod@search--departments",
            departmentsHandler);

        handlers.put(
            "rt3.messenger--mssngr--alpha-search-messages",
            messagesHandler);
        handlers.put(
            "rt3.messenger--mssngr--alpha-search-chats",
            chatsHandler);
        handlers.put(
            "rt3.messenger--mssngr--alpha-search-chat-members",
            membersDiffHandler);
        handlers.put(
            "rt3.messenger--mssngr--alpha-search-users",
            usersHandler);
        handlers.put(
            "rt3.messenger--mssngr--alpha-search-contacts",
            contactsHandler);
        handlers.put(
            "rt3.messenger--mssngr--alpha-search-buckets",
            usersBucketHandler);

        // New alpha logbroker lbkx
        handlers.put(
            "rt3.kafka-bs--mssngr@alpha@search--messages",
            messagesHandler);
        handlers.put(
            "rt3.kafka-bs--mssngr@alpha@search--chats",
            chatsHandler);
        handlers.put(
            "rt3.kafka-bs--mssngr@alpha@search--chat-members",
            membersDiffHandler);
        handlers.put(
            "rt3.kafka-bs--mssngr@alpha@search--users",
            usersHandler);
        handlers.put(
            "rt3.kafka-bs--mssngr@alpha@search--contacts",
            contactsHandler);
        handlers.put(
            "rt3.kafka-bs--mssngr@alpha@search--buckets",
            usersBucketHandler);
        handlers.put(
            "rt3.kafka-bs--mssngr@alpha@search--groups",
            groupsHandler);
        handlers.put(
            "rt3.kafka-bs--mssngr@alpha@search--departments",
            departmentsHandler);

        registerStater(producerStater);
        registerStater(
            messagesHandler.upstreamStater(config.metricsTimeFrame()));
        registerStater(chatsHandler.upstreamStater(config.metricsTimeFrame()));
        registerStater(membersDiffHandler.upstreamStater(config.metricsTimeFrame()));
        registerStater(usersHandler.upstreamStater(config.metricsTimeFrame()));
        registerStater(
            contactsHandler.upstreamStater(config.metricsTimeFrame()));
    }

    @Override
    public void start() throws IOException {
        if (cmntApiTvm2RenewalTask != null) {
            cmntApiTvm2RenewalTask.start();
        }
        super.start();
    }

    @Override
    public void close() throws IOException {
        if (cmntApiTvm2RenewalTask != null) {
            cmntApiTvm2RenewalTask.cancel();
        }
        super.close();
    }

    public AsyncClient rcaClient() {
        return rcaClient;
    }

    public AsyncClient cmntApiClient() {
        return cmntApiClient;
    }

    public AsyncClient chatsClient() {
        return chatsClient;
    }

    public AsyncClient messagesClient() {
        return messagesClient;
    }

    public AsyncClient producerClient() {
        return producerClient;
    }

    public AsyncClient mailProducerClient() {
        return mailProducerClient;
    }

    public AsyncClient moxyClient() {
        return moxyClient;
    }

    public AsyncClient maloClient() {
        return maloClient;
    }

    public String cmntApiTvm2Ticket() {
        if (cmntApiTvm2RenewalTask != null) {
            return cmntApiTvm2RenewalTask.ticket();
        } else {
            return null;
        }
    }

    public Cache<String, long[]> chatOrgsCache() {
        return chatOrgsCache;
    }

    public void badResponse(
        final String uri,
        final EGenericResponseStatus status,
        final TErrorInfo errorInfo)
        throws HttpException, UnsupportedEncodingException, IOException
    {
        int code;
        switch (status) {
            case Success:
                code = HttpStatus.SC_OK;
                break;
            case InternalError:
                code = HttpStatus.SC_INTERNAL_SERVER_ERROR;
                break;
            case AccessDenied:
                code = HttpStatus.SC_UNAUTHORIZED;
                break;
            case Misconfiguration:
                code = HttpStatus.SC_BAD_REQUEST;
                break;
            case EntityNotFound:
                code = HttpStatus.SC_NOT_FOUND;
                break;
            default:
                code = HttpStatus.SC_INTERNAL_SERVER_ERROR;
        }
        String reason =
            YandexReasonPhraseCatalog.INSTANCE.getReason(code, Locale.US);
        HttpResponse response = new BasicHttpResponse(
            HttpVersion.HTTP_1_1,
            code,
            reason);
        response.setEntity(
            new StringEntity(
                "Got bad response from messenger api: "
                    + "status=" + status + ", errorInfo: "
                    + errorInfo.getMessage()));
        throw new BadResponseException(uri, response);
    }

    public void badResponse(
        final String uri,
        final String message)
        throws HttpException, UnsupportedEncodingException, IOException
    {
        String reason =
            YandexReasonPhraseCatalog.INSTANCE.getReason(
                HttpStatus.SC_INTERNAL_SERVER_ERROR,
                Locale.US);
        HttpResponse response = new BasicHttpResponse(
            HttpVersion.HTTP_1_1,
            HttpStatus.SC_INTERNAL_SERVER_ERROR,
            reason);
        response.setEntity(new StringEntity(message));
        throw new BadResponseException(uri, response);
    }

    public void skipImmediately(final String message)
        throws HttpException, UnsupportedEncodingException, IOException
    {
        throw new ServerException(HttpStatus.SC_ACCEPTED, message);
    }

    @Override
    public void handle(final ProxySession session) throws HttpException {
        HttpRequest request = session.request();
        if (!(request instanceof HttpEntityEnclosingRequest)) {
            throw new BadRequestException("Payload expected");
        }
        final String topic = session.params().getString("topic");
        final MessengerIndexHandlerBase<?, ?> handler = handlers.get(topic);
        if (handler == null) {
            throw new BadRequestException("No handlers exists for topic: "
                + topic);
        }
        final String contentTypeHeader =
            session.headers().getString(HTTP.CONTENT_TYPE, null);
        ContentType contentType = null;
        if (contentTypeHeader != null) {
            try {
                contentType = ContentType.parse(contentTypeHeader);
            } catch (org.apache.http.ParseException
                | UnsupportedCharsetException e)
            {
                throw new BadRequestException("Bad content-type header");
            }
        }

        final boolean multiPart;
        if (contentType != null
            && contentType.getMimeType().equals("multipart/mixed"))
        {
            multiPart = true;
        } else {
            multiPart = false;
        }
        try {
            if (multiPart) {
                handler.handleMultipart(session, parseMultipart(session));
            } else {
                handler.handle(session);
            }
        } catch (ServerException e) {
            throw e;
        } catch (Exception e) {
            throw new BadRequestException("Can't handle request", e);
        }
    }

    private List<PostRequestPart> parseMultipart(
        final ProxySession session)
        throws HttpException
    {
        List<PostRequestPart> parts = new ArrayList<>();
        MimeTokenStream stream = new MimeTokenStream(
            DefaultMimeConfig.INSTANCE,
            null,
            new Utf8FieldBuilder(),
            new OverwritingBodyDescriptorBuilder());
        byte[] tempBuffer = new byte[TEMP_BUFFER_SIZE];
        try {
            HttpEntity entity =
                ((HttpEntityEnclosingRequest) session.request()).getEntity();
            String uri = null;
            stream.parseHeadless(
                entity.getContent(),
                entity.getContentType().getValue());
            EntityState state = stream.getState();
            while (state != EntityState.T_END_OF_STREAM) {
                switch (state) {
                    case T_FIELD:
                        Field field = stream.getField();
                        if (YandexHeaders.URI
                            .equals(field.getName()))
                        {
                            uri = field.getBody();
                        }
                        break;
                    case T_BODY:
                        int pos = 0;
                        try (InputStream is = stream.getDecodedInputStream())
                                //stream.getBodyDescriptor().getCharset()))
                        {
                            int read =
                                is.read(
                                    tempBuffer,
                                    pos,
                                    tempBuffer.length - pos);
                            while (read != -1) {
                                pos += read;
                                if (pos == tempBuffer.length) {
                                    tempBuffer = Arrays.copyOf(
                                        tempBuffer,
                                        tempBuffer.length << 1);
                                }
                                read = is.read(
                                    tempBuffer,
                                    pos,
                                    tempBuffer.length - pos);
                            }
                        }
                        byte[] body = Arrays.copyOf(tempBuffer, pos);
                        if (uri == null) {
                            throw new BadRequestException("Empty URI header for"
                                + " <" + parts.size() + "> multipart section");
                        }
                        UriParser uriParser = new UriParser(uri);
                        CgiParams cgiParams =
                            new CgiParams(uriParser.queryParser());
                        parts.add(
                            new PostRequestPart(
                                session,
                                uriParser,
                                cgiParams,
                                body));
                        uri = null;
                        break;
                    default:
                        break;
                }
                state = stream.next();
            }
        } catch (Throwable t) {
            throw new NotImplementedException(t);
        }
        return parts;
    }

    public static boolean urlIsBanned(final String url) {
        if (url == null) {
            return false;
        }
        String normalized = url.toLowerCase(Locale.ENGLISH);
        for (String banned: BANNED_RCA_URLS) {
            if (normalized.contains(banned)) {
                return true;
            }
        }
        return false;
    }
}

