package ru.yandex.search.messenger.indexer;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;

import NMessengerProtocol.Message.EGenericResponseStatus;
import NMessengerProtocol.Message.TChatCounters;
import NMessengerProtocol.Message.TChatHistoryResponse;
import NMessengerProtocol.Message.THistoryRequest;
import NMessengerProtocol.Message.THistoryRequest.TChatDataFilter;
import NMessengerProtocol.Message.THistoryResponse;
import com.google.protobuf.CodedOutputStream;
import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;

import ru.yandex.cityhash.CityHashingArrayOutputStream;
import ru.yandex.collection.IntPair;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.HttpStatusPredicates;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.NByteArrayEntityAsyncConsumerFactory;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.StatusCodeAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.server.UpstreamStater;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.Utf8JsonWriter;
import ru.yandex.ps.search.messenger.ChatFields;
import ru.yandex.search.json.fieldfunction.SumMapFunction;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.util.string.StringUtils;

public class MessengerChatInfoUpdateHandler
    extends MessengerIndexHandlerBase<
        UpdateChatInfoIndexSession,
        IndexableMessage>
{
    public static final Collection<String> PRESERVE_FIELDS = preserveFields();

    private static final String CHANNEL_PREFIX = "1/";
    private static final String COMMENTATOR_PREFIX = "0/3";
    private static final String COMMENTATOR_PREFIX2 = "0/14";
    private static final String CHAT_LAST_MESSAGE_TIMESTAMP =
        "chat_last_message_timestamp";
    private static final String CHAT_MESSAGE_COUNT =
        "chat_message_count";
    private static final String CHAT_TOTAL_MESSAGE_COUNT =
        "chat_total_message_count";
    private static final String CHAT_HIDDEN_MESSAGE_COUNT =
        "chat_hidden_message_count";
    private static final String CHAT_TOTAL_INDEXED_MESSAGE_COUNT =
        "chat_total_indexed_message_count";
    private static final String CHAT_MODERATION_INFO_MAP =
        "chat_moderation_info_map";
    private static final String CHAT_TEASER_DATA =
        "chat_teaser_data";
    private static final String CHAT_FIRST_MEMBER_TIMESTAMP =
        "chat_first_member_timestamp";
    private static final String CHAT_SECOND_MEMBER_TIMESTAMP =
        "chat_second_member_timestamp";
    private static final String MAX = "max";
    private static final String GET = "get";
    private static final String CHAT_ID = "chat_id";
    private static final String HISTORY_URI = "/history";
    private static final ThreadLocal<CityHashingArrayOutputStream> OUT_TLS =
        ThreadLocal.<CityHashingArrayOutputStream>withInitial(
            () -> new CityHashingArrayOutputStream());

    private final HttpHost moxy;
    private final HttpHost router;
    private final String chatsService;
    private final String messagesService;

    public MessengerChatInfoUpdateHandler(
        final Malo malo,
        final UpstreamStater producerStater)
    {
        super(malo, malo.config().chatsService(), producerStater);
        chatsService = malo.config().chatsService();
        messagesService = malo.config().messagesService();
        moxy = malo.config().moxy().host();
        router = malo.config().messages().host();
    }

    private static Collection<String> preserveFields() {
        LinkedHashSet<String> fields = new LinkedHashSet<>();
        fields.add(CHAT_ID);
        fields.add(CHAT_LAST_MESSAGE_TIMESTAMP);
        fields.add(CHAT_MESSAGE_COUNT);
        fields.add(CHAT_TOTAL_MESSAGE_COUNT);
        fields.add(CHAT_HIDDEN_MESSAGE_COUNT);
        fields.add(CHAT_TOTAL_INDEXED_MESSAGE_COUNT);
        fields.add(CHAT_MODERATION_INFO_MAP);
        fields.add(CHAT_TEASER_DATA);
        fields.add(CHAT_FIRST_MEMBER_TIMESTAMP);
        fields.add(CHAT_SECOND_MEMBER_TIMESTAMP);
        fields.add(ChatFields.NAMESPACE.stored());
        fields.add(MessengerChatMembersDiffHandler.CHAT_MEMBERS);
        fields.add(MessengerChatMembersDiffHandler.CHAT_MEMBERS_VERSION);
        fields.add(MessengerChatMembersDiffHandler.CHAT_MEMBER_COUNT);
        return Collections.unmodifiableSet(fields);
    }

    @Override
    public UpstreamStater upstreamStater(final long metricsTimeFrame) {
        return null;
    }

    @Override
    public UpdateChatInfoIndexSession indexSession(final MaloRequest request)
        throws HttpException, IOException
    {
        return new UpdateChatInfoIndexSession(request);
    }

    @Override
    public UpdateChatInfoIndexSession postIndexSession(
        final PostRequestPart postRequest)
        throws HttpException, IOException
    {
        return new UpdateChatInfoIndexSession(postRequest);
    }

    @Override
    public void handle(
        final UpdateChatInfoIndexSession session,
        final FutureCallback<IndexableMessage> callback)
    {
        chatOrgs(session.session(), session.chatId(), new ResolveOrgsCallback(callback, session));
    }

    private class ResolveOrgsCallback implements FutureCallback<long[]> {
        private final FutureCallback<IndexableMessage> callback;
        private final UpdateChatInfoIndexSession session;

        public ResolveOrgsCallback(
            final FutureCallback<IndexableMessage> callback,
            final UpdateChatInfoIndexSession session)
        {
            this.callback = callback;
            this.session = session;
        }

        @Override
        public void completed(final long[] orgs) {
            if (isChannelOrCommentator(session.chatId()) && (orgs == null || orgs.length == 0)) {
                requestAndCalcSuitability(session, callback);
            } else {
                submitUpdate(session, orgs, callback);
            }
        }

        @Override
        public void failed(final Exception e) {
            callback.failed(e);
        }

        @Override
        public void cancelled() {
            callback.cancelled();
        }
    }

    private boolean isCommentator(final String chatId) {
        return (chatId.startsWith(COMMENTATOR_PREFIX)
            || chatId.startsWith(COMMENTATOR_PREFIX2));
    }

    private boolean isChannelOrCommentator(final String chatId) {
        return (chatId.startsWith(CHANNEL_PREFIX) || isCommentator(chatId));
    }

    @SuppressWarnings("FutureReturnValueIgnored")
    private void requestAndCalcSuitability(
        final UpdateChatInfoIndexSession indexSession,
        final FutureCallback<IndexableMessage> callback)
    {
        final DoubleFutureCallback<Void, Map.Entry<Void, Void>> doubleCallback =
            new DoubleFutureCallback<>(
                new SubmitResponseCallback(indexSession, callback));

        final ProxySession session = indexSession.session();
        final String chatId = indexSession.chatId();

        session.logger().info(
            "Getting moderation counts for chat : " + chatId);

        AsyncClient client = malo.moxyClient().adjust(
            session.context());

        final BasicAsyncRequestProducerGenerator get =
            new BasicAsyncRequestProducerGenerator(
                "/sequential/printkeys?mordo_prigodnost&service="
                    + messagesService
                + "&prefix=" + chatId
                + "&user=" + chatId
                + "&field=message_moderation_action_p"
                + "&print-freqs"
                + "&skip-deleted"
                + "&max-freq=0"
                + "&json-type=dollar");
        client.execute(
            moxy,
            get,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.listener().createContextGeneratorFor(client),
            new MoxyPrintkeysResponseCallback(
                indexSession,
                doubleCallback.first()));
        try {
            sendHistoryOrCmntRequest(indexSession, doubleCallback.second());
        } catch (IOException e) {
            session.logger().log(Level.SEVERE, "sendHistoryRequest failed", e);
            doubleCallback.second().completed(null);
        }
    }

    private void sendHistoryOrCmntRequest(
        final UpdateChatInfoIndexSession indexSession,
        final FutureCallback<Map.Entry<Void, Void>> callback)
        throws IOException
    {
        final DoubleFutureCallback<Void, Void> doubleCallback =
            new DoubleFutureCallback<>(callback);
        final ProxySession session = indexSession.session();
        AsyncClient client = malo.messagesClient().adjust(session.context());

        THistoryRequest historyRequest =
            THistoryRequest.newBuilder()
                .setChatId(indexSession.chatId())
                .setLimit(0)
                .setDropPersonalFields(true)
                .setChatDataFilter(
                    TChatDataFilter.newBuilder().setMinVersion(0).build())
                .build();

        CityHashingArrayOutputStream out = OUT_TLS.get();
        out.reset();
        CodedOutputStream googleOut = CodedOutputStream.newInstance(out);

        historyRequest.writeTo(googleOut);
        googleOut.flush();

        byte[] postData =
            out.toByteArrayWithVersion(MessengerMessageHandler.VERSION);
//        Base64Encoder encoder = new Base64Encoder();
//        encoder.process(postData);
//        session.logger().info("Coded message: " + encoder.toString());
//        session.logger().info("ych: " + out.yandexCityHash());
//        session.logger().info("ch: " + out.cityHash());

        final BasicAsyncRequestProducerGenerator post =
            new BasicAsyncRequestProducerGenerator(
                HISTORY_URI,
                postData,
                ContentType.DEFAULT_BINARY);
        client.execute(
            router,
            post,
            new StatusCheckAsyncResponseConsumerFactory<IntPair<HttpEntity>>(
                HttpStatusPredicates.NON_PROTO_FATAL,
                new StatusCodeAsyncConsumerFactory<HttpEntity>(
                    NByteArrayEntityAsyncConsumerFactory.INSTANCE)),
            session.listener().createContextGeneratorFor(client),
            new HistoryResponseCallback(indexSession, doubleCallback.first()));
        if (isCommentator(indexSession.chatId())) {
            sendCmntRequest(indexSession, doubleCallback.second());
        } else {
            doubleCallback.second().completed(null);
        }
    }

    private void sendCmntRequest(
        final UpdateChatInfoIndexSession indexSession,
        final FutureCallback<Void> callback)
        throws IOException
    {
        resolveEntityId(indexSession, callback);
    }

    private void sendCmntTopPostsRequest(
        final UpdateChatInfoIndexSession indexSession,
        final FutureCallback<Void> callback)
        throws IOException
    {
        AsyncClient client = malo.cmntApiClient().adjust(
            indexSession.session().context());

        String uri =
            malo.config().cmntApi().uri().getRawPath() + '?';
        if (malo.config().cmntApi().uri().getRawQuery() != null) {
            uri += malo.config().cmntApi().uri().getRawQuery();
        }
        uri += "&entityId="
            + indexSession.entityId()
            + "&namespace=" + indexSession.namespace()
            + "&subservice=" + indexSession.subservice();

        final BasicAsyncRequestProducerGenerator get =
            new BasicAsyncRequestProducerGenerator(uri);
        String tvmTicket = malo.cmntApiTvm2Ticket();
        if (tvmTicket != null) {
            get.addHeader(
                YandexHeaders.X_YA_SERVICE_TICKET,
                tvmTicket);
        }
        client.execute(
            malo.config().cmntApi().host(),
            get,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            indexSession.session().listener()
                .createContextGeneratorFor(client),
            new CmntApiCommentsResponseCallback(indexSession, callback));
    }

    private void resolveEntityId(
        final UpdateChatInfoIndexSession indexSession,
        final FutureCallback<Void> callback)
        throws IOException
    {
        final ProxySession session = indexSession.session();
        final String chatId = indexSession.chatId();

        session.logger().info(
            "Getting entityId,service for chat : " + chatId);

        AsyncClient client = malo.moxyClient().adjust(
            session.context());

        final BasicAsyncRequestProducerGenerator get =
            new BasicAsyncRequestProducerGenerator(
                "/sequential/search-malo-cmnt?cmnt_top_posts&service="
                    + chatsService
                + "&prefix=0"
                + "&text=chat_id:(" + chatId + ")"
                + "&get=chat_entity_id,chat_subservice,chat_namespace"
                + "&length=1"
                + "&json-type=dollar");
        client.execute(
            moxy,
            get,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.listener().createContextGeneratorFor(client),
            new MoxyChatResponseCallback(
                indexSession,
                callback));
    }

    private void sendMetaApiRequest(
        final UpdateChatInfoIndexSession indexSession,
        final FutureCallback<Void> callback)
        throws IOException
    {
        AsyncClient client = malo.chatsClient().adjust(
            indexSession.session().context());

        final byte[] postData;
        final ByteArrayOutputStream baos = new ByteArrayOutputStream();
        baos.reset();
        try (
            Utf8JsonWriter writer = JsonType.NORMAL.create(baos))
        {
            baos.write(MessengerChatsHandler.REQUEST_RAVNO);
            writer.startObject();
            writer.key("method");
            writer.value("get_chat");
            writer.key("params");
            writer.startObject();
            writer.key(CHAT_ID);
            writer.value(indexSession.chatId());
            writer.endObject();
            writer.endObject();
            writer.flush();
            postData = baos.toByteArray();
        }

        final BasicAsyncRequestProducerGenerator post =
            new BasicAsyncRequestProducerGenerator(
                malo.config().chats().uri().getPath(),
                postData,
                ContentType.APPLICATION_FORM_URLENCODED);
        client.execute(
            malo.config().chats().host(),
            post,
            new StatusCheckAsyncResponseConsumerFactory<JsonObject>(
                HttpStatusPredicates.NON_PROTO_FATAL,
                JsonAsyncTypesafeDomConsumerFactory.INSTANCE),
                indexSession.session().listener()
                    .createContextGeneratorFor(client),
                new MetaApiChatsResponseCallback(indexSession, callback));
    }

    private void submitUpdate(
        final UpdateChatInfoIndexSession session,
        final long[] orgs,
        final FutureCallback<IndexableMessage> callback)
    {
        IndexableMessage message;
        if (orgs != null && orgs.length > 0) {
            message = new OrgChatUpdateMessage(session, malo.config().v2OrgChatsService(), orgs);
        } else {
            message = new UpdateChatMessage(session);
        }

        callback.completed(message);
    }

    private static class UpdateChatMessage extends ChatMessage {
        private final long messageTimestamp;
        private final long seqNo;
        private final String[] chatMembers;
        private final UpdateChatInfoIndexSession session;
        private final boolean isPvpChat;

        UpdateChatMessage(final UpdateChatInfoIndexSession session) {
            super(session.chatId(), true);
            this.session = session;
            this.messageTimestamp = session.messageTimestamp();
            this.seqNo = session.messageSeqNo();
            if (chatId.indexOf('/') == -1) {
                //1-1 chat, extract members
                chatMembers = chatId.split("_");
                isPvpChat = true;
            } else {
                chatMembers = null;
                isPvpChat = false;
            }
        }

        @Override
        protected void writeDocumentFields(final Utf8JsonWriter writer)
            throws IOException
        {
            //update only if stored timestamp less than new
            /*
            "chat_last_message_timestamp":{
                "max":[
                    {"get":["chat_last_message_timestamp"]},
                    NEW_TIMESTAMP
                ]
            }
            */
            writeFieldMaxValue(
                writer,
                CHAT_LAST_MESSAGE_TIMESTAMP,
                messageTimestamp);

            //update chat member timestamp in pvp chats
            String fromGuid = session.fromGuid();
            if (isPvpChat && fromGuid != null && !fromGuid.isEmpty()) {
                String field;
                if (chatId.startsWith(fromGuid)) {
                    field = CHAT_FIRST_MEMBER_TIMESTAMP;
                } else {
                    field = CHAT_SECOND_MEMBER_TIMESTAMP;
                }
                writeFieldMaxValue(writer, field, messageTimestamp);
            }

            if (seqNo == -1) {
                writer.key(CHAT_MESSAGE_COUNT);
                writer.startObject();
                writer.key("inc");
                writer.startArray();
                writer.value(1);
                writer.endArray();
                writer.endObject();
            } else {
                //use seqNo as message count, update only if stored count is
                // less than new one
                writeFieldMaxValue(writer, CHAT_MESSAGE_COUNT, seqNo);
            }

            if (chatMembers != null) {
                writer.key("chat_members");
                writer.value(StringUtils.join(chatMembers, '\n'));
            }

            if (session.totalMessageCount() != null) {
                writer.key("chat_total_message_count");
                writer.value(session.totalMessageCount());
            }
            if (session.hiddenMessageCount() != null) {
                writer.key("chat_hidden_message_count");
                writer.value(session.hiddenMessageCount());
            }

            Map<String, Long> moderationInfo = session.moderationInfo();
            if (moderationInfo != null) {
                writer.key("chat_moderation_info_map");
                writer.value(SumMapFunction.mapToString(moderationInfo));
            }
            if (session.totalCount() != null) {
                writer.key("chat_total_indexed_message_count");
                writer.value(session.totalCount());
            }

            if (session.updateChatEntityData()) {
                writer.key("chat_entity_id");
                writer.value(session.entityId());

                writer.key("chat_subservice");
                writer.value(session.subservice());

                writer.key("chat_parent_url");
                writer.value(session.parentUrl());
            }

            if (session.namespace() != null) {
                writer.key("chat_namespace");
                writer.value(session.namespace());
            } else if (isPvpChat) {
                writer.key("chat_namespace");
                writer.value("0");
            }

            if (session.teaserData() != null) {
                writer.key("chat_teaser_data");
                writer.value(session.teaserData());
            }

            writer.key(CHAT_ID);
            writer.value(chatId);
        }

        private void writeFieldMaxValue(
            final Utf8JsonWriter writer,
            final String field,
            long newValue)
            throws IOException
        {
            writer.key(field);
            writer.startObject();
            writer.key(MAX);
            writer.startArray();
            writer.value(newValue);
            writer.startObject();
            writer.key(GET);
            writer.startArray();
            writer.value(field);
            writer.endArray();
            writer.endObject();
            writer.endArray();
            writer.endObject();
        }

        @Override
        protected void writeGetFields(
            final Utf8JsonWriter writer,
            final Set<String> fields)
            throws IOException
        {
        }

        @Override
        public String uri(final String args) {
            return "/update?chat-id=" + chatId + args;
        }

        @Override
        public String prefixHash() {
            return prefix();
        }

        @Override
        public String prefix() {
            return "1";
        }
    }

    private static class OrgChatSubUpdateMessage extends UpdateChatMessage {
        private final LongPrefix prefix;
        private final String service;

        public OrgChatSubUpdateMessage(final UpdateChatInfoIndexSession session, final String service, final long orgId) {
            super(session);

            this.prefix = new LongPrefix(orgId);
            this.service = service;
        }

        @Override
        public String service() {
            return service;
        }

        @Override
        public String uri(final String args) {
            return "/update?db=v2org&chat-id=" + chatId + args;
        }

        @Override
        public String prefixHash() {
            return Long.toString(prefix.hash());
        }

        @Override
        public String prefix() {
            return Long.toString(prefix.prefix());
        }
    }

    private static class OrgChatUpdateMessage extends ChatMessage {
        private final UpdateChatInfoIndexSession session;
        private final String service;
        private final long[] orgs;
        public OrgChatUpdateMessage(final UpdateChatInfoIndexSession session, final String service, final long[] orgs) {
            super(session.chatId(), true);

            this.orgs = orgs;
            this.session = session;
            this.service = service;
        }

        @Override
        protected void writeDocumentFields(final Utf8JsonWriter writer)
            throws IOException
        {

        }

        @Override
        protected void writeGetFields(
            final Utf8JsonWriter writer,
            final Set<String> fields)
            throws IOException
        {
        }

        @Override
        public Collection<IndexableMessage> subMessages() {
            List<IndexableMessage> messages = new ArrayList<>(orgs.length);
            for (int i = 0; i < orgs.length; i++) {
                messages.add(new OrgChatSubUpdateMessage(session, service, orgs[i]));
            }
            return messages;
        }

        @Override
        public boolean multiMessage() {
            return true;
        }

        @Override
        public String uri(final String args) {
            return "/update?db=v2org&orgs=" + Arrays.toString(orgs) + "&chat-id=" + chatId + args;
        }

        @Override
        public String service() {
            return service;
        }

        @Override
        public String prefixHash() {
            return prefix();
        }

        @Override
        public String prefix() {
            return "1";
        }
    }

    @SuppressWarnings("HidingField")
    private class MoxyPrintkeysResponseCallback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final UpdateChatInfoIndexSession session;
        private final FutureCallback<Void> callback;

        MoxyPrintkeysResponseCallback(
            final UpdateChatInfoIndexSession session,
            final FutureCallback<Void> callback)
        {
            super(session.session());
            this.session = session;
            this.callback = callback;
        }

        @Override
        public void completed(final JsonObject response) {
            try {
                parseResponse(response);
            } catch (Exception e) {
                failed(new ServiceUnavailableException(e));
            }
        }

        private void parseResponse(final JsonObject response)
            throws JsonException, HttpException, IOException
        {
            session.session().logger().info("Moxy response: "
                + JsonType.NORMAL.toString(response));

            JsonMap map = response.asMap();
            for (Map.Entry<String, JsonObject> entry: map.entrySet()) {
                String key = entry.getKey();
                int sep = key.indexOf('#');
                if (sep == -1) {
                    continue;
                }
                String chatId = key.substring(0, sep);
                if (!chatId.equals(session.chatId())) {
                    break;
                }
                String moderationAction = key.substring(sep + 1);
                JsonMap keyInfo = entry.getValue().asMap();
                long freq = keyInfo.getLong("freq");
                session.addModerationInfo(moderationAction, freq);
                session.session().logger().info("ModerationAction: "
                    + moderationAction + '=' + freq);
            }
            callback.completed(null);
        }
    }

    private class MoxyChatResponseCallback
        extends AbstractFilterFutureCallback<JsonObject, Void>
    {
        private final UpdateChatInfoIndexSession session;
        private final FutureCallback<Void> callback;

        MoxyChatResponseCallback(
            final UpdateChatInfoIndexSession session,
            final FutureCallback<Void> callback)
        {
            super(callback);
            this.session = session;
            this.callback = callback;
        }

        @Override
        public void completed(final JsonObject response) {
            try {
                parseResponse(response);
            } catch (Exception e) {
                session.logger().log(
                    Level.SEVERE,
                    "Moxy chats response parse failed",
                    e);
                callback.completed(null);
            }
        }

        private void parseResponse(final JsonObject response)
            throws JsonException, HttpException, IOException
        {
            session.session().logger().info("Moxy chats response: "
                + JsonType.NORMAL.toString(response));

            JsonMap map = response.asMap();
            JsonList hits = map.getList("hitsArray");
            if (hits.size() == 0) {
                session.logger().severe("Chat not found: "
                    + JsonType.NORMAL.toString(response));
                sendMetaApiRequest(session, callback);
                return;
            }
            JsonMap hit = hits.get(0).asMap();
            String entityId = hit.getString("chat_entity_id", null);
            String namespace = hit.getString("chat_namespace", null);
            String subservice = hit.getString("chat_subservice", null);
            if (entityId == null || namespace == null
                || subservice == null)
            {
                session.logger().severe("No entityId, namespace or subservice"
                    + " is stored in index, will request from meta-api: "
                    + JsonType.NORMAL.toString(response));
                sendMetaApiRequest(session, callback);
            } else {
                session.entityId(entityId);
                session.subservice(subservice);
                session.namespace(namespace);
                sendCmntTopPostsRequest(session, callback);
            }
        }
    }

    private class CmntApiCommentsResponseCallback
        extends AbstractFilterFutureCallback<JsonObject, Void>
    {
        private final UpdateChatInfoIndexSession session;
        private final FutureCallback<Void> callback;

        CmntApiCommentsResponseCallback(
            final UpdateChatInfoIndexSession session,
            final FutureCallback<Void> callback)
        {
            super(callback);
            this.session = session;
            this.callback = callback;
        }

        @Override
        public void completed(final JsonObject response) {
            try {
                parseResponse(response);
                callback.completed(null);
            } catch (Exception e) {
                session.logger().log(
                    Level.SEVERE,
                    "CMNT-api comments response parse failed",
                    e);
                callback.completed(null);
            }
        }

        private void parseResponse(final JsonObject response)
            throws JsonException, HttpException, IOException
        {
            String data = JsonType.NORMAL.toString(response);
            session.session().logger().info("Comments response: "
                + data);

            session.teaserData(data);
        }
    }

    private class HistoryResponseCallback
        extends AbstractProxySessionCallback<IntPair<HttpEntity>>
    {
        private final UpdateChatInfoIndexSession session;
        private final FutureCallback<Void> callback;

        HistoryResponseCallback(
            final UpdateChatInfoIndexSession session,
            final FutureCallback<Void> callback)
        {
            super(session.session());
            this.session = session;
            this.callback = callback;
        }

        @Override
        public void completed(final IntPair<HttpEntity> response) {
            HttpEntity entity = response.second();
            int code = response.first();
            try {
                switch (code) {
                    case HttpStatus.SC_OK:
                        parseResponse(entity);
                        break;
                    case HttpStatus.SC_NOT_FOUND:
                        session.logger().info("History for chat with chatId: "
                            + session.chatId() + ','
                            + " not found");
                        callback.completed(null);
                        break;
                    default:
                        malo.badResponse(
                            HISTORY_URI,
                            "Message info getting error: code=" + code);
                }
            } catch (HttpException | IOException | ParseException e) {
                failed(e);
            }
        }

        private void parseResponse(final HttpEntity entity)
            throws IOException, HttpException, ParseException
        {
            InputStream is = entity.getContent();
            long skipped =
                is.skip(CityHashingArrayOutputStream.THEADER_SIZE);
            if (skipped != CityHashingArrayOutputStream.THEADER_SIZE) {
                malo.badResponse(
                    HISTORY_URI,
                    "Header skip failed");
            }
            THistoryResponse response =
                THistoryResponse.parseFrom(is);
            if (response.getStatus() != EGenericResponseStatus.Success) {
                malo.badResponse(
                    HISTORY_URI,
                    response.getStatus(),
                    response.getErrorInfos(0));
            }
            List<TChatHistoryResponse> chatsInfos = response.getChatsList();
            if (chatsInfos.size() == 0) {
                session.logger().info("Chats infos array is empty");
            }
            for (TChatHistoryResponse chatInfo: chatsInfos) {
                if (chatInfo.getChatId().equals(session.chatId())) {
                    session.logger().info("Found chat history");
                    TChatCounters counters = chatInfo.getCounters();
                    long totalMessageCount = counters.getTotalMessageCount();
                    long hiddenMessageCount = counters.getHiddenMessageCount();
                    session.logger().info("Chat counters: totalMessageCount="
                        + totalMessageCount + ", hiddenMessageCount="
                        + hiddenMessageCount);
                    session.totalMessageCount(totalMessageCount);
                    session.hiddenMessageCount(hiddenMessageCount);
                    callback.completed(null);
                    return;
                }
            }
            session.logger().info("No chat history found in response for"
                + " chat id: " + session.chatId());
            callback.completed(null);
        }
    }

    private class MetaApiChatsResponseCallback
        extends AbstractFilterFutureCallback<JsonObject, Void>
    {
        private final UpdateChatInfoIndexSession session;
        private final FutureCallback<Void> callback;

        MetaApiChatsResponseCallback(
            final UpdateChatInfoIndexSession session,
            final FutureCallback<Void> callback)
        {
            super(callback);
            this.session = session;
            this.callback = callback;
        }

        @Override
        public void completed(final JsonObject response) {
            try {
                parseResponseJson(response);
            } catch (ParseException e) {
                session.logger().log(
                    Level.SEVERE,
                    "Parse Error with chat id: " + session.chatId()
                        + ". temporary skipping chat ",
                    e);
                callback.completed(null);
            } catch (IOException | JsonException | HttpException e) {
                failed(e);
            }
        }

        private void parseResponseJson(final JsonObject response)
            throws HttpException, IOException, JsonException, ParseException
        {
            session.logger().info("Chats response: "
                + JsonType.HUMAN_READABLE.toString(response)
                + "/malo:" + malo);
            JsonMap map = response.asMap();
            String status = map.getOrNull("status");
            ChatMessage message = null;
            if ("ok".equals(status)) {
                JsonMap data = map.getMap(MessengerChatsHandler.DATA);
                parseChatInfo(data);
                sendCmntTopPostsRequest(session, callback);
                return;
            } else if ("error".equals(status)) {
                JsonMap data = map.getMap(MessengerChatsHandler.DATA);
                String code = data.getOrNull("code");
                if ("chat_not_found".equals(code)) {
                    session.logger().info("Chat with  id: " + session.chatId()
                        + " did not found");
                } else if ("unhandled".equals(status)) {
                    //TODO: should be fixed with another status code
                    session.logger().severe("Chat with id: " + session.chatId()
                        + " has no active users");
                } else {
                    session.logger().severe(
                        "Unhandled meta_api error code: " + code
                            + ". Expecting: chat_not_found|unhandled");
                }
            } else {
                session.logger().severe(
                    "Unhandled meta_api status: " + status
                        + ". Expecting: ok|error");
            }
            //bad cases
            callback.completed(null);
        }

        private void parseChatInfo(final JsonMap data) throws JsonException {
            String entityId = data.getString("entity_id", null);
            String subservice = data.getString("subservice", null);
            String namespace = data.getString("namespace", "0");
            String parentUrl = data.getString("parent_url", null);
            session.entityId(entityId);
            session.subservice(subservice);
            session.namespace(namespace);
            session.parentUrl(parentUrl);
            session.updateChatEntityData(true);
        }
    }

    private class SubmitResponseCallback
        extends AbstractFilterFutureCallback<
            Map.Entry<Void, Map.Entry<Void, Void>>,
            IndexableMessage>
    {
        private final UpdateChatInfoIndexSession session;
        private final FutureCallback<IndexableMessage> callback;

        SubmitResponseCallback(
            final UpdateChatInfoIndexSession session,
            final FutureCallback<IndexableMessage> callback)
        {
            super(callback);
            this.session = session;
            this.callback = callback;
        }

        @Override
        public void completed(
            final Map.Entry<Void, Map.Entry<Void, Void>> response)
        {
            submitUpdate(session, null, callback);
        }
    }
}
