package ru.yandex.search.messenger.indexer;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.logging.Level;

import NMessengerProtocol.Search.TDocument;
import com.googlecode.protobuf.format.JsonFormat;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.HttpStatusPredicates;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.server.UpstreamStater;
import ru.yandex.http.util.server.UpstreamStaterFutureCallback;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.Utf8JsonWriter;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.ps.search.messenger.ChatFields;
import ru.yandex.search.messenger.MessengerConstansts;
import ru.yandex.search.messenger.indexer.v2org.OrgChatMembersMessage;
import ru.yandex.util.string.StringUtils;

@SuppressWarnings("FutureReturnValueIgnored")
public class MessengerChatMembersDiffHandler
    extends MessengerIndexHandlerBase<ChatMembersDiffSession, IndexableMessage>
{
    public static final String GET = "get";
    public static final String MAX = "max";
    public static final String CHAT_ID = "chat_id";
    public static final String DATA = "data";
    public static final String CHAT_MEMBERS = "chat_members";
    public static final String CHAT_MEMBERS_VERSION = "chat_members_version";
    public static final String CHAT_MEMBER_COUNT = "chat_member_count";
    public static final String USER_CHATS = "user_chats";
    public static final String USER_CHATS_USER_ID = "user_chats_user_id";
    public static final String USER_CHATS_VERSIONS = "user_chats_versions";
    public static final String USER_CHATS_TMP = "user_chats_tmp";
    public static final byte[] REQUEST_RAVNO =
        "request=".getBytes(StandardCharsets.UTF_8);

    private static final String MAX_CHATS_PER_USER =
        Integer.toString(MessengerConstansts.MAX_CHATS_PER_USER);

    private final String uri;
    private final HttpHost moxy;
    private final HttpHost host;
    private UpstreamStater upstreamStater;
    private final String chatsService;

    public MessengerChatMembersDiffHandler(
        final Malo malo,
        final UpstreamStater producerStater)
    {
        super(malo, malo.config().chatsService(), producerStater);
        chatsService = malo.config().chatsService();
        uri = malo.config().chats().uri().getPath();
        host = malo.config().chats().host();
        moxy = malo.config().moxy().host();
    }

    @Override
    public UpstreamStater upstreamStater(final long metricsTimeFrame) {
        upstreamStater = new UpstreamStater(metricsTimeFrame, "meta-api-chats-members");
        return upstreamStater;
    }

    @Override
    public ChatMembersDiffSession indexSession(final MaloRequest request)
        throws HttpException, IOException
    {
        final String chatId = request.params().getString("chat-id");
        final Long orgId = request.params().getLong("org-id", 0L);
        final boolean reindex = request.params().getBoolean("reindex");
        request.session().logger().info("input CGI chat-id: uuid: " + chatId
            + ", reindex: " + reindex);
        return new ChatMembersDiffSession(request, chatId, orgId, reindex);
    }

    @Override
    public ChatMembersDiffSession postIndexSession(final PostRequestPart post)
        throws HttpException, IOException
    {
        TDocument inputMessage = TDocument.parseFrom(post.body());
        final String chatId = inputMessage.getUuid();
        final Long orgId = inputMessage.getOrganizationId();
        //post.session().logger().info("input TDocument: uuid: " + chatId);
        post.session().logger().info("input TDocument: uuid: " + chatId + " doc " + new JsonFormat().printToString(inputMessage));
        return new ChatMembersDiffSession(post, chatId, orgId, false);
    }

    @Override
    public void handle(
        final ChatMembersDiffSession session,
        final FutureCallback<IndexableMessage> callback)
        throws HttpException, IOException
    {
        chatOrgs(session.session(), session.chatId(), new OrgsCallback(callback, session));
    }

    private class OrgsCallback extends AbstractFilterFutureCallback<long[], IndexableMessage> {
        private final ChatMembersDiffSession session;
        private final FutureCallback<IndexableMessage> callback;

        public OrgsCallback(
            final FutureCallback<IndexableMessage> callback,
            final ChatMembersDiffSession session)
        {
            super(callback);
            this.session = session;
            this.callback = callback;
        }

        @Override
        public void completed(final long[] orgs) {
            session.logger().info("Orgs loaded " + Arrays.toString(orgs));
            if (session.reindex()) {
                try {
                    getMembersDiff(0L, 0, orgs, session, callback);
                } catch (IOException ioe) {
                    failed(ioe);
                    return;
                }
            } else {
                getCurrentMembersVersion(session, orgs, callback);
            }
        }
    }

    private void getCurrentMembersVersion(
        final ChatMembersDiffSession indexSession,
        final long[] orgs,
        final FutureCallback<IndexableMessage> callback)
    {
        final ProxySession session = indexSession.session();
        final String chatId = indexSession.chatId();

        session.logger().info(
            "Getting members version for chat : " + chatId);

        AsyncClient client = malo.moxyClient().adjust(
            session.context());

        QueryConstructor qc = new QueryConstructor("/sequential/search-malo-chat-members?chat-members-diff");
        try {
            if (orgs != null && orgs.length > 0) {
                qc.append("prefix", orgs[0]);
                qc.append("db", "v2org");
                qc.append("text","chat_id_p:(" + chatId + ")");
                qc.append("service", malo.config().v2OrgChatsService());
            } else {
                qc.append("prefix", 0);
                qc.append("text","chat_id:(" + chatId + ")");
                qc.append("service", chatsService);
            }
            qc.append("get", CHAT_MEMBERS_VERSION + ',' + CHAT_MEMBER_COUNT);
            qc.append("length", 1);
            qc.append("json-type", "dollar");
        } catch (BadRequestException bre) {
            callback.failed(bre);
            return;
        }

        final BasicAsyncRequestProducerGenerator get =
            new BasicAsyncRequestProducerGenerator(qc.toString());
        client.execute(
            moxy,
            get,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.listener().createContextGeneratorFor(client),
            new MoxyChatResponseCallback(
                indexSession,
                orgs,
                callback));
    }

    private void getMembersDiff(
        final long version,
        final int memberCount,
        final long[] orgs,
        final ChatMembersDiffSession session,
        final FutureCallback<IndexableMessage> callback)
        throws IOException
    {
        session.info(MaloYtField.INDEX_VERSION, Long.toString(version));
        session.info(MaloYtField.INDEX_MEMB_CNT, Long.toString(memberCount));

        AsyncClient client = malo.chatsClient().adjust(
            session.session().context());

        final byte[] postData;
        final ByteArrayOutputStream baos = baosTls.get();
        baos.reset();
        try (
            Utf8JsonWriter writer = JsonType.NORMAL.create(baos))
        {
            baos.write(REQUEST_RAVNO);
            writer.startObject();
            writer.key("method");
            writer.value("get_chat_members_diff");
            writer.key("params");
            writer.startObject();
            writer.key(CHAT_ID);
            writer.value(session.chatId());
            writer.key("version");
            writer.value(version);
            writer.endObject();
            writer.endObject();
            writer.flush();
            postData = baos.toByteArray();
        }
/*
        session.logger().finest(
            "Request: " + new String(postData, "UTF-8")
            + ", TVM: " + malo.metaApiTvm2Ticket());
*/
        final BasicAsyncRequestProducerGenerator post =
            new BasicAsyncRequestProducerGenerator(
                uri,
                postData,
                ContentType.APPLICATION_FORM_URLENCODED);
        client.execute(
            host,
            post,
            new StatusCheckAsyncResponseConsumerFactory<JsonObject>(
                HttpStatusPredicates.NON_PROTO_FATAL,
                JsonAsyncTypesafeDomConsumerFactory.INSTANCE),
            session.session().listener().createContextGeneratorFor(client),
            new UpstreamStaterFutureCallback<>(
                new ChatsMembersResponseCallback(session, memberCount, orgs, callback),
                upstreamStater));
    }

    private class ChatsMembersResponseCallback
        extends AbstractFilterFutureCallback<JsonObject, IndexableMessage>
    {
        private final ChatMembersDiffSession session;
        private final int memberCount;
        private final long[] orgs;

        ChatsMembersResponseCallback(
            final ChatMembersDiffSession session,
            final int memberCount,
            final long[] orgs,
            final FutureCallback<IndexableMessage> callback)
        {
            super(callback);
            this.session = session;
            this.memberCount = memberCount;
            this.orgs = orgs;
        }

        @Override
        public void completed(final JsonObject response) {
            try {
                ChatMembersMessage message = parseResponseJson(response);
//                if (message.addMembers() != null) {
//                    if (message.addMembers().size() < 10) {
//                        session.info(MaloYtField.MEMB_ADD, StringUtils.join(message.addMembers(), ','));
//                    } else {
//                        session.info(MaloYtField.MEMB_ADD, StringUtils.join(message.addMembers().subList(0, 10), ','));
//                    }
//
//                    session.info(MaloYtField.MEMB_ADD_CNT, Integer.toString(message.addMembers().size()));
//                } else {
//                    session.info(MaloYtField.MEMB_ADD_CNT, "0");
//                }
//
//                if (message.removeMembers() != null) {
//                    if (message.removeMembers().size() < 10) {
//                        session.info(MaloYtField.MEMB_RM, StringUtils.join(message.removeMembers(), ','));
//                    } else {
//                        session.info(MaloYtField.MEMB_RM, StringUtils.join(message.removeMembers().subList(0, 10), ','));
//                    }
//
//                    session.info(MaloYtField.MEMB_RM_CNT, Integer.toString(message.removeMembers().size()));
//                } else {
//                    session.info(MaloYtField.MEMB_RM_CNT, "0");
//                }

                callback.completed(message);
            } catch (ParseException e) {
                session.logger().log(
                    Level.SEVERE,
                    "Parse Error with chat id: " + session.chatId()
                        + ". temporary skipping chat ",
                    e);
                callback.completed(null);
            } catch (IOException | JsonException | HttpException e) {
                failed(e);
            }
        }

        private ChatMembersMessage parseResponseJson(final JsonObject response)
            throws HttpException, IOException, JsonException, ParseException
        {
            session.logger().info("Chat members response: "
                + JsonType.HUMAN_READABLE.toString(response)
                + "/malo:" + malo);
            JsonMap map = response.asMap();
            String status = map.getOrNull("status");
            ChatMembersMessage message = null;
            if ("ok".equals(status)) {
                JsonMap data = map.getMap(DATA);
                if (orgs == null || orgs.length == 0) {
                    message = new ChatMembersInfo(
                        session.chatId(),
                        data,
                        session.reindex(),
                        memberCount);
                } else {
                    session.logger().info("Orgv2 members diff, orgs " + Arrays.toString(orgs));
                    message = new OrgChatMembersMessage(
                        session.chatId(),
                        orgs,
                        malo.config().v2OrgChatsService(),
                        data,
                        session.reindex(),
                        memberCount);
                }

                if (message.chatId == null) {
                    malo.badResponse(uri, "Empty chat-id");
                }
            } else if ("error".equals(status)) {
                JsonMap data = map.getMap(DATA);
                String code = data.getOrNull("code");
                malo.badResponse(
                    uri,
                    "Unhandled meta_api error code: " + code
                        + ". Expecting: chat_not_found|unhandled");
            } else {
                malo.badResponse(
                    uri,
                    "Unhandled meta_api status: " + status
                        + ". Expecting: ok|error");
            }
            return message;
        }
    }

    private class MoxyChatResponseCallback
        extends AbstractFilterFutureCallback<JsonObject, IndexableMessage>
    {
        private final ChatMembersDiffSession session;
        private final long[] orgs;
        private final FutureCallback<IndexableMessage> callback;

        MoxyChatResponseCallback(
            final ChatMembersDiffSession session,
            final long[] orgs,
            final FutureCallback<IndexableMessage> callback)
        {
            super(callback);
            this.session = session;
            this.callback = callback;
            this.orgs = orgs;
        }

        @Override
        public void completed(final JsonObject response) {
            try {
                parseResponse(response);
            } catch (Exception e) {
                session.logger().log(
                    Level.SEVERE,
                    "Moxy chats response parse failed",
                    e);
                callback.completed(null);
            }
        }

        private void parseResponse(final JsonObject response)
            throws JsonException, HttpException, IOException
        {
            session.session().logger().info(
                "Moxy chats-memebers-version response: "
                    + JsonType.NORMAL.toString(response));

            long version = 0;
            JsonMap map = response.asMap();
            JsonList hits = map.getList("hitsArray");
            if (hits.size() == 0) {
                session.logger().severe("Chat not found: "
                    + JsonType.NORMAL.toString(response));
                getMembersDiff(version, 0, orgs, session, callback);
                return;
            }
            JsonMap hit = hits.get(0).asMap();
            version = hit.getLong(CHAT_MEMBERS_VERSION, 0L);
            int memberCount = hit.getInt(CHAT_MEMBER_COUNT, 0);
            session.logger().severe("Chat members version from index: "
                + version + ", memberCount: " + memberCount + ", json: "
                + JsonType.NORMAL.toString(response));
            getMembersDiff(version, memberCount, orgs, session, callback);
        }
    }

    private static class ChatMembersInfoMessage extends ChatMembersMessage {
        private final List<String> removeMembers;
        private final List<String> addMembers;
        private final long version;
        private final boolean reindex;
        private final int memberCount;

        ChatMembersInfoMessage(
            final String chatId,
            final List<String> removeMembers,
            final List<String> addMembers,
            final long version,
            final boolean reindex,
            final int memberCount)
        {
            super(chatId);
            this.removeMembers = removeMembers;
            this.addMembers = addMembers;
            this.version = version;
            this.reindex = reindex;
            this.memberCount = memberCount;
        }

        @Override
        protected void writeDocumentFields(final Utf8JsonWriter writer)
            throws IOException
        {
            //update only if stored version less than new
            /*
            "chat_members_version":{
                "max":[
                    {"get":["chat_members_version"]},
                    VERSION
                ]
            }
            */
            JsonObject checkVersion;
            if (reindex) {
                checkVersion = new JsonString("1");
            } else {
                checkVersion =
                    jsonFunction(
                        "gt",
                        Long.toString(version),
                        jsonFunction(
                            "get",
                            "chat_members_version"));
            }

            //members jopa
            //if (stored.membersVersion < membersVersion) {
            //    members.addAll(addedMembers);
            //    members.removeAll(removedMembers);
            //}
            if (!chatId.startsWith("1/")) {
                if (reindex) {
                    writer.key(CHAT_MEMBERS);
                    if (addMembers != null) {
                        writer.value(StringUtils.join(addMembers, '\n'));
                    } else {
                        writer.value("");
                    }
                } else if (addMembers != null || removeMembers != null) {
                    writer.key(CHAT_MEMBERS);
                    JsonObject membersFunc = jsonFunction("get", CHAT_MEMBERS);
                    if (removeMembers != null) {
                        membersFunc =
                            jsonFunction(
                                "remove_set",
                                StringUtils.join(removeMembers, '\n'),
                                membersFunc);
                    }
                    if (addMembers != null) {
                        membersFunc =
                            jsonFunction(
                                "make_set",
                                membersFunc,
                                StringUtils.join(addMembers, '\n'));
                    }

                    jsonFunction(
                        "set",
                        membersFunc,
                        checkVersion)
                        .writeValue(writer);

                }
            } else {
                writer.key(CHAT_MEMBERS);
                writer.value("");
            }

            if (PvpChatPredicate.INSTANCE.test(chatId)) {
                // on pvp chat create, we have only chat members notify; pvp chats always have 0 namespace
                writer.key(ChatFields.NAMESPACE.stored());
                writer.value(0);
            }

            // order of keys in json makes sense, we should check version for members
            // before with set one
            writer.key(CHAT_MEMBERS_VERSION);
            jsonFunction(
                "set",
                Long.toString(version),
                checkVersion)
                .writeValue(writer);

            writer.key(CHAT_MEMBER_COUNT);
            writer.value(memberCount);

            writer.key(CHAT_ID);
            writer.value(chatId);
        }

        @Override
        protected void writeGetFields(
            final Utf8JsonWriter writer,
            final Set<String> fields)
            throws IOException
        {
        }

        @Override
        public String uri(final String args) {
            return "/update?chat-members-info&chat-id=" + chatId + args;
        }

        @Override
        public String prefixHash() {
            return prefix();
        }

        @Override
        public String prefix() {
            return "1";
        }
    }

    private static class PerUserChatsMessage extends ChatMembersMessage {
        private final String chatId;
        private final String userId;
        private boolean remove;
        private long version;

        PerUserChatsMessage(
            final String chatId,
            final String userId,
            final boolean remove,
            final long version)
        {
            super(chatId);
            this.chatId = chatId;
            this.userId = userId;
            this.remove = remove;
            this.version = version;
        }

        @Override
        protected void writeDocumentFields(final Utf8JsonWriter writer)
            throws IOException
        {
            writer.key(USER_CHATS_USER_ID);
            writer.value(userId);

            writer.key(USER_CHATS_TMP);
            jsonFunction(
                "map_get",
                jsonFunction("get", USER_CHATS_VERSIONS),
                chatId)
                .writeValue(writer);

            JsonObject checkVersion =
                jsonFunction(
                    "gt",
                    Long.toString(version),
                    jsonFunction(
                        "get",
                        USER_CHATS_TMP));

            writer.key(USER_CHATS_VERSIONS);
            jsonFunction(
                "set",
                jsonFunction(
                    "map_set",
                    jsonFunction("get", USER_CHATS_VERSIONS),
                    new JsonString(chatId),
                    new JsonString(Long.toString(version)),
                    new JsonString(MAX_CHATS_PER_USER)),
                checkVersion)
                .writeValue(writer);

            writer.key(USER_CHATS);
            JsonObject addOrRemove;
            if (remove) {
                addOrRemove =
                    jsonFunction(
                        "remove_set",
                        chatId,
                        jsonFunction(
                            "get",
                            USER_CHATS));
            } else {
                addOrRemove =
                    jsonFunction(
                        "make_set",
                        jsonFunction(
                            "get",
                            USER_CHATS),
                        chatId,
                        MAX_CHATS_PER_USER);
            }
            jsonFunction(
                "set",
                addOrRemove,
                checkVersion)
                .writeValue(writer);
        }

        @Override
        protected void writeGetFields(
            final Utf8JsonWriter writer,
            final Set<String> fields)
            throws IOException
        {
        }

        @Override
        public String uri(final String args) {
            return "/update?chat-members-per-user&chat-id=" + chatId
                + "&member-id=" + userId
                + "&remove=" + remove + args;
        }

        @Override
        public String id() {
            return "user_chats_" + userId;
        }

        @Override
        public String type() {
            return "user_chats";
        }

        @Override
        public String prefixHash() {
            return prefix();
        }

        @Override
        public String prefix() {
            return "1";
        }
    }

    private static class UserChatsUpdateMessage extends ChatMembersMessage {
        private final String chatId;
        private final String userId;
        private boolean remove;

        UserChatsUpdateMessage(
            final String chatId,
            final String userId,
            final boolean remove)
        {
            super(chatId, false);
            this.chatId = chatId;
            this.userId = userId;
            this.remove = remove;
        }

        @Override
        protected void writeDocumentFields(final Utf8JsonWriter writer)
            throws IOException
        {
            writer.key(USER_CHATS);
            JsonObject addOrRemove;
            if (remove) {
                addOrRemove =
                    jsonFunction(
                        "remove_set",
                        chatId,
                        jsonFunction(
                            "get",
                            USER_CHATS));
            } else {
                addOrRemove =
                    jsonFunction(
                        "make_set",
                        jsonFunction(
                            "get",
                            USER_CHATS),
                        chatId,
                        MAX_CHATS_PER_USER);
            }
            jsonFunction(
                "set",
                addOrRemove)
                .writeValue(writer);
        }

        @Override
        protected void writeGetFields(
            final Utf8JsonWriter writer,
            final Set<String> fields)
            throws IOException
        {
        }

        @Override
        public String uri(final String args) {
            return "/update?chat-members-per-user&user-id=" + userId + args;
        }

        @Override
        public String id() {
            return "user_" + userId + '@' + prefix();
        }

        @Override
        public String type() {
            return "user";
        }

        @Override
        public String prefixHash() {
            return prefix();
        }

        @Override
        public String prefix() {
            return "0";
        }
    }

    private static class ChatMembersInfo extends ChatMembersMessage {
        private final List<String> removeMembers;
        private final List<String> addMembers;
        private final long version;
        private final boolean reindex;
        private final int memberCount;
        private final Collection<IndexableMessage> subMessages;

        ChatMembersInfo(
            final String chatId,
            final JsonMap json,
            final boolean reindex,
            final int prevMemberCount)
            throws JsonException
        {
            super(chatId);
            this.reindex = reindex;
            JsonList removedList = json.getListOrNull("removed");
            int memberCount = prevMemberCount;
            if (removedList != null && removedList.size() > 0) {
                removeMembers = new ArrayList<>(removedList.size());
                for (JsonObject o: removedList) {
                    removeMembers.add(o.asString());
                }
                memberCount -= removeMembers.size();
            } else {
                removeMembers = null;
            }

            JsonList addedList = json.getListOrNull("added");
            if (addedList != null && addedList.size() > 0) {
                addMembers = new ArrayList<>(addedList.size());
                for (JsonObject o: addedList) {
                    addMembers.add(o.asString());
                }
                memberCount += addMembers.size();
            } else {
                addMembers = null;
            }
            this.memberCount = memberCount;
            version = json.getLong("version", 0L);

            subMessages = new ArrayList<>();
            subMessages.add(
                new ChatMembersInfoMessage(
                    chatId,
                    removeMembers,
                    addMembers,
                    version,
                    reindex,
                    memberCount));
            if (addMembers != null) {
                for (String userId: addMembers) {
                    subMessages.add(
                        new PerUserChatsMessage(
                        chatId,
                        userId,
                        false,
                        version));
                    subMessages.add(
                        new UserChatsUpdateMessage(chatId, userId, false));
                }
            }
            if (removeMembers != null) {
                for (String userId: removeMembers) {
                    subMessages.add(
                        new PerUserChatsMessage(
                        chatId,
                        userId,
                        true,
                        version));
                    subMessages.add(
                        new UserChatsUpdateMessage(chatId, userId, true));
                }
            }
        }

        public List<String> removeMembers() {
            return removeMembers;
        }

        public List<String> addMembers() {
            return addMembers;
        }

        @Override
        public boolean multiMessage() {
            return true;
        }

        @Override
        public Collection<IndexableMessage> subMessages() {
            return subMessages;
        }

        @Override
        protected void writeDocumentFields(final Utf8JsonWriter writer)
            throws IOException
        {
            throw new IOException(
                "writeDocumentFields called on root multimessage");
        }

        @Override
        protected void writeGetFields(
            final Utf8JsonWriter writer,
            final Set<String> fields)
            throws IOException
        {
        }

        @Override
        public String uri(final String args) {
            return "/update?batch&chat-members&chat-id=" + chatId + args;
        }
    }
}
