package ru.yandex.search.messenger.indexer;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.text.ParseException;
import java.util.Arrays;
import java.util.logging.Level;

import NMessengerProtocol.Search.TDocument;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;

import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.HttpStatusPredicates;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.server.UpstreamStater;
import ru.yandex.http.util.server.UpstreamStaterFutureCallback;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.Utf8JsonWriter;
import ru.yandex.search.messenger.indexer.v2org.OrgChatInfo;

@SuppressWarnings("FutureReturnValueIgnored")
public class MessengerChatsHandler
    extends MessengerIndexHandlerBase<ChatIndexSession, IndexableMessage>
{
    public static final String CHAT_ID = "chat_id";
    public static final String DATA = "data";
    public static final String CHAT_MEMBERS = "chat_members";
    public static final byte[] REQUEST_RAVNO =
        "request=".getBytes(StandardCharsets.UTF_8);

    private final String uri;
    private final HttpHost host;
    private UpstreamStater upstreamStater;

    public MessengerChatsHandler(
        final Malo malo,
        final UpstreamStater producerStater)
    {
        super(malo, malo.config().chatsService(), producerStater);
        uri = malo.config().chats().uri().getPath();
        host = malo.config().chats().host();
    }

    @Override
    public UpstreamStater upstreamStater(final long metricsTimeFrame) {
        upstreamStater = new UpstreamStater(metricsTimeFrame, "meta-api-chats");
        return upstreamStater;
    }

    @Override
    public ChatIndexSession indexSession(final MaloRequest request)
        throws HttpException, IOException
    {
        final String chatId = request.params().getString("chat-id");
        request.session().logger().info("input CGI chat-id: uuid: " + chatId);
        return new ChatIndexSession(request, chatId);
    }

    @Override
    public ChatIndexSession postIndexSession(final PostRequestPart post)
        throws HttpException, IOException
    {
        TDocument inputMessage = TDocument.parseFrom(post.body());
        final String chatId = inputMessage.getUuid();
        post.session().logger().info("input TDocument: uuid: " + chatId);
        return new ChatIndexSession(post, chatId);
    }

    @Override
    public void handle(
        final ChatIndexSession session,
        final FutureCallback<IndexableMessage> callback)
        throws HttpException, IOException
    {
        AsyncClient client = malo.chatsClient().adjust(
            session.session().context());

        final byte[] postData;
        final ByteArrayOutputStream baos = baosTls.get();
        baos.reset();
        try (
            Utf8JsonWriter writer = JsonType.NORMAL.create(baos))
        {
            baos.write(REQUEST_RAVNO);
            writer.startObject();
            writer.key("method");
            writer.value("get_chat");
            writer.key("params");
            writer.startObject();
            writer.key(CHAT_ID);
            writer.value(session.chatId());
            writer.key("disable_members");
            writer.value(true);
            writer.endObject();
            writer.endObject();
            writer.flush();
            postData = baos.toByteArray();
        }

        final BasicAsyncRequestProducerGenerator post =
            new BasicAsyncRequestProducerGenerator(
                uri,
                postData,
                ContentType.APPLICATION_FORM_URLENCODED);
        client.execute(
            host,
            post,
            new StatusCheckAsyncResponseConsumerFactory<JsonObject>(
                HttpStatusPredicates.NON_PROTO_FATAL,
                JsonAsyncTypesafeDomConsumerFactory.INSTANCE),
            session.session().listener().createContextGeneratorFor(client),
            new UpstreamStaterFutureCallback<>(
                new ChatsResponseCallback(session, callback),
                upstreamStater));
    }

    private class ChatsResponseCallback
        extends AbstractFilterFutureCallback<JsonObject, IndexableMessage>
    {
        private final ChatIndexSession session;

        ChatsResponseCallback(
            final ChatIndexSession session,
            final FutureCallback<IndexableMessage> callback)
        {
            super(callback);
            this.session = session;
        }

        @Override
        public void completed(final JsonObject response) {
            try {
                ChatMessage message = parseResponseJson(response);
                callback.completed(message);
            } catch (ParseException e) {
                session.logger().log(
                    Level.SEVERE,
                    "Parse Error with chat id: " + session.chatId()
                        + ". temporary skipping chat ",
                    e);
                callback.completed(null);
            } catch (IOException | JsonException | HttpException e) {
                failed(e);
            }
        }

        private ChatMessage parseResponseJson(final JsonObject response)
            throws HttpException, IOException, JsonException, ParseException
        {
            session.logger().info("Chats response: "
                + JsonType.HUMAN_READABLE.toString(response)
                + "/malo:" + malo);
            JsonMap map = response.asMap();
            String status = map.getOrNull("status");
            ChatMessage message = null;
            if ("ok".equals(status)) {
                JsonMap data = map.getMap(DATA);
                boolean hasV2Org = data.containsKey("organizations_v2") && data.getList("organizations_v2").size() > 0;
                if (hasV2Org) {
                    OrgChatInfo orgChatInfo = new OrgChatInfo(data, malo.config().v2OrgChatsService());
                    malo.chatOrgsCache().put(session.chatId(), orgChatInfo.orgIds());
                    session.logger().info("Chat has orgs" + Arrays.toString(orgChatInfo.orgIds()));
                    message = orgChatInfo;
                } else {
                    message = new ChatInfo(data);
                    malo.chatOrgsCache().put(session.chatId(), EMPTY_ORGS);
                }

                if (message.chatId == null) {
                    malo.badResponse(uri, "Empty chat-id");
                }
            } else if ("error".equals(status)) {
                JsonMap data = map.getMap(DATA);
                String code = data.getOrNull("code");
                if ("chat_not_found".equals(code)) {
                    session.logger().info("Chat with  id: " + session.chatId()
                        + " did not found. Will send remove request");
                    message = new DeleteChat(session.chatId());
                } else if ("unhandled".equals(status)) {
                    //TODO: should be fixed with another status code
                    session.logger().info("Chat with id: " + session.chatId()
                        + " has no active users. Will send remove request");
                    message = new DeleteChat(session.chatId());
                } else {
                    malo.badResponse(
                        uri,
                        "Unhandled meta_api error code: " + code
                            + ". Expecting: chat_not_found|unhandled");
                }
            } else {
                malo.badResponse(
                    uri,
                    "Unhandled meta_api status: " + status
                        + ". Expecting: ok|error");
            }
            return message;
        }
    }

}
