package ru.yandex.search.messenger.indexer;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.logging.Level;

import NMessengerProtocol.Search.TDocument;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;

import ru.yandex.collection.IntPair;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.FilterFutureCallback;
import ru.yandex.http.util.HttpStatusPredicates;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.server.UpstreamStater;
import ru.yandex.http.util.server.UpstreamStaterFutureCallback;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.DollarJsonWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.json.writer.Utf8JsonWriter;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.searchmap.SearchMap;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.messenger.MessengerDocument;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.StringPrefix;
import ru.yandex.stater.CountAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;

@SuppressWarnings("FutureReturnValueIgnored")
public class MessengerContactsHandler
    extends MessengerIndexHandlerBase<ContactsIndexSession, IndexableMessage>
{
    private static final byte[] REQUEST_RAVNO =
        "request=".getBytes(StandardCharsets.UTF_8);
    private static final String CONTACT_NAME = "contact_name";
    private static final String CONTACT_VERSION = "contact_version";
    private static final String CONTACT_GLOBAL_VERSION = "contact_global_version";
    private static final String HITS_ARRAY = "hitsArray";
    private static final String GUID = "guid";
    private static final String DATA = "data";
    private static final String STATUS = "status";
    private static final String USER_WITH_ID = "User with id: ";
    private static final String USER_UID = "user_uid";
    private static final String VERSION = "version";

    private final String uri;
    private final HttpHost host;
    private final HttpHost moxy;
    private UpstreamStater upstreamStater;
    private TimeFrameQueue<Long> reindexesContacts;

    public MessengerContactsHandler(
        final Malo malo,
        final UpstreamStater producerStater)
    {
        super(malo, malo.config().messagesService(), producerStater);
        uri = malo.config().users().uri().getPath();
        host = malo.config().users().host();
        moxy = malo.config().moxy().host();

        reindexesContacts = new TimeFrameQueue<>(malo.config().metricsTimeFrame());
        malo.registerStater(
            new PassiveStaterAdapter<>(
                reindexesContacts,
                new NamedStatsAggregatorFactory<>(
                    "contacts-reindex_ammm",
                    CountAggregatorFactory.INSTANCE)));
    }

    @Override
    public UpstreamStater upstreamStater(final long metricsTimeFrame) {
        upstreamStater = new UpstreamStater(
            metricsTimeFrame,
            "meta-api-contacts");
        return upstreamStater;
    }

    @Override
    public ContactsIndexSession indexSession(final MaloRequest request)
        throws HttpException, IOException
    {
        final String userId = request.params().getString("user-id");
        request.session().logger().info("input CGI user-id: guid: " + userId);
        return new ContactsIndexSession(request, userId);
    }

    @Override
    public ContactsIndexSession postIndexSession(final PostRequestPart post)
        throws HttpException, IOException
    {
        TDocument inputMessage = TDocument.parseFrom(post.body());
        final String userId = inputMessage.getUuid();
        //boolean reindex = "reindex".equalsIgnoreCase(inputMessage.getSubType());
        post.session().logger().info("input TDocument: uuid: " + userId);
        //return new ContactsIndexSession(post, userId, reindex);
        return new ContactsIndexSession(post, userId, false);
    }

    @Override
    public void handle(
        final ContactsIndexSession indexSession,
        final FutureCallback<IndexableMessage> callback)
        throws HttpException, IOException
    {
        if (indexSession.reindex()) {
            reindexesContacts.accept(1L);
            indexSession.logger().info("Reindexing contacts for " + indexSession.userId());
        }
        final ProxySession session = indexSession.session();
        final String userId = indexSession.userId();
        if (indexSession.contactsVersionParam() == -1L) {
            session.logger().info(
                "Getting global version for user contacts: " + userId);
            QueryConstructor query =
                new QueryConstructor(
                    "/sequential/search?contacts_version&json-type=dollar");
            query.append("service", service);
            query.append("prefix", userId);
            query.append("get", CONTACT_GLOBAL_VERSION);
            query.append("length", 1);
            query.append(
                "text",
                MessengerDocument.primaryKeyField()
                    + ':' + CONTACT_GLOBAL_VERSION + '@' + userId);
            final BasicAsyncRequestProducerGenerator get =
                new BasicAsyncRequestProducerGenerator(query.toString());
            AsyncClient client = malo.moxyClient().adjust(
                session.context());
            client.execute(
                moxy,
                get,
                JsonAsyncTypesafeDomConsumerFactory.OK,
                session.listener().createContextGeneratorFor(client),
                new MoxyContactGlobalVersionResponseCallback(indexSession, callback));
        } else {
            session.logger().info(
                "Skipping global version request, version is set explicitly to: "
                    + indexSession.contactsVersionParam());
            handleIndex(indexSession, callback, indexSession.contactsVersionParam());
        }
    }

    @SuppressWarnings("HidingField")
    private class MoxyContactGlobalVersionResponseCallback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final ContactsIndexSession session;
        private final FutureCallback<IndexableMessage> callback;

        MoxyContactGlobalVersionResponseCallback(
            final ContactsIndexSession session,
            final FutureCallback<IndexableMessage> callback)
        {
            super(session.session());
            this.session = session;
            this.callback = callback;
        }

        @Override
        public void completed(final JsonObject response) {
            try {
                parseResponse(response);
            } catch (Exception e) {
                failed(new ServiceUnavailableException(e));
            }
        }

        private void parseResponse(final JsonObject response)
            throws JsonException, HttpException, IOException
        {
            JsonMap map = response.asMap();
            JsonList hits = map.getList(HITS_ARRAY);
            if (hits.size() == 0) {
                session.logger().severe(
                    "Empty search results for contact global version request");
                sendMaxVersionRequest();
            } else {
                JsonMap doc = hits.get(0).asMap();
                String version = doc.getOrNull(CONTACT_GLOBAL_VERSION);
                if (version == null) {
                    session.logger().info(
                        "Null version in search results");
                    sendMaxVersionRequest();
                } else {
                    long longVersion = Long.parseLong(version);
                    handleIndex(session, callback, longVersion);
                }
            }
        }

        private void sendMaxVersionRequest() {
            final ProxySession proxySession = session.session();
            final String userId = session.userId();
            proxySession.logger().info(
                "Getting last version for user contacts: "
                    + userId);
            AsyncClient client = malo.moxyClient().adjust(
                proxySession.context());
            final BasicAsyncRequestProducerGenerator get =
                new BasicAsyncRequestProducerGenerator(
                    "/sequential/search?contacts_version&service=" + service
                        + "&prefix=" + userId
                        + "&get=contact_version&json-type=dollar&length=1"
                        + "&sort=contact_version"
                        + "&text=type_p:contact");
            client.execute(
                moxy,
                get,
                JsonAsyncTypesafeDomConsumerFactory.OK,
                proxySession.listener().createContextGeneratorFor(client),
                new MoxyContactsResponseCallback(session, callback));
        }
    }

    public void handleIndex(
        final ContactsIndexSession indexSession,
        final FutureCallback<IndexableMessage> callback,
        final long version)
        throws HttpException, IOException
    {
        indexSession.info(MaloYtField.INDEX_VERSION, Long.toString(version));
        final ProxySession session = indexSession.session();
        final String userId = indexSession.userId();

        AsyncClient client = malo.chatsClient().adjust(session.context());

        final byte[] postData;
        final ByteArrayOutputStream baos = baosTls.get();
        baos.reset();
        try (
            Utf8JsonWriter writer = JsonType.NORMAL.create(baos))
        {
            baos.write(REQUEST_RAVNO);
            writer.startObject();
            writer.key("method");
            writer.value("get_contacts");
            writer.key("params");
            writer.startObject();
            writer.key(GUID);
            writer.value(userId);
            writer.key(VERSION);
            writer.value(version);
            writer.endObject();
            writer.endObject();
            writer.flush();
            postData = baos.toByteArray();
        }

        final BasicAsyncRequestProducerGenerator post =
            new BasicAsyncRequestProducerGenerator(
                uri,
                postData,
                ContentType.APPLICATION_FORM_URLENCODED);
        session.logger().info("postData: " + new String(postData, "UTF-8"));
        client.execute(
            host,
            post,
            new StatusCheckAsyncResponseConsumerFactory<JsonObject>(
                HttpStatusPredicates.NON_PROTO_FATAL,
                JsonAsyncTypesafeDomConsumerFactory.INSTANCE),
            session.listener().createContextGeneratorFor(client),
            new UpstreamStaterFutureCallback<>(
                new ContactsResponseCallback(indexSession, callback),
                upstreamStater));
    }

    private class ContactsResponseCallback
        extends AbstractFilterFutureCallback<JsonObject, IndexableMessage>
    {
        private final String userId;
        private final ContactsIndexSession session;

        ContactsResponseCallback(
            final ContactsIndexSession session,
            final FutureCallback<IndexableMessage> callback)
        {
            super(callback);
            this.session = session;
            this.userId = session.userId();
        }

        @Override
        public void completed(final JsonObject response) {
            try {
                ContactMessage message = parseResponseJson(response);
                if (message != null && message.version() != -1L) {
                    session.contactsVersionGlobal(message.version());
                    session.info(MaloYtField.DB_VERSION, Long.toString(message.version()));
                }
                callback.completed(message);
            } catch (ParseException e) {
                session.logger().log(
                    Level.SEVERE,
                    "Parse Error with user id: " + userId
                        + ". temporary skipping contacts ",
                    e);
                callback.completed(null);
            } catch (IOException | JsonException | HttpException e) {
                failed(e);
            }
//            session.response(HttpStatus.SC_BAD_REQUEST);
        }

        private ContactMessage parseResponseJson(final JsonObject response)
            throws HttpException, IOException, JsonException, ParseException
        {
            session.logger().info("Contacts response: "
                + JsonType.HUMAN_READABLE.toString(response)
                + "/malo:" + malo);
            JsonMap map = response.asMap();
            String status = map.getOrNull(STATUS);
            ContactMessage message = null;
            if ("ok".equals(status)) {
                JsonList data = map.getList(DATA);
                message = new ContactsInfo(session, userId, data);
                session.info(MaloYtField.CNTCT_CNT, Integer.toString(data.size()));
            } else if ("error".equals(status)) {
                JsonMap data = map.getMap(DATA);
                String code = data.getOrNull("code");
                if (code != null) {
                    session.info(MaloYtField.ERROR_CODE, code);
                }
                if ("user_does_not_exist".equals(code)
                    || "user_not_found".equals(code))
                {
                    session.logger().info(USER_WITH_ID + userId
                        + " was not found. Will send remove request");
                    message = new DropContacts(new StringPrefix(userId));
                } else if ("unhandled".equals(code)) {
                    //TODO: should be fixed with another status code
                    session.logger().info(USER_WITH_ID + userId
                        + " has no active users. Will send remove request");
                    message = new DropContacts(new StringPrefix(userId));
                } else if ("unprocessable_entity".equals(code)) {
                    //TODO: should be fixed with another status code
                    session.logger().info(USER_WITH_ID + userId
                        + " has invalid id format. Will send remove request.");
                    message = new DropContacts(new StringPrefix(userId));
                } else {
                    malo.badResponse(
                        uri,
                        "Unhandled meta_api error code: " + code
                            + ". Expecting: user_does_not_exist|unhandled");
                }
            } else {
                malo.badResponse(
                    uri,
                    "Unhandled meta_api status: " + status
                        + ". Expecting: ok|error");
            }
            return message;
        }
    }

    private abstract static class ContactMessage
        extends PrefixedIndexableMessage
    {
        protected final String guid;

        ContactMessage(final Prefix prefix, final String guid) {
            super(prefix, false);
            this.guid = guid;
        }

        @Override
        public String id() {
            return MessengerDocument.contactDocumentId(prefix.toStringFast(), guid);
        }

        @Override
        public String type() {
            return "contact";
        }

        public abstract long version();
    }

    private static class DeleteContact extends ContactMessage {
        DeleteContact(final Prefix prefix, final String guid) {
            super(prefix, guid);
        }

        DeleteContact(final Prefix prefix, final JsonMap map) {
            super(prefix, map.getOrNull(GUID));
        }

        @Override
        protected void writeDocumentFields(final Utf8JsonWriter writer)
            throws IOException
        {
        }

        @Override
        protected void writeGetFields(
            final Utf8JsonWriter writer,
            final Set<String> fields)
            throws IOException
        {
        }

        @Override
        public String uri(final String args) {
            return "/delete?user-id=" + prefix.toString()
                + '&' + "contact-id=" + guid + args;
        }

        @Override
        public long version() {
            return 0L;
        }
    }

    private static class DropContacts extends ContactMessage {
        DropContacts(final Prefix prefix) {
            super(prefix, "");
        }

        @Override
        public boolean getRequest() {
            return true;
        }

        @Override
        protected void writeDocumentFields(final Utf8JsonWriter writer)
            throws IOException
        {
        }

        @Override
        protected void writeGetFields(
            final Utf8JsonWriter writer,
            final Set<String> fields)
            throws IOException
        {
        }

        @Override
        public String uri(final String args) {
            return "/delete?user-id=" + prefix.toString()
                + "&drop-all-contacts"
                + "&prefix=" + prefix.toString()
                + "&text=contact_user_id:" + prefix.toString()
                + args;
        }

        @Override
        public long version() {
            return 0L;
        }
    }

    private static class ContactInfo extends ContactMessage {
        private final long version;
        private final String contactName;

        ContactInfo(final Prefix prefix, final JsonMap map)
            throws JsonException
        {
            super(prefix, map.getOrNull(GUID));
            version = map.getLong(VERSION);
            contactName = map.getOrNull(CONTACT_NAME);
        }

        @Override
        protected void writeDocumentFields(final Utf8JsonWriter writer)
            throws IOException
        {
            writer.key("contact_user_id");
            writer.value(prefix.toString());

            writer.key("contact_id");
            writer.value(guid);

            writer.key(CONTACT_VERSION);
            writer.value(version);

            writer.key(CONTACT_NAME);
            writer.value(contactName);
        }

        @Override
        protected void writeGetFields(
            final Utf8JsonWriter writer,
            final Set<String> fields)
            throws IOException
        {
            throw new IOException("writeGetFields is not implemented");
        }

        @Override
        public String uri(final String args) {
            return "/modify?sub&user-id=" + prefix.toString()
                + "&contact-id=" + guid + args;
        }

        @Override
        public long version() {
            return version;
        }
    }

    private static class BatchDeleteContacts extends PrefixedIndexableMessage {
        public BatchDeleteContacts(final Prefix prefix) {
            super(prefix, false);
        }

        public int size() {
            if (mergedMessages == null) {
                return 0;
            }
            return mergedMessages.size();
        }

        @Override
        protected void writeDocument(final Utf8JsonWriter writer) throws IOException {
        }

        @Override
        protected void writeDocumentFields(final Utf8JsonWriter writer)
            throws IOException
        {
        }

        @Override
        public String id() {
            return null;
        }

        @Override
        public String type() {
            return null;
        }

        @Override
        public boolean writeIdField() {
            return false;
        }

        @Override
        protected void writeGetFields(
            final Utf8JsonWriter writer,
            final Set<String> fields)
            throws IOException
        {
        }

        @Override
        public String uri(final String args) {
            String base = "/delete?contacts&user-id=" + prefix.toString()
                + "&batch-size=" + mergedMessages.size();
            if (mergedMessages.size() > 0) {
                base += "&first=" + mergedMessages.iterator().next().id();
            }

            return base + args;
        }
    }

    private static class BatchModifyContacts extends PrefixedIndexableMessage {
        public BatchModifyContacts(final Prefix prefix) {
            super(prefix, false);
        }

        @Override
        protected void writeDocument(final Utf8JsonWriter writer) throws IOException {
        }

        @Override
        protected void writeDocumentFields(final Utf8JsonWriter writer)
            throws IOException
        {
        }

        @Override
        public String id() {
            return null;
        }

        @Override
        public String type() {
            return null;
        }

        @Override
        public boolean writeIdField() {
            return false;
        }

        @Override
        protected void writeGetFields(
            final Utf8JsonWriter writer,
            final Set<String> fields)
            throws IOException
        {
        }

        public int size() {
            if (mergedMessages == null) {
                return 0;
            }

            return mergedMessages.size();
        }

        @Override
        public String uri(final String args) {
            String base = "/modify?user-id=" + prefix.toString() + "&batch-size=" + mergedMessages.size();
            if (mergedMessages.size() > 0) {
                base += "&first=" + mergedMessages.iterator().next().id();
            }
            return base + args;
        }
    }

    private static class ContactsInfo extends ContactMessage {
        private final Collection<IndexableMessage> subRequests;
        private long version = -1L;

        ContactsInfo(
            final ContactsIndexSession session,
            final String userId,
            final JsonList json)
            throws JsonException
        {
            super(new StringPrefix(userId), null);
            subRequests = new ArrayList<>(json.size());
            if (session.reindex()) {
                ContactMessage dropContacts = new DropContacts(prefix);
                subRequests.add(dropContacts);
                version = dropContacts.version();
            }

            long addedCount = 0L;
            long deletedCount = 0L;
            StringBuilder addedInfo = new StringBuilder();
            StringBuilder deletedInfo = new StringBuilder();
            BatchDeleteContacts deleteBatch = new BatchDeleteContacts(prefix);
            BatchModifyContacts updateBatch = new BatchModifyContacts(prefix);
            List<ContactMessage> removed = new ArrayList<>(json.size());
            for (JsonObject obj: json) {
                JsonMap map = obj.asMap();
                boolean deleted = map.getBoolean("deleted", false);
                ContactMessage contact;
                if (deleted) {
                    contact = new DeleteContact(prefix, map);
                    deletedCount++;
                    if (deletedCount < 10) {
                        deletedInfo.append(contact.guid).append(',');
                    }
                    deleteBatch.addMessage(contact);
                } else {
                    contact = new ContactInfo(prefix, map);
                    addedCount++;
                    if (addedCount < 10) {
                        addedInfo.append(contact.guid).append(',');
                    }
                    updateBatch.addMessage(contact);
                }
                version = Math.max(version, contact.version());
            }
            if (deleteBatch.size() > 0) {
                subRequests.add(deleteBatch);
            }

            if (updateBatch.size() > 0) {
                subRequests.add(updateBatch);
            }
            session.info(MaloYtField.CNTCT_ADD, addedInfo.toString());
            session.info(MaloYtField.CNTCT_ADD_CNT, Long.toString(addedCount));
            session.info(MaloYtField.CNTCT_RM, deletedInfo.toString());
            session.info(MaloYtField.CNTCT_RM_CNT, Long.toString(deletedCount));
        }

        @Override
        public boolean multiMessage() {
            return true;
        }

        @Override
        public Collection<IndexableMessage> subMessages() {
            return subRequests;
        }

        @Override
        protected void writeDocumentFields(final Utf8JsonWriter writer)
            throws IOException
        {
            throw new IOException(
                "writeDocumentFields called on root multimessage");
        }

        @Override
        protected void writeGetFields(
            final Utf8JsonWriter writer,
            final Set<String> fields)
            throws IOException
        {
            for (String field: fields) {
                switch (field) {
                    case USER_UID:
                        writer.key(USER_UID);
                        writer.value(prefix.toString());
                        break;
                    default:
                        break;
                }
            }
        }

        @Override
        public String uri(final String args) {
            return "/modify?user-id=" + prefix.toString()
                + "&contact" + "-id=" + guid + args;
        }

        @Override
        public long version() {
            return version;
        }
    }

    @SuppressWarnings("HidingField")
    private class MoxyContactsResponseCallback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final ContactsIndexSession session;
        private final FutureCallback<IndexableMessage> callback;

        MoxyContactsResponseCallback(
            final ContactsIndexSession session,
            final FutureCallback<IndexableMessage> callback)
        {
            super(session.session());
            this.session = session;
            this.callback = callback;
        }

        @Override
        public void completed(final JsonObject response) {
            try {
                parseResponse(response);
            } catch (Exception e) {
                failed(new ServiceUnavailableException(e));
            }
        }

        private void parseResponse(final JsonObject response)
            throws JsonException, HttpException, IOException
        {
            JsonMap map = response.asMap();
            JsonList hits = map.getList(HITS_ARRAY);
            if (hits.size() == 0) {
                session.logger().severe(
                    "Empty search results for contact last version request");
                handleIndex(session, callback, 0);
            } else {
                JsonMap doc = hits.get(0).asMap();
                String version = doc.getOrNull(CONTACT_VERSION);
                if (version == null) {
                    session.logger().info(
                        "Null version in search results");
                    handleIndex(session, callback, 0);
                } else {
                    long longVersion = Long.parseLong(version);
                    handleIndex(session, callback, longVersion);
                    session.contactsVersionGlobal(longVersion);
                }
            }
        }
    }

    @Override
    protected FutureCallback<IntPair<String>> indexResponseCallback(
        final ContactsIndexSession session,
        final IndexableMessage message)
    {
        return new SaveContactsGlobalVersionCallback(
            session,
            super.indexResponseCallback(session, message));
    }

    @Override
    protected FutureCallback<IntPair<String>> multiIndexResponseCallback(
        final List<ContactsIndexSession> sessions,
        final List<IndexableMessage> messages)
    {
        return new MultiSaveContactsGlobalVersionCallback(
            sessions,
            super.multiIndexResponseCallback(sessions, messages));
    }

    private class SaveContactsGlobalVersionCallback
        extends FilterFutureCallback<IntPair<String>>
    {
        private final ContactsIndexSession session;

        SaveContactsGlobalVersionCallback(
            final ContactsIndexSession session,
            final FutureCallback<IntPair<String>> callback)
        {
            super(callback);
            this.session = session;
        }

        @Override
        public void completed(final IntPair<String> response) {
            if (session.contactsVersionGlobal() == -1L) {
                super.completed(response);
            } else {
                SaveCallback saveCallback = new SaveCallback(
                    session.session().logger(),
                    callback,
                    response);
                sendRequest(session, saveCallback);
            }
        }

        protected String queryBody(final ContactsIndexSession session) {
            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = new DollarJsonWriter(sbw)) {
                writer.startObject();
                writer.key("prefix");
                writer.value(session.userId());
                writer.key("AddIfNotExists");
                writer.value(true);
                writer.key("docs");
                writer.startArray();
                writer.startObject();
                writer.key(MessengerDocument.primaryKeyField());
                writer.value(CONTACT_GLOBAL_VERSION + '@' + session.userId());
                writer.key(CONTACT_GLOBAL_VERSION);
                writer.value(session.contactsVersionGlobal());
                writer.endObject();
                writer.endArray();
                writer.endObject();
            } catch (IOException e) {
                session.logger().log(
                    Level.SEVERE,
                    "Contacts global version save request query error",
                    e);
                return "";
            }
            return sbw.toString();
        }

        protected void sendRequest(
            final ContactsIndexSession session,
            final FutureCallback<Object> callback)
        {
            String body = queryBody(session);
            if (body == null || body.isEmpty()) {
                callback.completed(null);
                return;
            }
            QueryConstructor query = new QueryConstructor("/update?");
            BasicAsyncRequestProducerGenerator producerGenerator =
                new BasicAsyncRequestProducerGenerator(
                    query.toString(),
                    body,
                    ContentType.APPLICATION_JSON);

            producerGenerator.addHeader(YandexHeaders.SERVICE, service);
            producerGenerator.addHeader(
                YandexHeaders.ZOO_SHARD_ID,
                String.valueOf(
                    new StringPrefix(session.userId()).hash()
                        % SearchMap.SHARDS_COUNT));

            AsyncClient client =
                malo.producerClient().adjust(session.session().context());
            client.execute(
                producerHost,
                producerGenerator,
                EmptyAsyncConsumerFactory.OK,
                session.session().listener().createContextGeneratorFor(client),
                callback);
        }
    }

    private class MultiSaveContactsGlobalVersionCallback
        extends SaveContactsGlobalVersionCallback
    {
        private final List<ContactsIndexSession> sessions;

        MultiSaveContactsGlobalVersionCallback(
            final List<ContactsIndexSession> sessions,
            final FutureCallback<IntPair<String>> callback)
        {
            super(sessions.get(0), callback);
            this.sessions = sessions;
        }

        @Override
        public void completed(final IntPair<String> response) {
            MultiFutureCallback<Object> multiCallback =
                new MultiFutureCallback<>(
                    new MultiSaveCallback(
                        sessions.get(0).session().logger(),
                        callback,
                        response));
            for (ContactsIndexSession session: sessions) {
                if (session.contactsVersionGlobal() != -1L) {
                    sendRequest(session, multiCallback.newCallback());
                }
            }
            multiCallback.done();
        }
    }

    private static class SaveCallback implements FutureCallback<Object> {
        private final PrefixedLogger logger;
        private final FutureCallback<? super IntPair<String>> callback;
        private final IntPair<String> response;

        public SaveCallback(
            final PrefixedLogger logger,
            final FutureCallback<? super IntPair<String>> callback,
            final IntPair<String> response)
        {
            this.logger = logger;
            this.callback = callback;
            this.response = response;
        }

        @Override
        public void completed(final Object result) {
            callback.completed(response);
        }

        @Override
        public void failed(Exception e) {
            logger.log(
                Level.SEVERE,
                "Contacts global version save request error",
                e);
            completed(null);
        }

        @Override
        public void cancelled() {
            logger.warning("Contacts global version save request cancelled");
            completed(null);
        }
    }

    private static class MultiSaveCallback
        implements FutureCallback<List<Object>>
    {
        private final PrefixedLogger logger;
        private final FutureCallback<? super IntPair<String>> callback;
        private final IntPair<String> response;

        public MultiSaveCallback(
            final PrefixedLogger logger,
            final FutureCallback<? super IntPair<String>> callback,
            final IntPair<String> response)
        {
            this.logger = logger;
            this.callback = callback;
            this.response = response;
        }

        @Override
        public void completed(final List<Object> result) {
            callback.completed(response);
        }

        @Override
        public void failed(Exception e) {
            logger.log(
                Level.SEVERE,
                "Contacts global version multi save request error",
                e);
            completed(null);
        }

        @Override
        public void cancelled() {
            logger.warning("Contacts global version multi save request cancelled");
            completed(null);
        }
    }
}
