package ru.yandex.search.messenger.indexer;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.logging.Level;
import java.util.stream.Collectors;

import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.FormBodyPartBuilder;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.ByteArrayBody;
import org.apache.http.entity.mime.content.StringBody;
import org.apache.http.nio.entity.NByteArrayEntity;

import ru.yandex.collection.IntPair;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.HttpStatusPredicates;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.NotImplementedException;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.StatusCodeAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.server.UpstreamStater;
import ru.yandex.http.util.server.UpstreamStaterFutureCallback;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.Utf8JsonWriter;

@SuppressWarnings("FutureReturnValueIgnored")
public abstract class MessengerIndexHandlerBase<
    T extends IndexSession,
    V extends IndexableMessage>
    implements ProxyRequestHandler
{
    private static final String TOPIC = "&topic=";
    private static final String OFFSET = "&offset=";
    private static final String MIXED = "mixed";
    private static final String MESSAGE_JSON = "message.json";
    private static final String GET_METHOD = "GET";
    protected static final long[] EMPTY_ORGS = new long[0];
    public static final byte[] REQUEST_RAVNO =
        "request=".getBytes(StandardCharsets.UTF_8);

    protected final Malo malo;
    protected final ThreadLocal<ByteArrayOutputStream> baosTls =
        ThreadLocal.<ByteArrayOutputStream>withInitial(
            () -> new ByteArrayOutputStream());
    protected final String service;
    protected final HttpHost producerHost;
    protected final UpstreamStater producerStater;

    public MessengerIndexHandlerBase(
        final Malo malo,
        final String service,
        final UpstreamStater producerStater)
    {
        this.malo = malo;
        this.service = service;
        this.producerStater = producerStater;
        producerHost = malo.config().producer().host();
    }

    public abstract UpstreamStater upstreamStater(final long metricsTimeFrame);

    @Override
    public void handle(final ProxySession session)
        throws HttpException, IOException
    {
        T indexSession;
        if (session.request() instanceof HttpEntityEnclosingRequest) {
            indexSession = postIndexSession(new PostRequestPart(session));
        } else {
            indexSession = indexSession(new MaloRequest(session));
        }
        handle(indexSession, new FilterProxyCallback(indexSession));
    }

    public void chatOrgs(
        final ProxySession session,
        final String chatId,
        final FutureCallback<long[]> callback)
    {
        AsyncClient client = malo.chatsClient().adjust(
            session.context());

        long[] cached = malo.chatOrgsCache().getIfPresent(chatId);
        if (cached != null) {
            callback.completed(cached);
            return;
        }
        final byte[] postData;
        final ByteArrayOutputStream baos = baosTls.get();
        baos.reset();
        try (
            Utf8JsonWriter writer = JsonType.NORMAL.create(baos))
        {
            baos.write(REQUEST_RAVNO);
            writer.startObject();
            writer.key("method");
            writer.value("get_chat");
            writer.key("params");
            writer.startObject();
            writer.key("chat_id");
            writer.value(chatId);
            writer.key("disable_members");
            writer.value(true);
            writer.endObject();
            writer.endObject();
            writer.flush();
            postData = baos.toByteArray();
        } catch (IOException ioe) {
            callback.failed(ioe);
            return;
        }

        final BasicAsyncRequestProducerGenerator post =
            new BasicAsyncRequestProducerGenerator(
                malo.config().chats().uri().getPath(),
                postData,
                ContentType.APPLICATION_FORM_URLENCODED);
        client.execute(
            malo.config().chats().host(),
            post,
            new StatusCheckAsyncResponseConsumerFactory<JsonObject>(
                HttpStatusPredicates.NON_PROTO_FATAL,
                JsonAsyncTypesafeDomConsumerFactory.INSTANCE),
            session.listener().createContextGeneratorFor(client),
            new ChatsResponseCallback(session, chatId, callback));
    }

    private class ChatsResponseCallback
        extends AbstractFilterFutureCallback<JsonObject, long[]>
    {
        private final String chatId;
        private final ProxySession session;

        ChatsResponseCallback(
            final ProxySession session,
            final String chatId,
            final FutureCallback<long[]> callback)
        {
            super(callback);
            this.chatId = chatId;
            this.session = session;
        }

        @Override
        public void completed(final JsonObject response) {
            try {
                parseResponseJson(response);
            } catch (ParseException e) {
                session.logger().log(
                    Level.SEVERE,
                    "Parse Error with chat id: " + chatId
                        + ". temporary skipping chat ",
                    e);
                callback.completed(null);
            } catch (IOException | JsonException | HttpException e) {
                failed(e);
            }
        }

        private void parseResponseJson(final JsonObject response)
            throws HttpException, IOException, JsonException, ParseException
        {
            session.logger().info("Chats response: "
                                      + JsonType.HUMAN_READABLE.toString(response)
                                      + "/malo:" + malo);
            JsonMap map = response.asMap();
            String status = map.getOrNull("status");

            if ("ok".equals(status)) {
                JsonMap json = map.getMap("data");
                JsonList orgsList = json.getListOrNull("organizations_v2");
                if (orgsList == null || orgsList.size() == 0) {
                    malo.chatOrgsCache().put(chatId, EMPTY_ORGS);
                    callback.completed(EMPTY_ORGS);
                } else {
                    long[] orgs = new long[orgsList.size()];
                    for (int i = 0; i < orgsList.size(); i++) {
                        orgs[i] = orgsList.get(i).asMap().getLong("organization_id");
                    }

                    malo.chatOrgsCache().put(chatId, orgs);
                    callback.completed(orgs);
                }
            } else if ("error".equals(status)) {
                JsonMap data = map.getMap("data");
                String code = data.getOrNull("code");
                if ("chat_not_found".equals(code)) {
                    malo.chatOrgsCache().invalidate(chatId);
                    callback.completed(EMPTY_ORGS);
                } else if ("unhandled".equals(status)) {
                    //TODO: should be fixed with another status code
                    malo.chatOrgsCache().invalidate(chatId);
                    callback.completed(EMPTY_ORGS);
                } else {
                    callback.failed(new Exception(
                        "Unhandled meta_api error code: " + code
                            + ". Expecting: chat_not_found|unhandled"));
                }
            } else {
                callback.failed(new Exception(
                    "Unhandled meta_api error code: " + status
                        + ". Expecting: chat_not_found|unhandled"));
            }
        }
    }

    public abstract void handle(
        final T session,
        final FutureCallback<V> indexCallback)
        throws HttpException, IOException;

    public void handleMulti(
        final List<T> sessions,
        final FutureCallback<List<V>> indexCallback)
        throws HttpException, IOException
    {
        MultiFutureCallback<V> multiCallback =
            new MultiFutureCallback<>(
                new MultiFilterProxyCallback(sessions));
        for (T session: sessions) {
            handle(session, multiCallback.newCallback());
        }
        multiCallback.done();
    }

    public void handleMultipart(
        final ProxySession session,
        final List<PostRequestPart> parts)
        throws HttpException, IOException
    {
        List<T> indexSessions = new ArrayList<>(parts.size());
        for (PostRequestPart part: parts) {
            indexSessions.add(postIndexSession(part));
        }
        handleMulti(
            indexSessions,
            new MultiIndexProxyCallback(indexSessions));
    }

    public abstract T indexSession(final MaloRequest request)
        throws HttpException, IOException;

    public T postIndexSession(
        final PostRequestPart postRequest)
        throws HttpException, IOException
    {
        throw new NotImplementedException("Post handling is not implemeted");
    }

    protected void preprocessMessage(
        final T session,
        final V message)
        throws HttpException
    {
        preprocessMessage(
            session,
            message,
            new IndexProxyCallback(session));
    }

    protected void preprocessMessage(
        final T session,
        final V message,
        final FutureCallback<V> nextCallback)
        throws HttpException
    {
        nextCallback.completed(message);
    }

    protected void preprocessMessages(
        final List<T> sessions,
        final List<V> messages)
        throws HttpException
    {
        preprocessMessages(
            sessions,
            messages,
            new MultiIndexProxyCallback(sessions));
    }

    protected void preprocessMessages(
        final List<T> sessions,
        final List<V> messages,
        final FutureCallback<List<V>> nextCallback)
        throws HttpException
    {
        nextCallback.completed(messages);
    }

    protected void indexMessage(
        final T session,
        final V message)
    {
        indexMessage(
            session,
            message,
            indexResponseCallback(session, message));
    }

    protected void indexMessage(
        final T session,
        final V message,
        final FutureCallback<IntPair<String>> finalCallback)
    {
        if (message == null) {
            session.logger().info("Skipping null message with offset:"
                + ' ' + session.offset());
            finalCallback.completed(
                new IntPair<>(HttpStatus.SC_NO_CONTENT, "-1"));
            return;
        }
        try {
            String offset = session.offset();
            String producerName = session.producerName();
            final BasicAsyncRequestProducerGenerator post;
            if (message.multiMessage()) {
                MultipartEntityBuilder builder =
                    MultipartEntityBuilder.create();
                builder.setMimeSubtype(MIXED);
                writeMultiMessage(
                    message,
                    builder,
                    TOPIC + producerName + OFFSET + offset,
                    session.producerPosition());
                post = new BasicAsyncRequestProducerGenerator(
                    message.uri(TOPIC + producerName + OFFSET + offset),
                    builder.build());
                try {
                    session.logger().info("Index Multi Message " + CharsetUtils.toString(builder.build()));
                } catch (IOException | HttpException ioe) {
                    session.logger().log(Level.WARNING, "Failed to log multi message ", ioe);
                }
            } else {
                if (message.getRequest()) {
                    //GET
                    post = new BasicAsyncRequestProducerGenerator(
                        message.uri(TOPIC + producerName + OFFSET + offset));

                } else {
                    byte[] data = writeSingleMessage(message);
                    post = new BasicAsyncRequestProducerGenerator(
                        message.uri(TOPIC + producerName + OFFSET + offset),
                        data,
                        ContentType.APPLICATION_JSON);

                    session.logger().info("Index Single Message data " + new String(data, StandardCharsets.UTF_8));
                }

            }
            String service = this.service;
            if (message.service() != null) {
                service = message.service();
            }
            post.addHeader(YandexHeaders.SERVICE, service);
            if (!message.multiMessage()) {
                post.addHeader(
                    YandexHeaders.ZOO_SHARD_ID,
                    message.prefixHash());
            }
            if (!session.ignorePosition()) {
                post.addHeader(YandexHeaders.PRODUCER_NAME, producerName);
                if (!message.multiMessage()) {
                    post.addHeader(YandexHeaders.PRODUCER_POSITION, session.producerPosition());
                }
            }

            session.logger().info("Index Message "  + post);
            AsyncClient client =
                malo.producerClient().adjust(session.session().context());
            client.execute(
                producerHost,
                post,
                new StatusCheckAsyncResponseConsumerFactory<IntPair<String>>(
                    x -> x < HttpStatus.SC_BAD_REQUEST
                        || x == HttpStatus.SC_CONFLICT,
                    new StatusCodeAsyncConsumerFactory<String>(
                        AsyncStringConsumerFactory.INSTANCE)),
                session.session().listener().createContextGeneratorFor(client),
                new UpstreamStaterFutureCallback<>(
                    finalCallback,
                    producerStater));
        } catch (IOException e) {
            session.session().handleException(
                new ServerException(HttpStatus.SC_INTERNAL_SERVER_ERROR, e));
        }
    }

    protected boolean check(final List<V> messages, final String service) {
        for (V mes: messages) {
            if (mes.service() == null && (service == null || service == this.service)) {
                continue;
            }

            if (mes.service() == null && (service != this.service && service != null)) {
                return false;
            }

            if (!mes.service().equals(service)) {
                return false;
            }
        }

        return true;
    }

    protected void indexMulti(
        final List<T> sessions,
        final List<V> messages)
    {
        String service = null;
        if (messages.size() > 0 && messages.get(0) != null) {
            service = messages.get(0).service();
        }

        if (service == null) {
            service = this.service;
        }

        int index = -1;
        for (int i = 0; i < messages.size(); i++) {
            if (messages.get(i) == null) {
                continue;
            }
            String mesService = messages.get(i).service();
            if (mesService == null) {
                mesService = this.service;
            }
            if (!service.equals(mesService)) {
                index = i;
                break;
            }
        }
        if (index < 0) {
            indexMulti(
                sessions,
                messages,
                service,
                multiIndexResponseCallback(sessions, messages));
        } else {
            if (messages.size() == sessions.size()) {
                List<V> subList = messages.subList(0, index);
                if (!check(subList, service)) {
                    multiIndexResponseCallback(sessions, messages).failed(new Exception("Check failed"));
                    return;
                }
                indexMulti(
                    sessions.subList(0, index),
                    subList,
                    service,
                    new NextCallback(
                        multiIndexResponseCallback(sessions, messages),
                        index,
                        messages,
                        sessions));
            } else {
                multiIndexResponseCallback(sessions, messages).failed(
                    new Exception("Multiservices not supported "
                                      + String.join(",", messages.stream().map(IndexableMessage::service)
                        .collect(Collectors.toSet()))));
            }
        }
    }

    private class NextCallback implements FutureCallback<IntPair<String>> {
        private final int start;
        private final List<V> messages;
        private final List<T> sessions;
        private final FutureCallback<IntPair<String>> callback;
        public NextCallback(
            final FutureCallback<IntPair<String>> callback,
            final int start,
            final List<V> messages,
            final List<T> sessions)
        {
            this.start = start;
            this.messages = messages;
            this.sessions = sessions;
            this.callback = callback;
        }

        @Override
        public void failed(final Exception e) {
            callback.failed(e);
        }

        @Override
        public void cancelled() {
            callback.cancelled();
        }

        @Override
        public void completed(final IntPair<String> result) {
            sessions.get(0).logger().info("Finished index " + start + " code " + result.first());
            int index = start;
            String service = messages.get(start).service();
            if (service == null) {
                service = MessengerIndexHandlerBase.this.service;
            }
            for (; index < messages.size(); index++) {
                if (messages.get(index) == null) {
                    continue;
                }
                String mesService = messages.get(index).service();
                if (mesService == null) {
                    mesService = MessengerIndexHandlerBase.this.service;
                }
                if (!service.equals(mesService)) {
                    break;
                }
            }

            List<V> subList = messages.subList(start, index);
            if (!check(subList, service)) {
                failed(new Exception("Check failed"));
                return;
            }
            FutureCallback<IntPair<String>> next;
            if (index >= messages.size()) {
                next = callback;
            } else {
                next = new NextCallback(callback, index, messages, sessions);
            }
            indexMulti(
                sessions.subList(start, index),
                subList,
                service,
                next);
        }
    }

    protected void indexMulti(
        final List<T> sessions,
        final List<V> messages,
        final String service,
        final FutureCallback<IntPair<String>> finalCallback)
    {
        final T firstSession = sessions.get(0);
        final T lastSession = sessions.get(sessions.size() - 1);
        String producerName = firstSession.producerName();
        StringBuilder commonUri = new StringBuilder("/batch_store?topic=");
        commonUri.append(producerName);
        commonUri.append("&offsets=");
        commonUri.append(firstSession.offset());
        commonUri.append('-');
        commonUri.append(lastSession.offset());
        commonUri.append("&batch-size=");
        commonUri.append(messages.size());

        MultipartEntityBuilder builder = MultipartEntityBuilder.create();
        builder.setMimeSubtype(MIXED);
        Iterator<T> sessionIter = sessions.iterator();
        int notNull = 0;
        for (V message: messages) {
            T session = sessionIter.next();
            if (message == null) {
                session.logger().info("Skipping null message with offset: "
                    + session.offset());
                continue;
            }
            String uriSuffix = TOPIC + producerName + OFFSET + session.offset();
            try {
                if (message.multiMessage()) {
                    writeMultiMessage(
                        message,
                        builder,
                        uriSuffix,
                        session.offset());
                } else {
                    FormBodyPartBuilder partBuilder =
                        FormBodyPartBuilder.create();
                    partBuilder.addField(
                        YandexHeaders.URI,
                        message.uri(uriSuffix));
                    partBuilder.addField(
                        YandexHeaders.PRODUCER_POSITION,
                        session.producerPosition());
                    partBuilder.addField(
                        YandexHeaders.ZOO_SHARD_ID,
                        message.prefixHash());
                    if (!message.getRequest()) {
                        byte[] data = writeSingleMessage(message);
                        partBuilder.setBody(
                            new ByteArrayBody(
                                data,
                                ContentType.APPLICATION_JSON,
                                null));
                        partBuilder.setName(MESSAGE_JSON);
                    } else {
                        partBuilder.addField(
                            YandexHeaders.ZOO_HTTP_METHOD,
                            GET_METHOD);
                        partBuilder.setBody(new StringBody(""));
                        partBuilder.setName(GET_METHOD);
                    }
                    builder.addPart(partBuilder.build());
                }
            } catch (IOException e) {
                firstSession.logger().log(
                    Level.SEVERE,
                    "Message batch prepare failed for message: " + message
                        + " with offset: " + session.offset(),
                    e);
                firstSession.session().handleException(
                    new ServerException(
                        HttpStatus.SC_INTERNAL_SERVER_ERROR,
                        e));
            }
            notNull++;
        }
        if (notNull == 0) {
            finalCallback.completed(
                new IntPair<>(HttpStatus.SC_NO_CONTENT, "-100500"));
            return;
        }
        try {
            try {
                firstSession.logger().info("Index Multi Message " + CharsetUtils.toString(builder.build()));
            } catch (IOException | HttpException ioe) {
                firstSession.logger().log(Level.WARNING, "Failed to log multi message ", ioe);
            }

            final BasicAsyncRequestProducerGenerator post =
                new BasicAsyncRequestProducerGenerator(
                    new String(commonUri),
                    builder.build());
            post.addHeader(YandexHeaders.SERVICE, service);
            post.addHeader(YandexHeaders.PRODUCER_NAME, producerName);
            AsyncClient client =
                malo.producerClient().adjust(firstSession.session().context());
            client.execute(
                producerHost,
                post,
                new StatusCheckAsyncResponseConsumerFactory<IntPair<String>>(
                    x -> x < HttpStatus.SC_BAD_REQUEST
                        || x == HttpStatus.SC_CONFLICT,
                    new StatusCodeAsyncConsumerFactory<String>(
                        AsyncStringConsumerFactory.INSTANCE)),
                firstSession.session().listener()
                    .createContextGeneratorFor(client),
                new UpstreamStaterFutureCallback<>(
                    finalCallback,
                    producerStater));
        } catch (IOException e) {
            //This should never be happen
            firstSession.logger().log(
                Level.SEVERE,
                "Message batch send failed",
                e);
            firstSession.session().handleException(
                new ServerException(HttpStatus.SC_INTERNAL_SERVER_ERROR, e));
        }
    }

    protected byte[] writeSingleMessage(final IndexableMessage message)
        throws IOException
    {
        final ByteArrayOutputStream baos = baosTls.get();
        baos.reset();
        try (
            Utf8JsonWriter writer = JsonType.NORMAL.create(baos))
        {
            message.writeValue(writer);
            writer.flush();
        }
        return baos.toByteArray();
    }

    //CSOFF: ParameterNumber
    private void writeMultiMessage(
        final IndexableMessage message,
        final MultipartEntityBuilder builder,
        final String uriSuffix,
        final String producerPosition)
        throws IOException
    {
        int subPosition = 0;
        for (IndexableMessage subMessage: message.subMessages()) {
            FormBodyPartBuilder form = FormBodyPartBuilder.create();
            form.addField(YandexHeaders.URI, subMessage.uri(uriSuffix));
            String position = producerPosition
                 + '.' + String.format("%06d", ++subPosition);
            form.addField(
                YandexHeaders.PRODUCER_POSITION,
                position);
            form.addField(YandexHeaders.ZOO_SHARD_ID, subMessage.prefixHash());
            if (subMessage.getRequest()) {
                form.addField(
                    YandexHeaders.ZOO_HTTP_METHOD,
                    GET_METHOD);
                form.setBody(new StringBody(""));
                form.setName(GET_METHOD);
            } else {
                byte[] data = writeSingleMessage(subMessage);
                form.setBody(
                    new ByteArrayBody(
                        data,
                        ContentType.APPLICATION_JSON,
                        null));
                form.setName(MESSAGE_JSON);
            }
            builder.addPart(form.build());
        }
    }
    //CSOFF: ParameterNumber

    protected FutureCallback<IntPair<String>> indexResponseCallback(
        final T session,
        final V message)
    {
        return new IndexResponseCallback(session, message);
    }

    protected FutureCallback<IntPair<String>> multiIndexResponseCallback(
        final List<T> sessions,
        final List<V> messages)
    {
        return new MultiIndexResponseCallback(
            sessions,
            messages);
    }

    @SuppressWarnings("HidingField")
    private class IndexResponseCallback
        extends AbstractProxySessionCallback<IntPair<String>>
    {
        private final T session;
        private final IndexableMessage message;

        IndexResponseCallback(
            final T session,
            final IndexableMessage message)
        {
            super(session.session());
            this.session = session;
            this.message = message;
        }

        @Override
        public void completed(final IntPair<String> response) {
            final String id;
            if (message != null) {
                id = message.id();
            } else {
                id = null;
            }
            session.session().logger().info(
                "Message id: "
                + id + " successfuly indexed with response: "
                + response.second());
            int httpCode;
            if (response.first() == HttpStatus.SC_CONFLICT) {
                httpCode = HttpStatus.SC_ACCEPTED;
            } else {
                httpCode = HttpStatus.SC_OK;
            }
            sendResponse(httpCode);
        }

        private void sendResponse(final int code) {
            if (session.get().size() > 0) {
                final ByteArrayOutputStream baos = baosTls.get();
                baos.reset();
                try (
                    Utf8JsonWriter writer = JsonType.NORMAL.create(baos))
                {
                    writer.startObject();
                    writer.key(message.id());
                    writer.startObject();
                    message.writeGetFields(writer, session.get());
                    writer.endObject();
                    writer.endObject();
                    writer.flush();
                } catch (IOException e) {
                    failed(e);
                    return;
                }
                byte[] data = baos.toByteArray();
                session.session().response(code, new NByteArrayEntity(data));
            } else {
                session.session().response(code);
            }
        }
    }

    private class MultiIndexResponseCallback
        extends AbstractProxySessionCallback<IntPair<String>>
    {
        private final List<T> sessions;
        private final List<V> messages;

        MultiIndexResponseCallback(
            final List<T> sessions,
            final List<V> messages)
        {
            super(sessions.get(0).session());
            this.sessions = sessions;
            this.messages = messages;
        }

        @Override
        public void completed(final IntPair<String> response) {
            session.logger().info(
                "Batch (size=" + messages.size()
                    + ") successfuly indexed with response: "
                    + response.second());
            int httpCode;
            if (response.first() == HttpStatus.SC_CONFLICT) {
                httpCode = HttpStatus.SC_ACCEPTED;
            } else {
                httpCode = HttpStatus.SC_OK;
            }
            sendResponse(httpCode);
        }

        private void sendResponse(final int code) {
            if (sessions.get(0).get().size() > 0) {
                final ByteArrayOutputStream baos = baosTls.get();
                baos.reset();
                try (
                    Utf8JsonWriter writer = JsonType.NORMAL.create(baos))
                {
                    writer.startObject();
                    for (V message: messages) {
                        writer.key(message.id());
                        writer.startObject();
                        message.writeGetFields(writer, sessions.get(0).get());
                        writer.endObject();
                    }
                    writer.endObject();
                    writer.flush();
                } catch (IOException e) {
                    failed(e);
                    return;
                }
                byte[] data = baos.toByteArray();
                session.response(code, new NByteArrayEntity(data));
            } else {
                session.response(code);
            }
        }
    }

    @SuppressWarnings("HidingField")
    private class IndexProxyCallback
        extends AbstractProxySessionCallback<V>
    {
        private final T session;

        IndexProxyCallback(final T session) {
            super(session.session());
            this.session = session;
        }

        @Override
        public void completed(final V message) {
            indexMessage(session, message);
        }
    }

    private class MultiIndexProxyCallback
        extends AbstractProxySessionCallback<List<V>>
    {
        private final List<T> indexSessions;

        MultiIndexProxyCallback(
            final List<T> indexSessions)
        {
            super(indexSessions.get(0).session());
            this.indexSessions = indexSessions;
        }

        @Override
        public void completed(final List<V> messages) {
            indexMulti(indexSessions, messages);
        }
    }

    @SuppressWarnings("HidingField")
    private class FilterProxyCallback
        extends AbstractProxySessionCallback<V>
    {
        private final T session;

        FilterProxyCallback(final T session) {
            super(session.session());
            this.session = session;
        }

        @Override
        public void completed(final V message) {
            try {
                preprocessMessage(session, message);
            } catch (HttpException e) {
                failed(e);
            }
        }
    }

    private class MultiFilterProxyCallback
        extends AbstractProxySessionCallback<List<V>>
    {
        private final List<T> indexSessions;

        MultiFilterProxyCallback(
            final List<T> indexSessions)
        {
            super(indexSessions.get(0).session());
            this.indexSessions = indexSessions;
        }

        @Override
        public void completed(final List<V> messages) {
            try {
                preprocessMessages(indexSessions, messages);
            } catch (HttpException e) {
                failed(e);
            }
        }
    }
}
