package ru.yandex.search.messenger.indexer;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.text.ParseException;
import java.util.AbstractMap.SimpleImmutableEntry;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;

import NMessengerProtocol.Message.TOutMessage;
import com.googlecode.concurrentlinkedhashmap.ConcurrentLinkedHashMap;
import com.googlecode.concurrentlinkedhashmap.EntryWeigher;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.FormBodyPartBuilder;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.ByteArrayBody;

import ru.yandex.collection.IntPair;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.NotImplementedException;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.StatusCodeAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.Utf8JsonWriter;
import ru.yandex.parser.searchmap.SearchMap;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.prefix.Prefix;

@SuppressWarnings("FutureReturnValueIgnored")
public class MessengerIndexPerUserHandler implements ProxyRequestHandler {
    private static final long CACHE_CAPACITY = 1024 * 1024 * 64;
    private static final int OBJECT_WEIGHT = 96;
    private static final double MAX_UIDS = 1000000L;
    private static final String HITS_ARRAY = "hitsArray";
    private static final String PARTITION = "&partition=";
    private static final String OFFSET = "&offset=";
    private static final String IGNORE_POSITION = "&ignore-position=true";
    private static final String GET = "&get=";
    private static final int MAX_URI_LENGTH = 2048;

    protected final ThreadLocal<ByteArrayOutputStream> baosTls =
        ThreadLocal.<ByteArrayOutputStream>withInitial(
            () -> new ByteArrayOutputStream());

    private final Malo malo;
    private final String chatsService;
    private final String usersService;
    private final String mailMessagesService;
    private final HttpHost moxy;
    private final HttpHost producerHost;
    private final ConcurrentLinkedHashMap<String, Long> uidsCache;

    public MessengerIndexPerUserHandler(final Malo malo) {
        this.malo = malo;
        chatsService = malo.config().chatsService();
        usersService = malo.config().usersService();
        mailMessagesService = malo.config().mailMessagesService();
        moxy = malo.config().moxy().host();
        producerHost = malo.config().mailProducer().host();
        uidsCache = new ConcurrentLinkedHashMap.Builder<String, Long>()
            .concurrencyLevel(Runtime.getRuntime().availableProcessors())
            .maximumWeightedCapacity(CACHE_CAPACITY)
            .weigher(new Weigher())
            .build();
    }

    @Override
    public void handle(final ProxySession session)
        throws HttpException, IOException
    {
        if (session.request() instanceof HttpEntityEnclosingRequest) {
            handlePost(new MessageIndexSession(new PostRequestPart(session)));
        } else {
            throw new NotImplementedException("GET handling is not implemeted");
        }
    }

    private void handlePost(final MessageIndexSession session)
        throws HttpException, IOException
    {
        TOutMessage tOutMessage = TOutMessage.parseFrom(session.post.body());
        MessengerMessage message;
        try {
            message = MessengerMessage.fromTOutMessage(tOutMessage);
        } catch (ParseException e) {
            throw new BadRequestException(e);
        }
        session.session().logger().info(
            "Getting members for chat: " + message.chatId());
        AsyncClient client = malo.moxyClient().adjust(
            session.session().context());

        final BasicAsyncRequestProducerGenerator get =
            new BasicAsyncRequestProducerGenerator(
                "/sequential/search?uids&service=" + chatsService
                + "&prefix=0&get=chat_members&json-type=dollar&length=1"
                + "&text=chat_id:" + message.chatId());
        client.execute(
            moxy,
            get,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.session().listener().createContextGeneratorFor(client),
            new MoxyChatsResponseCallback(session, message));
    }

    private void resolveUids(
        final MessageIndexSession session,
        final MessengerMessage message,
        final Set<String> guids)
    {
        resolveUids(
            session,
            message,
            new HashMap<>(),
            guids,
            new HashSet<>());
    }

    //CSOFF: ParameterNumber
    private void resolveUids(
        final MessageIndexSession session,
        final MessengerMessage message,
        final Map<String, Long> guidUidMap,
        final Set<String> unresolved,
        final Set<String> failed)
    {
        session.session().logger().info("Resolving uids. Members count: "
             + unresolved.size()
             + ", failed: " + failed.size()
             + ", resolved so far: " + guidUidMap.size());
        Iterator<String> guidsIter = unresolved.iterator();
        while (guidsIter.hasNext()) {
            String guid = guidsIter.next();
            Long uid = uidsCache.get(guid);
            if (uid != null) {
                guidUidMap.put(guid, uid);
                guidsIter.remove();
            }
        }
        session.session().logger().info("Got " + guidUidMap.size()
            + " from cache");
        if (unresolved.size() == 0) {
            indexMessage(session, message, guidUidMap.values());
        } else {
            StringBuilder sb = new StringBuilder(
                "/sequential/search?service=" + usersService
                + "&prefix=0&get=user_id,user_uid&json-type=dollar"
                + "&group=user_id&merge_func=none");

            sb.append("&text=user_id:(");
            String sep = "";
            guidsIter = unresolved.iterator();
            int batchSize = 0;
            while (guidsIter.hasNext()) {
                batchSize++;
                String guid = guidsIter.next();
                sb.append(sep);
                sep = "+";
                sb.append(guid);
                guidsIter.remove();
                failed.add(guid);
                if (sb.length() >= MAX_URI_LENGTH) {
                    break;
                }
            }
            sb.append(")&length=");
            sb.append(batchSize);

            String uri = new String(sb);
            session.session().logger().info("Resolving rest " + batchSize
                + " guids");
            AsyncClient client = malo.moxyClient().adjust(
                session.session().context());

            final BasicAsyncRequestProducerGenerator get =
                new BasicAsyncRequestProducerGenerator(uri);
            client.execute(
                moxy,
                get,
                JsonAsyncTypesafeDomConsumerFactory.OK,
                session.session().listener().createContextGeneratorFor(client),
                new MoxyUsersResponseCallback(
                    session,
                    message,
                    guidUidMap,
                    unresolved,
                    failed));
        }
    }

    //CSOFF: ParameterNumber
    private void resolveRestUids(
        final MessageIndexSession session,
        final MessengerMessage message,
        final Map<String, Long> resolved,
        final Set<String> unresolved)
    {
        session.session().logger().info("Resolving unresolved uids throught "
            + "meta-api. count: " + unresolved.size());
        String commonUri = "/index-user?multi&topic=ondemand_"
            + session.topic() + PARTITION + session.partition()
            + OFFSET + session.offset() + IGNORE_POSITION
            + GET + MessengerUsersHandler.USER_UID;

        MultiFutureCallback<Map.Entry<String, Long>> multiCallback =
            new MultiFutureCallback<>(
                new MultiUsersResolveResponseCallback(
                    session,
                    message,
                    resolved));
        AsyncClient client = malo.maloClient().adjust(
            session.session().context());
        for (String guid: unresolved) {
            final BasicAsyncRequestProducerGenerator get =
                new BasicAsyncRequestProducerGenerator(
                    commonUri + "&user-id=" + guid);

            client.execute(
                malo.config().malo().host(),
                get,
                JsonAsyncTypesafeDomConsumerFactory.OK,
                session.session().listener().createContextGeneratorFor(client),
                new UserResolveFilterCallback(
                    multiCallback.newCallback(),
                    session,
                    guid));
        }
        multiCallback.done();
    }
    //CSON: ParameterNumber

    private void indexMessage(
        final MessageIndexSession session,
        final MessengerMessage message,
        final Collection<Long> uids)
    {
        session.logger().info(
            "Indexing message for uids: " + uids.toString());
        String producerName = session.partition() + '@'
            + session.topic();
        StringBuilder commonUri =
            new StringBuilder("/messenger/message?topic=");
        commonUri.append(producerName);
        commonUri.append(OFFSET);
        commonUri.append(session.offset());
        commonUri.append("&message_id=");
        commonUri.append(message.id());
        commonUri.append("&batch-size=");
        commonUri.append(uids.size());
        commonUri.append("&delete=");
        commonUri.append(message instanceof MessengerMessage.DeleteMessage);
        int commonLen = commonUri.length();

        MultipartEntityBuilder builder = MultipartEntityBuilder.create();
        builder.setMimeSubtype("mixed");
        int uidNum = 0;
        long offset = Long.parseLong(session.offset());
        for (Long uid: uids) {
            if (uid == -1) {
                session.logger().info("Skiping uid == -1 (robot user?)");
                continue;
            }
            commonUri.setLength(commonLen);
            commonUri.append("&uid=");
            commonUri.append(uid);
            String uri = new String(commonUri);

            final ByteArrayOutputStream baos = baosTls.get();
            baos.reset();
            Prefix prefix = new LongPrefix(uid);
            try (
                Utf8JsonWriter writer = JsonType.NORMAL.create(baos))
            {
                writeMessage(writer, message, prefix);
                writer.flush();
            } catch (IOException e) {
                session.logger().log(
                    Level.SEVERE,
                    "Message batch prepare failed for message: " + message
                        + " with offset: " + session.offset()
                        + ", uid=" + uid,
                    e);
                session.session().handleException(
                    new ServerException(
                        HttpStatus.SC_INTERNAL_SERVER_ERROR,
                        e));
            }
            byte[] data = baos.toByteArray();
            double position = offset + ((double) uidNum / MAX_UIDS);
            builder.addPart(
                FormBodyPartBuilder
                    .create()
                    .addField(YandexHeaders.URI, uri)
                    .addField(
                        YandexHeaders.PRODUCER_POSITION,
                        Double.toString(position))
                    .addField(
                        YandexHeaders.ZOO_SHARD_ID,
                        Long.toString(prefix.hash() % SearchMap.SHARDS_COUNT))
                    .setBody(
                        new ByteArrayBody(
                            data,
                            ContentType.APPLICATION_JSON,
                            null))
                    .setName("message.json")
                    .build());
            uidNum++;
        }
        if (uidNum == 0) {
            session.logger().info(
                "No uids situable for indexing. Skipping message");
            session.session().response(HttpStatus.SC_NO_CONTENT);
            return;
        }
        try {
            commonUri.setLength(commonLen);
            final BasicAsyncRequestProducerGenerator post =
                new BasicAsyncRequestProducerGenerator(
                    new String(commonUri),
                    builder.build());
            post.addHeader(YandexHeaders.SERVICE, mailMessagesService);
            post.addHeader(YandexHeaders.PRODUCER_NAME, producerName);
            AsyncClient client =
                malo.mailProducerClient().adjust(session.session().context());
            client.execute(
                producerHost,
                post,
                new StatusCheckAsyncResponseConsumerFactory<IntPair<String>>(
                    x -> x < HttpStatus.SC_BAD_REQUEST
                        || x == HttpStatus.SC_CONFLICT,
                    new StatusCodeAsyncConsumerFactory<String>(
                        AsyncStringConsumerFactory.INSTANCE)),
                session.session().listener()
                    .createContextGeneratorFor(client),
                new IndexResponseCallback(session.session(), uids));
        } catch (IOException e) {
            //This should never be happen
            session.logger().log(
                Level.SEVERE,
                "Message batch send failed",
                e);
            session.session().handleException(
                new ServerException(HttpStatus.SC_INTERNAL_SERVER_ERROR, e));
        }
    }

    private void chatNotFound(
        final MessageIndexSession session,
        final MessengerMessage message)
        throws IOException
    {
        session.session().logger().info(
            "Trying to resolve members with meta-api");
        AsyncClient client = malo.maloClient().adjust(
            session.session().context());

        final BasicAsyncRequestProducerGenerator get =
            new BasicAsyncRequestProducerGenerator(
                "/index-chat?chat-id=" + message.chatId()
                + "&topic=ondemand_" + session.topic()
                + PARTITION + session.partition() + '&'
                + OFFSET + session.offset()
                + IGNORE_POSITION
                + GET + MessengerChatsHandler.CHAT_MEMBERS);

        client.execute(
            malo.config().malo().host(),
            get,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.session().listener().createContextGeneratorFor(client),
            new MaloChatsResponseCallback(session, message));
    }

    private void writeMessage(
        final Utf8JsonWriter writer,
        final MessengerMessage message,
        final Prefix prefix)
        throws IOException
    {
        writer.startObject();
        writer.key("prefix");
        writer.value(prefix.toString());

        writer.key("docs");
        writer.startArray();

        writer.startObject();
        writer.key("url");
        writer.value(message.id());
        message.writeDocumentFields(writer);
        writer.key("type");
        writer.value(message.type());

        writer.key("last");
        writer.value("true");

        writer.endObject();

        writer.endArray();
        writer.endObject();
    }

    @SuppressWarnings("HidingField")
    private class MoxyChatsResponseCallback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final MessageIndexSession session;
        private final MessengerMessage message;

        MoxyChatsResponseCallback(
            final MessageIndexSession session,
            final MessengerMessage message)
        {
            super(session.session());
            this.session = session;
            this.message = message;
        }

        @Override
        public void completed(final JsonObject response) {
            try {
                parseResponse(response);
            } catch (JsonException | IOException e) {
                failed(new ServiceUnavailableException(e));
            }
        }

        @SuppressWarnings("StringSplitter")
        private void parseResponse(final JsonObject response)
            throws JsonException, IOException
        {
            JsonMap map = response.asMap();
            JsonList hits = map.getList(HITS_ARRAY);
            if (hits.size() == 0) {
                session.session().logger().severe(
                    "Empty search results for chat-id: "
                        + message.chatId());
                chatNotFound(session, message);
            } else {
                JsonMap doc = hits.get(0).asMap();
                String members = doc.getOrNull("chat_members");
                if (members == null) {
                    session.session().logger().severe(
                        "Null members list in results for chat-id: "
                            + message.chatId());
                    chatNotFound(session, message);
                } else {
                    String[] membersArray = members.split("\n");
                    Set<String> membersSet = new HashSet<>();
                    for (String member: membersArray) {
                        member = member.trim();
                        if (members.length() > 0) {
                            membersSet.add(member);
                        }
                    }
                    if (message.fromGuid.length() > 0) {
                        membersSet.add(message.fromGuid);
                    }
                    resolveUids(session, message, membersSet);
                }
            }
        }
    }

    @SuppressWarnings("HidingField")
    private class MoxyUsersResponseCallback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final MessageIndexSession session;
        private final MessengerMessage message;
        private final Map<String, Long> uids;
        private final Set<String> unresolved;
        private final Set<String> failed;

        //CSOFF: ParameterNumber
        MoxyUsersResponseCallback(
            final MessageIndexSession session,
            final MessengerMessage message,
            final Map<String, Long> uids,
            final Set<String> unresolved,
            final Set<String> failed)
        {
            super(session.session());
            this.session = session;
            this.message = message;
            this.uids = uids;
            this.unresolved = unresolved;
            this.failed = failed;
        }
        //CSON: ParameterNumber

        @Override
        public void completed(final JsonObject response) {
            try {
                parseResponse(response);
            } catch (JsonException e) {
                failed(new ServiceUnavailableException(e));
            }
        }

        private void parseResponse(final JsonObject response)
            throws JsonException
        {
            JsonMap map = response.asMap();
            JsonList hits = map.getList(HITS_ARRAY);
            if (hits.size() == 0) {
//                failed(
//                    new ServiceUnavailableException(
                session.logger().severe(
                    "Empty search results for guids request");
                if (unresolved.size() > 0) {
                    //try next batch
                    resolveUids(session, message, uids, unresolved, failed);
                } else {
                    resolveRestUids(session, message, uids, failed);
                }
            } else {
                for (int i = 0; i < hits.size(); i++) {
                    JsonMap doc = hits.get(i).asMap();
                    String guid = doc.getOrNull("user_id");
                    if (guid == null) {
//                        failed(
//                            new ServiceUnavailableException(
//                                "Null guid in search results"));
                        session.logger().info(
                            "Null guid in search results");
                        continue;
                    }
                    String uid = doc.getOrNull("user_uid");
                    if (uid == null) {
                        session.logger().info(
                            "Null uid in results for guid: "
                                + guid);
                        continue;
//                        failed(
//                            new ServiceUnavailableException(
//                                "Null uid for guid in results for guid: "
//                                    + guid));
                    }
                    Long parsedUid = Long.parseLong(uid);
                    uidsCache.put(guid, parsedUid);
                    uids.put(guid, parsedUid);
                    failed.remove(guid);
                }
                if (unresolved.size() > 0) {
                    resolveUids(session, message, uids, unresolved, failed);
                } else if (failed.size() > 0) {
                    resolveRestUids(session, message, uids, failed);
                } else {
                    indexMessage(session, message, uids.values());
                }
            }
        }
    }

    private static class MessageIndexSession extends IndexSession {
        private final PostRequestPart post;

        MessageIndexSession(final PostRequestPart post)
            throws HttpException, IOException
        {
            super(post);
            this.post = post;
        }
    }

    private static class IndexResponseCallback
        extends AbstractProxySessionCallback<IntPair<String>>
    {
        private final Collection<Long> uids;

        IndexResponseCallback(
            final ProxySession session,
            final Collection<Long> uids)
        {
            super(session);
            this.uids = uids;
        }

        @Override
        public void completed(final IntPair<String> response) {
            session.logger().info(
                "Batch with messages (size=" + uids.size()
                    + ") successfuly indexed with response: "
                    + response.second());
            if (response.first() == HttpStatus.SC_CONFLICT) {
                session.response(HttpStatus.SC_ACCEPTED);
            } else {
                session.response(HttpStatus.SC_OK);
            }
        }
    }

    @SuppressWarnings("HidingField")
    private class MaloChatsResponseCallback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final MessageIndexSession session;
        private final MessengerMessage message;

        MaloChatsResponseCallback(
            final MessageIndexSession session,
            final MessengerMessage message)
        {
            super(session.session());
            this.session = session;
            this.message = message;
        }

        @Override
        public void completed(final JsonObject response) {
            try {
                JsonMap json = response.asMap();
                JsonMap chatInfo =
                    json.getMapOrNull("chat_" + message.chatId());
                if (chatInfo == null) {
                    failed(
                        new ServiceUnavailableException(
                            "No chat info in malo response for chat-id: "
                                + message.chatId()));
                } else {
                    JsonList members = json.getListOrNull(
                        MessengerChatsHandler.CHAT_MEMBERS);
                    if (members == null || members.size() == 0) {
                        session.logger().info(
                            "Empty members list in malo response for "
                            + "chat-id: " + message.chatId()
                            + ". Skipping messages");
                        session.session().response(HttpStatus.SC_NO_CONTENT);
                        return;
//                        failed(
//                            new ServiceUnavailableException(
//                                "Empty members list in malo response for "
//                                    + "chat-id: " + message.chatId()));
                    } else {
                        HashSet<String> chatMembers = new HashSet<>();
                        for (JsonObject member: members) {
                            chatMembers.add(member.asString());
                        }
                        resolveUids(session, message, chatMembers);
                    }
                }
            } catch (JsonException e) {
                failed(e);
            }
        }
    }

    @SuppressWarnings("HidingField")
    private class MultiUsersResolveResponseCallback
        extends AbstractProxySessionCallback<List<Map.Entry<String, Long>>>
    {
        private final MessageIndexSession session;
        private final MessengerMessage message;
        private final Map<String, Long> uids;

        MultiUsersResolveResponseCallback(
            final MessageIndexSession session,
            final MessengerMessage message,
            final Map<String, Long> uids)
        {
            super(session.session());
            this.session = session;
            this.message = message;
            this.uids = uids;
        }

        @Override
        public void completed(final List<Map.Entry<String, Long>> response) {
            for (Map.Entry<String, Long> entry: response) {
                uids.put(entry.getKey(), entry.getValue());
                uidsCache.put(entry.getKey(), entry.getValue());
            }
            indexMessage(session, message, uids.values());
        }
    }

    private static class UserResolveFilterCallback
        extends AbstractFilterFutureCallback<
            JsonObject,
            Map.Entry<String, Long>>
    {
        private final MessageIndexSession session;
        private final String guid;

        UserResolveFilterCallback(
            final FutureCallback<Map.Entry<String, Long>> nextCallback,
            final MessageIndexSession session,
            final String guid)
        {
            super(nextCallback);
            this.session = session;
            this.guid = guid;
        }

        @Override
        public void completed(final JsonObject response) {
            try {
                JsonMap json = response.asMap();
                JsonMap userInfo = json.getMapOrNull("user_" + guid);
                if (userInfo == null) {
                    throw new ServiceUnavailableException(
                        "No user-info for guid: " + guid);
                }
                long uid = userInfo.getLong(MessengerUsersHandler.USER_UID);
                callback.completed(new SimpleImmutableEntry<>(guid, uid));
            } catch (JsonException | HttpException e) {
                session.logger().log(
                    Level.SEVERE,
                    "Can't resolve uid for guid: " + guid,
                    e);
                failed(e);
            }
        }
    }

    private static class Weigher
        implements EntryWeigher<String, Long>
    {
        @Override
        public int weightOf(
            final String key,
            final Long value)
        {
            return OBJECT_WEIGHT + (key.length() << 1);
        }
    }
}
