package ru.yandex.search.messenger.indexer;

import java.io.IOException;
import java.text.ParseException;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;

import NMessengerProtocol.Client.TClientMessage;
import NMessengerProtocol.Client.TFileInfo;
import NMessengerProtocol.Client.TPlain;
import NMessengerProtocol.Client.TPlain.TGallery;
import NMessengerProtocol.Client.TPlain.TGallery.TItem;
import NMessengerProtocol.Client.TPlain.TImage;
import NMessengerProtocol.Client.TPlain.TMiscFile;
import NMessengerProtocol.Client.TPlain.TPoll;
import NMessengerProtocol.Client.TPlain.TText;
import NMessengerProtocol.Client.TPlain.TVoice;
import NMessengerProtocol.Client.TServerMessage;
import NMessengerProtocol.Client.TServerMessage.TForwardedMessage;
import NMessengerProtocol.Client.TServerMessage.TServerMessageInfo;
import NMessengerProtocol.Client.TUserInfo;
import NMessengerProtocol.Message.TOutMessage;
import com.google.protobuf.CodedOutputStream;

import ru.yandex.io.HexOutputStream;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.writer.Utf8JsonWriter;
import ru.yandex.ps.search.messenger.MessageFields;
import ru.yandex.search.prefix.StringPrefix;
import ru.yandex.util.string.HexStrings;
import ru.yandex.util.string.StringUtils;

public abstract class MessengerMessage extends PrefixedIndexableMessage {
    private static final int TIMESTAMP_TO_MILLIS_DIV = 1000;
    private static final String MESSAGE_GALLERY_IMAGES = "message_gallery_images";
    private static final String MESSAGE_TEXT = "message_text";
    private static final String MESSAGE_VOICE_DURATION = "message_voice_duration";
    private static final String MESSAGE_VOICE_WAS_RECOGNIZED = "message_voice_was_recognized";
    private static final String MESSAGE_VOICE_WAVEFORM = "message_voice_waveform";
    private static final String MESSAGE_POLL_TITLE = "message_poll_title";
    private static final String MESSAGE_POLL_ANSWERS = "message_poll_answers";

    private static final Set<String> SPECIAL_CHATS =
        new HashSet<String>();

    static {
        SPECIAL_CHATS.add("1/2/b86a77a5-d9ed-419d-8051-ff97b6734be3");
        SPECIAL_CHATS.add("1/2/e40a3554-19b3-45f4-ad7b-1e428deddff8");
        SPECIAL_CHATS.add("1/2/ea82f5f9-be4b-4e09-913b-8d04285bef7f");
        SPECIAL_CHATS.add("1/2/33ea21d9-28cd-401a-9a96-f4aa5f163cea");
        SPECIAL_CHATS.add("1/2/33ac52bd-2785-41b6-aea6-45f71c5e6f52");
    }

    protected final TOutMessage message;
    protected final String id;
    protected final String parentId;
    protected final String chatId;
    protected final int hid;
    protected final Boolean important;
    protected final long timestamp;
    protected final long lastEditTimestamp;
    protected final long seqNo;
    protected final String fromDisplayName;
    protected final String fromGuid;
    protected final String fromPhoneId;
    protected final TServerMessage serverMessage;
    protected final TClientMessage clientMessage;
//    protected final TPlain plain;
    protected final String moderationAction;
    protected final String moderationVerdicts;
    protected final boolean forwarded;

    MessengerMessage(
        final String chatId,
        final int hid,
        final TOutMessage message,
        final boolean forwarded)
    {
        super(new StringPrefix(chatId), false);
        this.chatId = chatId;
        this.message = message;
        this.hid = hid;
        this.forwarded = forwarded;
        serverMessage = message.getServerMessage();
        clientMessage = serverMessage.getClientMessage();
        TServerMessageInfo info = serverMessage.getServerMessageInfo();
        moderationAction = info.getModerationAction().toString();
        if (clientMessage.hasPlain()) {
            TPlain plain = clientMessage.getPlain();
            StringBuilder sb = new StringBuilder(plain.getModerationVerdictsCount() * 10);
            for (String verdict: plain.getModerationVerdictsList()) {
                sb.append(verdict);
                sb.append('\n');
            }

            if (sb.length() > 0) {
                sb.setLength(sb.length() - 1);
            }

            this.important = plain.getIsImportant();
            moderationVerdicts = sb.toString();
        } else {
            moderationVerdicts = "";
            this.important = false;
        }

        timestamp = info.getTimestamp();
        lastEditTimestamp = info.getLastEditTimestamp();
        String id = chatId + '/' + Long.toString(timestamp);
        if (hid > 0) {
            parentId = id;
            id += '/' + Integer.toString(hid);
        } else {
            parentId = null;
        }
        this.id = id;

        seqNo = info.getSeqNo();
        TUserInfo from = info.getFrom();
        fromDisplayName = from.getDisplayName();
        fromGuid = from.getGuid();
        fromPhoneId = from.getPhoneId();
    }

    private MessengerMessage(
        final String chatId,
        final int hid,
        final long timestamp)
    {
        super(new StringPrefix(chatId), false);
        this.chatId = chatId;
        this.timestamp = timestamp;
        String id = chatId + '/' + Long.toString(timestamp);
        if (hid > 0) {
            parentId = id;
            id += '/' + Integer.toString(hid);
        } else {
            parentId = null;
        }
        this.id = id;
        this.hid = hid;
        this.lastEditTimestamp = 0;
        this.seqNo = 0;
        this.fromDisplayName = "";
        this.fromGuid = "";
        this.fromPhoneId = "";
        this.important = null;
        this.serverMessage = null;
        this.clientMessage = null;
        this.moderationAction = "UNDEFINED";
        this.moderationVerdicts = "";
        this.forwarded = false;
        this.message = createFakeTOutMessage(
            chatId,
            timestamp);
    }

    public TOutMessage createFakeTOutMessage(
        final String chatId,
        final TOutMessage realMessage)
    {
        TServerMessage serverMessage = message.getServerMessage();
        TServerMessageInfo info = serverMessage.getServerMessageInfo();
        long timestamp = info.getTimestamp();
        return createFakeTOutMessage(chatId, timestamp);
    }

    public TOutMessage createFakeTOutMessage(
        final String chatId,
        final long timestamp)
    {
        return
            TOutMessage.newBuilder()
                .setServerMessage(
                    TServerMessage.newBuilder()
                        .setClientMessage(
                            TClientMessage.newBuilder()
                                .setPlain(
                                    TPlain.newBuilder()
                                        .setChatId(chatId)
                                        .setTimestamp(timestamp)
                                        .build())
                                .build())
                    .setServerMessageInfo(
                        TServerMessageInfo.newBuilder()
                            .setTimestamp(timestamp)
                            .build()))
                .build();
    }

    public long seqNo() {
        return seqNo;
    }

    public String fromGuid() {
        return fromGuid;
    }

    public abstract String subType();

    public abstract void toStringBuilder(final StringBuilder sb);

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(type());
        sb.append('[');
        sb.append("chatId=");
        sb.append(chatId);
        sb.append(", timestamp=");
        sb.append(timestamp);
        sb.append(", lastEditTimestamp=");
        sb.append(lastEditTimestamp);
        sb.append(", seqNo=");
        sb.append(seqNo);
        sb.append(", fromDisplayName=");
        sb.append(fromDisplayName);
        sb.append(", fromGuid=");
        sb.append(fromGuid);
        sb.append(", fromPhoneId=");
        sb.append(fromPhoneId);
        toStringBuilder(sb);
        sb.append(']');
        return new String(sb);
    }

    public static boolean isSpecialChat(final String chatId) {
        return SPECIAL_CHATS.contains(chatId) || chatId.startsWith("1/");
    }

    @Override
    public String id() {
        return "message_" + id;
    }

    @Override
    public String type() {
        return subType() + "_message";
    }

    public String chatId() {
        return chatId;
    }

    public long timestamp() {
        return timestamp;
    }

    public boolean multiItem() {
        return false;
    }

    @Override
    protected void writeDocumentFields(final Utf8JsonWriter writer)
        throws IOException
    {
        writer.key("message_id");
        writer.value(id);

        writer.key("message_parent_id");
        writer.value(parentId);

        writer.key("message_hid");
        writer.value(hid);

        writer.key("message_forwarded");
        writer.value(forwarded);

        writer.key("message_multi_item");
        writer.value(multiItem());

        writer.key("message_chat_id_hash");
        writer.value((short) chatId.hashCode());

        writer.key("message_chat_id");
        writer.value(chatId);

        writer.key("message_moderation_action");
        writer.value(moderationAction);

        writer.key(MessageFields.MODERATION_VERDICTS.stored());
        writer.value(moderationVerdicts);

        if (important != null) {
            writer.key(MessageFields.IMPORTANT.stored());
            if (important) {
                writer.value(1);
            } else {
                writer.value(JsonNull.INSTANCE);
            }
        }

        writer.key("message_timestamp");
        writer.value(timestamp);

        writer.key("message_last_edit_timestamp");
        writer.value(lastEditTimestamp);

        writer.key("message_seq_no");
        writer.value(seqNo);

        writer.key("message_from_display_name");
        writer.value(fromDisplayName);

        writer.key("message_from_guid");
        writer.value(fromGuid);

        writer.key("message_from_phone_id");
        writer.value(fromPhoneId);

        writer.key("message_data");
        writer.startString();

        HexOutputStream hexOut =
            new HexOutputStream(writer, HexStrings.UPPER);
        CodedOutputStream googleOut = CodedOutputStream.newInstance(hexOut);
        message.writeTo(googleOut);
        googleOut.flush();
        writer.endString();
    }

    @Override
    protected void writeGetFields(
        final Utf8JsonWriter writer,
        final Set<String> fields)
        throws IOException
    {
    }

    @Override
    public String uri(final String args) {
        return "/modify?message-id=" + id + args
            + "&prefix=" + prefix() + "&type=" + type();
    }

//    @Override
    public long ctime() {
        return timestamp / TIMESTAMP_TO_MILLIS_DIV;
    }

    public static DeleteMessage deleteMessage(
        final String chatId,
        final long timestamp)
    {
        return new DeleteMessage(chatId, timestamp);
    }

    public static MessengerMessage fromTOutMessage(final TOutMessage message)
        throws ParseException
    {
//        List<String> guids = message.getGuids();
        TServerMessage serverMessage = message.getServerMessage();
        TClientMessage clientMessage = serverMessage.getClientMessage();
        TServerMessageInfo info = serverMessage.getServerMessageInfo();


        long timestamp = info.getTimestamp();
        if (!clientMessage.hasPlain()) {
            throw new InvalidTypeException(
                "TOutMessage::ServerMessage::ClientMessage"
                    + " is not a type " + "of TPlain: "
                    + clientMessage.getBodyCase().toString(),
                clientMessage.getBodyCase().toString()
                    .toLowerCase(Locale.ROOT));
        }
        TPlain plain = clientMessage.getPlain();
        final String chatId = plain.getChatId();
        List<TForwardedMessage> forwardedMessages =
            serverMessage.getForwardedMessagesList();
//        String forwardedText = null;
//        if (forwardedMessages.size() > 0) {
//            fowardedText =
//        }

        MessengerMessage result = null;
        //TODO: Remove this heuristics
        if (plain.getPayloadCase() == TPlain.PayloadCase.PAYLOAD_NOT_SET
            && forwardedMessages.size() == 0)
        {
            result = new DeleteMessage(chatId, timestamp);
        } else if (info.getDeleted()) {
            result = new DeleteMessage(chatId, timestamp);
        } else {
            result = createMessage(chatId, 0, plain, message);
            for (int i = 0; i < forwardedMessages.size(); i++) {
                TForwardedMessage forwarded = forwardedMessages.get(i);
                result.addMessage(
                    createMessage(
                        chatId,
                        i + 1,
                        forwarded.getPayload(),
                        message));
            }
        }
        return result;
    }

    private static MessengerMessage createMessage(
        final String chatId,
        final int hid,
        final TItem item,
        final TOutMessage message)
        throws ParseException
    {
        MessengerMessage result;
        switch (item.getContentCase()) {
            case IMAGE:
                result =
                    new ImageMessage(chatId, hid, item.getImage(), message, false);
                break;
            default:
                throw new ParseException("unhandled TItem::Content type:"
                    + item.getContentCase().toString(), 0);
        }
        return result;
    }

    private static MessengerMessage createMessage(
        final String chatId,
        final int hid,
        final TPlain plain,
        final TOutMessage message)
        throws ParseException
    {
        MessengerMessage result;
        boolean forwarded = !plain.getChatId().isEmpty()
            && !chatId.equalsIgnoreCase(plain.getChatId());
        switch (plain.getPayloadCase()) {
            case TEXT:
                result = new TextMessage(chatId, hid, plain.getText(), message, forwarded);
                break;
            case IMAGE:
                result =
                    new ImageMessage(chatId, hid, plain.getImage(), message, forwarded);
                break;
            case MISCFILE:
                result =
                    new FileMessage(chatId, hid, plain.getMiscFile(), message, forwarded);
                break;
            case STICKER:
            case PAYLOAD_NOT_SET:
            case CARD:
            case NEVERTOBESUPPORTED:
                result = new RawMessage(
                    chatId,
                    hid,
                    message,
                    plain.getPayloadCase().toString()
                        .toLowerCase(Locale.ROOT),
                    forwarded);
                break;
            case GALLERY:
                result =
                    new GalleryMessage(chatId, hid, plain.getGallery(), message, forwarded);
                break;
            case VOICE:
                result =
                    new VoiceMessage(chatId, hid, plain.getVoice(), message, forwarded);
                break;
            case POLL:
                result =
                    new PollMessage(chatId, hid, plain.getPoll(), message, forwarded);
                break;
            default:
                throw new ParseException("unhandled TPlain::Payload type:"
                    + plain.getPayloadCase().toString(), 0);
        }
        return result;
    }

    static class TextMessage extends MessengerMessage {
        private final String text;
        private final Set<String> urls;

        TextMessage(
            final String chatId,
            final int hid,
            final TText tText,
            final TOutMessage message,
            final boolean forwarded)
        {
            super(chatId, hid, message, forwarded);
            text = tText.getMessageText();
            urls = MessengerMessageHandler.parseUrls(text);
        }

        @Override
        protected void writeDocumentFields(final Utf8JsonWriter writer)
            throws IOException
        {
            writer.key(MESSAGE_TEXT);
            writer.value(text);
            writer.key(MessageFields.LINKS.stored());
            if (urls.isEmpty()) {
                writer.nullValue();
            } else {
                writer.value(StringUtils.join(urls, '\n'));
            }

            super.writeDocumentFields(writer);
        }

        public String text() {
            return text;
        }

        @Override
        public String subType() {
            return "text";
        }

        @Override
        public void toStringBuilder(final StringBuilder sb) {
            sb.append(",  text=");
            sb.append(text);
        }
    }

    private static class GalleryMessage extends MessengerMessage {
        private final String text;
        private final String images;

        GalleryMessage(
            final String chatId,
            final int hid,
            final TGallery tGallery,
            final TOutMessage message,
            final boolean forwarded)
            throws ParseException
        {
            super(chatId, hid, message, forwarded);
            text = tGallery.getText();
            List<TItem> items = tGallery.getItemsList();
            TOutMessage fakeMessage = createFakeTOutMessage(
                chatId,
                message);
            int subHid = 1;
            StringBuilder sb = new StringBuilder();
            for (TItem item: items) {
                MessengerMessage subMsg =
                    createMessage(
                        chatId,
                        subHid++,
                        item,
                        fakeMessage);
                if (subMsg instanceof FileMessage) {
                    sb.append(hid);
                    sb.append(':');
                    sb.append(((FileMessage) subMsg).filename);
                    sb.append('\n');
                }
            }
            images = new String(sb);
        }

        @Override
        protected void writeDocumentFields(final Utf8JsonWriter writer)
            throws IOException
        {
            writer.key(MESSAGE_TEXT);
            writer.value(text);

            writer.key(MESSAGE_GALLERY_IMAGES);
            writer.value(images);

            super.writeDocumentFields(writer);
        }

        public boolean multiItem() {
            return true;
        }

        @Override
        public String subType() {
            return "gallery";
        }

        @Override
        public void toStringBuilder(final StringBuilder sb) {
            sb.append(", text=");
            sb.append(text);
        }
    }

    private static class VoiceMessage extends FileMessage {
        private final String text;
        private final int duration;
        private final byte[] waveform;
        private final boolean wasRecognized;

        VoiceMessage(
            final String chatId,
            final int hid,
            final TVoice tVoice,
            final TOutMessage message,
            final boolean forwarded)
        {
            super(chatId, hid, tVoice.getFileInfo(), message, forwarded);
            text = tVoice.getText();
            duration = tVoice.getDuration();
            waveform = tVoice.getWaveform().toByteArray();
            wasRecognized = tVoice.getWasRecognized();
        }

        @Override
        protected void writeDocumentFields(final Utf8JsonWriter writer)
            throws IOException
        {
            writer.key(MESSAGE_TEXT);
            writer.value(text);

            writer.key(MESSAGE_VOICE_DURATION);
            writer.value(duration);

            writer.key(MESSAGE_VOICE_WAS_RECOGNIZED);
            writer.value(wasRecognized);

            writer.key(MESSAGE_VOICE_WAVEFORM);
            writer.startString();
            HexOutputStream hexOut =
                new HexOutputStream(writer, HexStrings.UPPER);
            hexOut.write(waveform);
            hexOut.flush();
            writer.endString();

            super.writeDocumentFields(writer);
        }

        @Override
        public String subType() {
            return "voice";
        }

        @Override
        public void toStringBuilder(final StringBuilder sb) {
            sb.append(", text=");
            sb.append(text);
            sb.append(", recognized=");
            sb.append(wasRecognized);
            sb.append(", duration=");
            sb.append(duration);
            super.toStringBuilder(sb);
        }
    }

    private static class PollMessage extends MessengerMessage {
        private final String title;
        private final List<String> answers;

        PollMessage(
            final String chatId,
            final int hid,
            final TPoll tPoll,
            final TOutMessage message,
            final boolean forwarded)
        {
            super(chatId, hid, message, forwarded);
            title = tPoll.getTitle();
            answers = tPoll.getAnswersList();
        }

        @Override
        protected void writeDocumentFields(final Utf8JsonWriter writer)
            throws IOException
        {
            writer.key(MESSAGE_POLL_TITLE);
            writer.value(title);

            writer.key(MESSAGE_POLL_ANSWERS);
            if (answers.isEmpty()) {
                writer.nullValue();
            } else {
                writer.value(StringUtils.join(answers, '\n'));
            }

            super.writeDocumentFields(writer);
        }

        @Override
        public String subType() {
            return "poll";
        }

        @Override
        public void toStringBuilder(final StringBuilder sb) {
            sb.append(", pollTitle=");
            sb.append(title);
            sb.append(", pollAnswers=");
            sb.append(StringUtils.join(answers, ','));
        }
    }

    private static class RawMessage extends MessengerMessage {
        private final String subType;

        RawMessage(
            final String chatId,
            final int hid,
            final TOutMessage message,
            final String subType,
            final boolean forwarded)
        {
            super(chatId, hid, message, forwarded);
            this.subType = subType;
        }

        @Override
        protected void writeDocumentFields(final Utf8JsonWriter writer)
            throws IOException
        {
            super.writeDocumentFields(writer);
        }

        @Override
        public String subType() {
            return subType;
        }

        @Override
        public void toStringBuilder(final StringBuilder sb) {
            sb.append(", raw=true");
        }
    }

    private static class FileMessage extends MessengerMessage {
        protected final long fileId;
        protected final String filename;
        protected final int fileSize;

        FileMessage(
            final String chatId,
            final int hid,
            final TMiscFile tMiscFile,
            final TOutMessage message,
            final boolean forwarded)
        {
            this(chatId, hid, tMiscFile.getFileInfo(), message, forwarded);
        }

        FileMessage(
            final String chatId,
            final int hid,
            final TFileInfo fileInfo,
            final TOutMessage message,
            final boolean forwarded)
        {
            super(chatId, hid, message, forwarded);
            fileId = fileInfo.getId();
            filename = fileInfo.getName();
            fileSize = fileInfo.getSize();
        }

        @Override
        protected void writeDocumentFields(final Utf8JsonWriter writer)
            throws IOException
        {
            writer.key("message_filename");
            writer.value(filename);

            writer.key("message_file_id");
            writer.value(fileId);

            writer.key("message_file_size");
            writer.value(fileSize);

            super.writeDocumentFields(writer);
        }

        @Override
        public String subType() {
            return "file";
        }

        @Override
        public void toStringBuilder(final StringBuilder sb) {
            sb.append(", filename=");
            sb.append(filename);
            sb.append(", fileId=");
            sb.append(fileId);
            sb.append(", fileSize=");
            sb.append(fileSize);
        }
    }

    private static class ImageMessage extends FileMessage {
        private final int width;
        private final int height;

        ImageMessage(
            final String chatId,
            final int hid,
            final TImage image,
            final TOutMessage message,
            final boolean forwarded)
        {
            super(chatId, hid, image.getFileInfo(), message, forwarded);
            width = image.getWidth();
            height = image.getHeight();
        }
        //CSON: ParameterNumber

        @Override
        protected void writeDocumentFields(final Utf8JsonWriter writer)
            throws IOException
        {
            writer.key("message_image_width");
            writer.value(width);

            writer.key("message_image_height");
            writer.value(height);

            super.writeDocumentFields(writer);
        }

        @Override
        public String subType() {
            return "image";
        }

        @Override
        public void toStringBuilder(final StringBuilder sb) {
            super.toStringBuilder(sb);
            sb.append(", width=");
            sb.append(width);
            sb.append(", height=");
            sb.append(height);
        }
    }

    static class DeleteMessage extends MessengerMessage {
        DeleteMessage(
            final String chatId,
            final long timestamp)
        {
            super(chatId, 0, timestamp);
        }

        @Override
        protected void writeDocumentFields(final Utf8JsonWriter writer)
            throws IOException
        {
            writer.key("message_deleted");
            writer.value(true);
            writer.key(MESSAGE_TEXT);
            writer.value((String) null);
        }

        @Override
        protected void writeGetFields(
            final Utf8JsonWriter writer,
            final Set<String> fields)
            throws IOException
        {
        }

        @Override
        public String updateRequest() {
            return "message_id:" + id
                + " OR message_id:" + id + "/*";
        }

        @Override
        public boolean writeIdField() {
            return false;
        }

        @Override
        public void toStringBuilder(final StringBuilder sb) {
        }

        @Override
        public String subType() {
            return "delete";
        }

        @Override
        public String uri(final String args) {
            return "/update?message-id=" + id;
        }
    }

    static class InvalidTypeException extends ParseException {
        private static final long serialVersionUID = 7610279873310183585L;

        private final String type;

        InvalidTypeException(final String error, final String type) {
            super(error, 0);
            this.type = type;
        }

        public String type() {
            return type;
        }
    }
}
