package ru.yandex.search.messenger.indexer;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.text.ParseException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;

import NMessengerProtocol.Message.EGenericResponseStatus;
import NMessengerProtocol.Message.TMessageInfoRequest;
import NMessengerProtocol.Message.TMessageInfoResponse;
import NMessengerProtocol.Message.TOutMessage;
import NMessengerProtocol.Message.TOutMessageRef;
import com.google.protobuf.CodedOutputStream;
import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.FormBodyPartBuilder;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.ByteArrayBody;
import org.apache.http.entity.mime.content.StringBody;

import ru.yandex.base64.Base64Encoder;
import ru.yandex.cityhash.CityHashingArrayOutputStream;
import ru.yandex.collection.IntPair;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.HttpStatusPredicates;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.NByteArrayEntityAsyncConsumerFactory;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.StatusCodeAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.server.UpstreamStater;
import ru.yandex.http.util.server.UpstreamStaterFutureCallback;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.StringPrefix;
import ru.yandex.stater.StatsConsumer;
import ru.yandex.url.processor.UrlProcessor;
import ru.yandex.util.string.StringUtils;

@SuppressWarnings("FutureReturnValueIgnored")
public class MessengerMessageHandler
    extends MessengerIndexHandlerBase<MessageIndexSession, MessengerMessage>
{
    public static final int VERSION = 2;
    private static final int URLS_MAX_SIZE = 5;
    private static final String CHAT_ID = "chat-id";
    private static final String MESSAGE = "message";
    private static final String MESSAGE_BIN = "message.bin";
    private static final String MIXED = "mixed";
    private static final String URI = "uri";

    private final ThreadLocal<CityHashingArrayOutputStream> outTls =
        ThreadLocal.<CityHashingArrayOutputStream>withInitial(
            () -> new CityHashingArrayOutputStream());
    private final String uri;
    private final HttpHost host;
    private final String tasksService;
    private MessengerRouterStater upstreamStater;

    public MessengerMessageHandler(
        final Malo malo,
        final UpstreamStater producerStater)
    {
        super(malo, malo.config().messagesService(), producerStater);
        uri = malo.config().messages().uri().getPath();
        host = malo.config().messages().host();
        tasksService = malo.config().tasksService();
    }

    @Override
    public UpstreamStater upstreamStater(final long metricsTimeFrame) {
        upstreamStater = new MessengerRouterStater(
            metricsTimeFrame,
            "messenger-router-messages");
        return upstreamStater;
    }

    @Override
    public MessageIndexSession indexSession(final MaloRequest request)
        throws HttpException, IOException
    {
        final String chatId = request.params().getString(CHAT_ID);
        final long timestamp = request.params().getLong("timestamp");
        String clearChatGuid = request.params().getString("clear_chat_guid", null);
        request.session().logger().info("input CGI mesage: chat-id: " + chatId
            + ", timestamp: " + timestamp);
        return new MessageIndexSession(request, chatId, timestamp, clearChatGuid);
    }

    @Override
    public MessageIndexSession postIndexSession(final PostRequestPart post)
        throws HttpException, IOException
    {
        TOutMessageRef inputMessage = TOutMessageRef.parseFrom(post.body());
        final String chatId = inputMessage.getChatId();
        final long timestamp = inputMessage.getTimestamp();
        String clearChatGuid = null;
        if (inputMessage.hasNonHistoryMessage()) {
            if (inputMessage.getNonHistoryMessage().getClientMessage().hasClearUserHistory()) {
                clearChatGuid = inputMessage.getNonHistoryMessage().getServerMessageInfo().getFrom().getGuid();
            }
        }
        post.session().logger().info("input TOutMessageRef: ChatId: "
            + chatId + ", "
            + "timestamp: " + timestamp + " clearChatGuid " + clearChatGuid);
        if (post.session().logger().isLoggable(Level.FINE)) {
            post.session().logger().fine("input TOutMessage: " + inputMessage);
        }
        if (post.session().logger().isLoggable(Level.FINEST)) {
            Base64Encoder encoder = new Base64Encoder();
            encoder.process(post.body());
            post.session().logger().finest("input byte array: "
                + encoder.toString());
        }
        if (chatId.isEmpty() && timestamp == 0) {
            String msg = "Corrupted TOutMessageRef: "
                + "empty chatId, timestamp = 0. Skipping";
            post.session().logger().info(msg);
            malo.skipImmediately(msg);
        }
        return new MessageIndexSession(post, chatId, timestamp, clearChatGuid);
    }

    @Override
    public void handle(
        final MessageIndexSession indexSession,
        final FutureCallback<MessengerMessage> callback)
        throws HttpException, IOException
    {
        final ProxySession session = indexSession.session();

        if (indexSession.clearChatMessage()) {
            session.logger().info("Clear chat message for " + indexSession.chatId());

            ClearChatMessage clearChatMessage = new ClearChatMessage(
                indexSession.clearChatGuid(),
                indexSession.chatId(),
                indexSession.timestamp());

            byte[] data = writeSingleMessage(clearChatMessage);

            BasicAsyncRequestProducerGenerator post = new BasicAsyncRequestProducerGenerator(
                clearChatMessage.uri("&topic=" + indexSession.topic() + "&offset=" + indexSession.offset()),
                data,
                ContentType.APPLICATION_JSON);
            post.addHeader(YandexHeaders.SERVICE, malo.config().usersService());
            post.addHeader(
                YandexHeaders.ZOO_SHARD_ID,
                clearChatMessage.prefixHash());

            AsyncClient client =
                malo.producerClient().adjust(indexSession.session().context());
            client.execute(
                producerHost,
                post,
                new StatusCheckAsyncResponseConsumerFactory<IntPair<String>>(
                    x -> x < HttpStatus.SC_BAD_REQUEST
                        || x == HttpStatus.SC_CONFLICT,
                    new StatusCodeAsyncConsumerFactory<String>(
                        AsyncStringConsumerFactory.INSTANCE)),
                indexSession.session().listener().createContextGeneratorFor(client),
                new UpstreamStaterFutureCallback<>(
                    new AbstractProxySessionCallback<Object>(indexSession.session()) {
                        @Override
                        public void completed(final Object t) {
                            callback.completed(null);
                        }
                    },
                    producerStater));
            return;
        }

        AsyncClient client = malo.messagesClient().adjust(session.context());
        TMessageInfoRequest messageRequest =
            TMessageInfoRequest.newBuilder()
                .setChatId(indexSession.chatId())
                .setTimestamp(indexSession.timestamp())
                .build();

        CityHashingArrayOutputStream out = outTls.get();
        out.reset();
        CodedOutputStream googleOut = CodedOutputStream.newInstance(out);

        messageRequest.writeTo(googleOut);
        googleOut.flush();

        byte[] postData = out.toByteArrayWithVersion(VERSION);
//        Base64Encoder encoder = new Base64Encoder();
//        encoder.process(postData);
//        session.logger().info("Coded message: " + encoder.toString());
//        session.logger().info("ych: " + out.yandexCityHash());
//        session.logger().info("ch: " + out.cityHash());

        final BasicAsyncRequestProducerGenerator post =
            new BasicAsyncRequestProducerGenerator(
                uri,
                postData,
                ContentType.DEFAULT_BINARY);
        client.execute(
            host,
            post,
            new StatusCheckAsyncResponseConsumerFactory<IntPair<HttpEntity>>(
                HttpStatusPredicates.NON_PROTO_FATAL,
                new StatusCodeAsyncConsumerFactory<HttpEntity>(
                    NByteArrayEntityAsyncConsumerFactory.INSTANCE)),
            session.listener().createContextGeneratorFor(client),
            new UpstreamStaterFutureCallback<>(
                new MessageResponseCallback(indexSession, callback),
                upstreamStater));
    }

    private byte[] messageToByteArray(final MessengerMessage message)
        throws IOException
    {
        final ByteArrayOutputStream baos = baosTls.get();
        baos.reset();
        CodedOutputStream googleOut = CodedOutputStream.newInstance(baos);

        message.message.writeTo(googleOut);
        googleOut.flush();

        return baos.toByteArray();
    }

    //CSOFF: MultipleStringLiterals
    //CSOFF: ReturnCount
    private String updateMessageInfoUri(
        final String producerName,
        final MessengerMessage message,
        final MessageIndexSession session)
        throws HttpException
    {
        if (message.subType().equals("text")) {
            String text = ((MessengerMessage.TextMessage) message).text();
            Set<String> urls = parseUrls(text);
            Iterator<String> iter = urls.iterator();

            if (iter.hasNext()) {
                StringBuilder urlsParam = new StringBuilder(iter.next());
                for (int i = 1; i < URLS_MAX_SIZE && iter.hasNext(); i++) {
                    urlsParam.append('\n');
                    urlsParam.append(iter.next());
                }
                QueryConstructor query =
                    new QueryConstructor("/update-message-info?");
                query.append("rca-url", urlsParam.toString());
                query.append(CHAT_ID, message.chatId());
                query.append("message-id", message.id());
                query.append("message-timestamp", message.timestamp());
                query.append("message-type", message.type());
                query.append("topic", producerName);
                query.append("partition", session.partition());
                query.append("offset", session.offset());
                return query.toString();
            } else {
                session.logger().info("No urls in " + text);
                return null;
            }
        } else {
            return null;
        }
    }
    //CSON: ReturnCount

    private String updateInfoUri(
        final String producerName,
        final MessengerMessage message,
        final MessageIndexSession session)
    {
        return "/update-chat-info?message-id=" + message.id()
            + "&chat-id=" + message.chatId()
            + "&message-timestamp=" + message.timestamp()
            + "&message-seqno=" + message.seqNo()
            + "&message-from-guid=" + message.fromGuid()
            + "&topic=" + producerName
            + "&partition=" + session.partition()
            + "&offset=" + session.offset();
    }

    public static Set<String> parseUrls(final String text) {
        UrlCollector collector = new UrlCollector();
        UrlProcessor processor = new UrlProcessor(collector);
        String fixed = text // https://st.yandex-team.ru/PS-3815
            .replaceAll("\\[.*]\\(", " ")
            .replaceAll("\\)\\[.*]", " ")
            .replaceAll("^\\(|\\s\\(|]\\(|\\)$|\\)\\s|\\)\\[", " ");
        processor.process(fixed.toCharArray());
        processor.process();
        return collector.urls();
    }

    //CSOFF: ParameterNumber
    private String indexPerUserUri(
        final String producerName,
        final MessengerMessage message,
        final MessageIndexSession session,
        final boolean delete)
    {
        return "/index-per-user?message-id=" + message.id()
            + "&chat-id=" + message.chatId()
            + "&message-timestamp=" + message.timestamp()
            + "&message-seqno=" + message.seqNo()
            + "&topic=" + producerName
            + "&partition=" + session.partition()
            + "&offset=" + session.offset()
            + "&delete=" + delete;
    }
    //CSON: ParameterNumber
    //CSON: MultipleStringLiterals

    private String producerName(final MessageIndexSession session) {
        return session.partition() + '@' + session.topic() + "_tasks";
    }

    //CSOFF: ReturnCount
    //CSOFF: MethodLength
    @Override
    protected void preprocessMessage(
        final MessageIndexSession session,
        final MessengerMessage message,
        final FutureCallback<MessengerMessage> nextCallback)
        throws HttpException
    {
        if (session.skipTasks()) {
            nextCallback.completed(message);
        } else if (message == null) {
            session.logger().info("Skipping tasks for null message with"
                + " offset: " + session.offset());
            nextCallback.completed(null);
        } else {
            try {
                boolean delete = false;
                if (message instanceof MessengerMessage.DeleteMessage) {
                    session.logger().info("Skipping rca task for <Delete> "
                        + "message offset: " + session.offset());
                    delete = true;
                }

                String producerName = producerName(session);
                StringBuilder commonUri =
                    new StringBuilder("/tasks_store?topic=");
                commonUri.append(producerName);
                commonUri.append("&offset=");
                commonUri.append(session.offset());
                commonUri.append("&partition=" + session.partition());

                MultipartEntityBuilder builder =
                    MultipartEntityBuilder.create();
                builder.setMimeSubtype(MIXED);

                int parts = 0;
                FormBodyPartBuilder partBuilder = FormBodyPartBuilder.create();
                partBuilder.addField(
                    YandexHeaders.URI,
                    updateInfoUri(producerName, message, session));
                if (!session.ignoreTaskPosition()) {
                    partBuilder.addField(
                        YandexHeaders.PRODUCER_POSITION,
                        session.offset());
                }
                partBuilder.addField(
                    YandexHeaders.ZOO_SHARD_ID,
                    session.partition());
                partBuilder.setBody(
                    new StringBody(
                        "",
                        ContentType.APPLICATION_FORM_URLENCODED));
                partBuilder.setName(URI);
                builder.addPart(partBuilder.build());

                partBuilder = FormBodyPartBuilder.create();
                parts++;
                    //rca
                if (!delete) {
                    String updateMessageInfoUri = updateMessageInfoUri(
                        producerName,
                        message,
                        session);
                    if (updateMessageInfoUri != null) {
                        partBuilder.addField(
                            YandexHeaders.URI,
                            updateMessageInfoUri);
                        partBuilder.addField(
                            YandexHeaders.ZOO_SHARD_ID,
                            //session.partition());
                            String.valueOf(new StringPrefix(message.chatId()).hash() % 61));
                        partBuilder.setBody(
                            new StringBody(
                                "",
                                ContentType.APPLICATION_FORM_URLENCODED));
                        partBuilder.setName(URI);
                        builder.addPart(partBuilder.build());

                        partBuilder = FormBodyPartBuilder.create();
                        parts++;
                    }
                }
                if (malo.mailProducerClient() != null) {
                    partBuilder.addField(
                        YandexHeaders.URI,
                        indexPerUserUri(
                            producerName,
                            message,
                            session,
                            delete));
                    partBuilder.addField(
                        YandexHeaders.ZOO_SHARD_ID,
                        session.partition());
                    partBuilder.setBody(
                        new ByteArrayBody(
                            messageToByteArray(message),
                            ContentType.DEFAULT_BINARY,
                            MESSAGE_BIN));
                    partBuilder.setName(MESSAGE);
                    builder.addPart(partBuilder.build());
                    parts++;
                }
                if (parts == 0) {
                    session.logger().info("Skipping tasks for message: "
                        + message + ", offset: " + session.offset()
                        + ": no tasks collected");
                    nextCallback.completed(message);
                    return;
                }
                final BasicAsyncRequestProducerGenerator post =
                    new BasicAsyncRequestProducerGenerator(
                        new String(commonUri),
                    builder.build());
                post.addHeader(YandexHeaders.SERVICE, tasksService);
                if (!session.ignoreTaskPosition()) {
                    post.addHeader(YandexHeaders.PRODUCER_NAME, producerName);
                }
                AsyncClient client =
                    malo.producerClient().adjust(session.session().context());
                client.execute(
                    producerHost,
                    post,
                    new StatusCheckAsyncResponseConsumerFactory<
                        IntPair<String>>(
                        x -> x < HttpStatus.SC_BAD_REQUEST
                            || x == HttpStatus.SC_CONFLICT,
                        new StatusCodeAsyncConsumerFactory<String>(
                            AsyncStringConsumerFactory.INSTANCE)),
                    session.session().listener()
                        .createContextGeneratorFor(client),
                    new UpstreamStaterFutureCallback<>(
                        new PreprocessMessageCallback(
                            session.session(),
                            message,
                            nextCallback),
                        producerStater));
            } catch (IOException e) {
                //This should never be happen
                session.logger().log(
                    Level.SEVERE,
                    "Message tasks send failed",
                    e);
                nextCallback.failed(e);
            }
        }
    }

    @Override
    protected void preprocessMessages(
        final List<MessageIndexSession> sessions,
        final List<MessengerMessage> messages,
        final FutureCallback<List<MessengerMessage>> nextCallback)
        throws HttpException
    {
        final MessageIndexSession firstSession = sessions.get(0);
        if (firstSession.skipTasks()) {
            nextCallback.completed(messages);
            return;
        }
        final MessageIndexSession lastSession =
            sessions.get(sessions.size() - 1);
        String producerName = producerName(firstSession);
        StringBuilder commonUri = new StringBuilder("/batch_task_store?topic=");
        commonUri.append(producerName);
        commonUri.append("&offsets=");
        commonUri.append(firstSession.offset());
        commonUri.append('-');
        commonUri.append(lastSession.offset());
        commonUri.append("&batch-size=");
        commonUri.append(messages.size());

        MultipartEntityBuilder builder = MultipartEntityBuilder.create();
        builder.setMimeSubtype(MIXED);
        Iterator<MessageIndexSession> sessionIter = sessions.iterator();
        int parts = 0;
        for (MessengerMessage message: messages) {
            MessageIndexSession session = sessionIter.next();
            if (message == null) {
                session.logger().info("Skipping task for null message with "
                    + "offset: " + session.offset());
                continue;
            }
            boolean delete = false;
            if (message instanceof MessengerMessage.DeleteMessage) {
                session.logger().info("Skipping some chat task for <Delete> "
                    + "message offset: " + session.offset());
                delete = true;
            }
            int msgParts = 0;
            FormBodyPartBuilder partBuilder = FormBodyPartBuilder.create();
            partBuilder.addField(
                YandexHeaders.URI,
                updateInfoUri(producerName, message, session));
            if (!session.ignoreTaskPosition()) {
                partBuilder.addField(
                    YandexHeaders.PRODUCER_POSITION,
                    session.offset());
            }
            partBuilder.addField(
                YandexHeaders.ZOO_SHARD_ID,
                session.partition());
            partBuilder.setBody(
                new StringBody(
                    "",
                    ContentType.APPLICATION_FORM_URLENCODED));
            partBuilder.setName(URI);
            builder.addPart(partBuilder.build());
            partBuilder = FormBodyPartBuilder.create();
            msgParts++;

            if (!delete) {
                //rca
                String updateMessageInfoUri = updateMessageInfoUri(
                    producerName,
                    message,
                    session);
                if (updateMessageInfoUri != null) {
                    partBuilder.addField(
                        YandexHeaders.URI,
                        updateMessageInfoUri);
                    partBuilder.addField(
                        YandexHeaders.ZOO_SHARD_ID,
                        String.valueOf(new StringPrefix(message.chatId()).hash() % 61));
                    partBuilder.setBody(
                        new StringBody(
                            "",
                            ContentType.APPLICATION_FORM_URLENCODED));
                    partBuilder.setName(URI);
                    builder.addPart(partBuilder.build());
                    partBuilder = FormBodyPartBuilder.create();
                    msgParts++;
                }
            }

            try {
                if (malo.mailProducerClient() != null) {
                    partBuilder.addField(
                        YandexHeaders.URI,
                        indexPerUserUri(
                            producerName,
                            message,
                            session,
                            delete));
                    partBuilder.addField(
                        YandexHeaders.ZOO_SHARD_ID,
                        session.partition());
                    partBuilder.setBody(
                        new ByteArrayBody(
                            messageToByteArray(message),
                            ContentType.DEFAULT_BINARY,
                            MESSAGE_BIN));
                    partBuilder.setName(MESSAGE);
                    builder.addPart(partBuilder.build());
                    msgParts++;
                }
            } catch (IOException e) {
                //This should never be happen
                firstSession.logger().log(
                    Level.SEVERE,
                    "Message tasks composing failed",
                    e);
                nextCallback.failed(e);
                return;
            }
            if (msgParts == 0) {
                session.logger().info("Skipping tasks for message:  "
                    + message + ", offset:  " + session.offset()
                    + ": no tasks collected ");
                continue;
            }

            parts++;
        }
        if (parts == 0) {
            firstSession.logger().info(
                "No indexable (non null) messages found, skipping tasks");
            nextCallback.completed(messages);
            return;
        }
        try {
            final BasicAsyncRequestProducerGenerator post =
                new BasicAsyncRequestProducerGenerator(
                    new String(commonUri),
                    builder.build());
            post.addHeader(YandexHeaders.SERVICE, tasksService);
            if (!firstSession.ignoreTaskPosition()) {
                post.addHeader(YandexHeaders.PRODUCER_NAME, producerName);
            }
            AsyncClient client =
                malo.producerClient().adjust(firstSession.session().context());
            client.execute(
                producerHost,
                post,
                new StatusCheckAsyncResponseConsumerFactory<IntPair<String>>(
                    x -> x < HttpStatus.SC_BAD_REQUEST
                        || x == HttpStatus.SC_CONFLICT,
                    new StatusCodeAsyncConsumerFactory<String>(
                        AsyncStringConsumerFactory.INSTANCE)),
                firstSession.session().listener()
                    .createContextGeneratorFor(client),
                new UpstreamStaterFutureCallback<>(
                    new PreprocessMessagesCallback(
                        firstSession.session(),
                        messages,
                        nextCallback),
                    producerStater));
        } catch (IOException e) {
            //This should never be happen
            firstSession.logger().log(
                Level.SEVERE,
                "Message tasks batch send failed",
                e);
            nextCallback.failed(e);
        }
    }
    //CSON: ReturnCount

    private class MessageResponseCallback
        extends AbstractFilterFutureCallback<
            IntPair<HttpEntity>,
            MessengerMessage>
    {
        private final MessageIndexSession session;

        MessageResponseCallback(
            final MessageIndexSession session,
            final FutureCallback<MessengerMessage> callback)
        {
            super(callback);
            this.session = session;
        }

        @Override
        public void completed(final IntPair<HttpEntity> response) {
            HttpEntity entity = response.second();
            int code = response.first();
            try {
                switch (code) {
                    case HttpStatus.SC_OK:
                        parseResponse(entity);
                        break;
                    case HttpStatus.SC_NOT_FOUND:
                        session.logger().info("Message with chatId: "
                            + session.chatId() + ','
                            + ", timestamp:  "
                            + session.timestamp() + " not found");
                        callback.completed(
                            MessengerMessage.deleteMessage(
                                session.chatId(),
                                session.timestamp()));
                        break;
                    default:
                        malo.badResponse(
                            uri,
                            "Message info getting error: code=" + code);
                }
            } catch (HttpException | IOException | ParseException e) {
                failed(e);
            }
        }

        private void parseResponse(final HttpEntity entity)
            throws IOException, HttpException, ParseException
        {
            InputStream is = entity.getContent();
            long skipped =
                is.skip(CityHashingArrayOutputStream.THEADER_SIZE);
            if (skipped != CityHashingArrayOutputStream.THEADER_SIZE) {
                malo.badResponse(
                    uri,
                    "Header skip failed");
            }
            TMessageInfoResponse response =
                TMessageInfoResponse.parseFrom(is);
            if (response.getStatus() != EGenericResponseStatus.Success) {
                malo.badResponse(
                    uri,
                    response.getStatus(),
                    response.getErrorInfo());
            }
            TOutMessage message = response.getMessage();
            try {
                MessengerMessage parsedMessage =
                    MessengerMessage.fromTOutMessage(message);
                session.logger().info("Message info: " + parsedMessage);
//                if (parsedMessage == null) {
//                    session.logger().info(
//                        "Unindexable message type received, skiping");
//                    callback.completed(null);
//                } else {
                callback.completed(parsedMessage);
//                }
            } catch (MessengerMessage.InvalidTypeException e) {
                final String type = e.type().intern();
                upstreamStater.addInvalidType(type);
                session.logger().info(
                    "Invalid message type received: " + type + ", skiping");
                callback.completed(null);
            } catch (ParseException e) {
                session.logger().severe("Error parsing message_info response "
                    + " with chat-id: " + session.chatId()
                    + " timestamp: " + session.timestamp());
                throw e;
            }
        }
    }

    private static class PreprocessMessageCallback
        extends AbstractFilterFutureCallback<IntPair<String>, MessengerMessage>
    {
        private final ProxySession session;
        private final MessengerMessage message;

        PreprocessMessageCallback(
            final ProxySession session,
            final MessengerMessage message,
            final FutureCallback<MessengerMessage> finalCallback)
        {
            super(finalCallback);
            this.session = session;
            this.message = message;
        }

        @Override
        public void completed(final IntPair<String> response) {
            session.logger().info(
                "Tasks for Message id: "
                + message.id() + " successfuly indexed with response: "
                + response.second());
            callback.completed(message);
        }
    }

    private static class PreprocessMessagesCallback
        extends AbstractFilterFutureCallback<
            IntPair<String>,
            List<MessengerMessage>>
    {
        private final ProxySession session;
        private final List<MessengerMessage> messages;

        PreprocessMessagesCallback(
            final ProxySession session,
            final List<MessengerMessage> messages,
            final FutureCallback<List<MessengerMessage>> finalCallback)
        {
            super(finalCallback);
            this.session = session;
            this.messages = messages;
        }

        @Override
        public void completed(final IntPair<String> response) {
            session.logger().info(
                "Tasks batch successfuly indexed with response: "
                    + response.second());
            callback.completed(messages);
        }
    }

    private static class MessengerRouterStater extends UpstreamStater {
        private static final String INVALIDS = "-invalid-messages_ammm";

        private final String prefix;
        private final TimeFrameQueue<String> invalidTypeMessages;

        MessengerRouterStater(
            final long metricsTimeFrame,
            final String prefix)
        {
            super(metricsTimeFrame, prefix);
            invalidTypeMessages = new TimeFrameQueue<>(metricsTimeFrame);
            this.prefix = prefix + '-';
        }

        @Override
        public <E extends Exception> void stats(
            final StatsConsumer<? extends E> statsConsumer)
            throws E
        {
            super.stats(statsConsumer);
            Map<String, int[]> invalidTypes = new HashMap<>();
            int invalidTotal = 0;
            for (String type: invalidTypeMessages) {
                invalidTypes.computeIfAbsent(type, (k) -> new int[1])[0]++;
                invalidTotal++;
            }

            for (Map.Entry<String, int[]> entry: invalidTypes.entrySet()) {
                int count = entry.getValue()[0];
                statsConsumer.stat(
                    StringUtils.concat(prefix, entry.getKey(), INVALIDS),
                    count);
            }
            statsConsumer.stat(
                StringUtils.concat(prefix, "total", INVALIDS),
                invalidTotal);
        }

        public void addInvalidType(final String type) {
            invalidTypeMessages.accept(type);
        }
    }
}

