package ru.yandex.search.messenger.indexer;

import java.io.IOException;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.server.UpstreamStater;
import ru.yandex.http.util.server.UpstreamStaterFutureCallback;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.Utf8JsonWriter;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.StringPrefix;

@SuppressWarnings("FutureReturnValueIgnored")
public class MessengerMessageInfoUpdateHandler
    extends MessengerIndexHandlerBase<
        UpdateMessageInfoIndexSession,
        IndexableMessage>
{
    private final Cache<String, JsonObject> rcaCache;

    private static final long MINUTES_EXPIRE = 60 * 6;
    private static final long MAX_SIZE = 500000;

    private static final String MESSAGE_RCA_DATA =
        "message_rca_data";
    private final HttpHost rcaHost;
    private UpstreamStater upstreamStater;

    public MessengerMessageInfoUpdateHandler(
        final Malo malo,
        final UpstreamStater producerStater)
    {
        super(malo, malo.config().messagesService(), producerStater);
        this.rcaHost = malo.config().rca().host();
        this.rcaCache =
            CacheBuilder.newBuilder()
                .expireAfterWrite(MINUTES_EXPIRE, TimeUnit.MINUTES)
                .maximumSize(MAX_SIZE)
                .concurrencyLevel(malo.config().workers()).build();
    }

    @Override
    public UpstreamStater upstreamStater(final long metricsTimeFrame) {
        upstreamStater = new UpstreamStater(metricsTimeFrame, "rca");
        return upstreamStater;
    }

    @Override
    public UpdateMessageInfoIndexSession indexSession(final MaloRequest request)
        throws HttpException, IOException
    {
        return new UpdateMessageInfoIndexSession(request);
    }

    @Override
    public UpdateMessageInfoIndexSession postIndexSession(
        final PostRequestPart postRequest)
        throws HttpException, IOException
    {
        return new UpdateMessageInfoIndexSession(postRequest);
    }

    @Override
    public void handle(
        final UpdateMessageInfoIndexSession session,
        final FutureCallback<IndexableMessage> callback)
        throws HttpException
    {
        if (malo.rcaClient() != null && session.rcaUrls() != null && !session.rcaUrls().isEmpty()) {
            MultiFutureCallback<JsonObject> multiCallback
                = new MultiFutureCallback<>(
                new UpstreamStaterFutureCallback<>(
                    new RcaCallback(session, callback),
                    upstreamStater));
            for (int i = 0; i < session.linksToExtractMax() && i < session.rcaUrls().size(); i++) {
                String url = session.rcaUrls().get(i);
                FutureCallback<JsonObject> oneUrlCallback = multiCallback.newCallback();
                if (!Malo.urlIsBanned(url)) {
                    JsonObject rcaResp = rcaCache.getIfPresent(url);
                    if (rcaResp != null) {
                        oneUrlCallback.completed(rcaResp);
                        continue;
                    }
                    QueryConstructor query =
                        new QueryConstructor(
                            "/urls?account=messenger_search&crawl=1");
                    query.append("url", url);
                    session.session().logger().info(
                        "Sending request to RCA for url: " + url);
                    AsyncClient client = malo.rcaClient().adjust(
                        session.session().context());

                    final BasicAsyncRequestProducerGenerator get =
                        new BasicAsyncRequestProducerGenerator(query.toString());
                    client.execute(
                        rcaHost,
                        get,
                        JsonAsyncTypesafeDomConsumerFactory.INSTANCE,
                        session.session().listener().createContextGeneratorFor(
                            client),
                        oneUrlCallback);
                } else {
                    oneUrlCallback.completed(JsonMap.EMPTY);
                }
            }
            multiCallback.done();
        } else {
            callback.completed(null);
        }
    }

    @SuppressWarnings("HidingField")
    private class RcaCallback
        extends AbstractProxySessionCallback<List<JsonObject>>
    {
        private final FutureCallback<IndexableMessage> callback;
        private final UpdateMessageInfoIndexSession session;

        RcaCallback(
            final UpdateMessageInfoIndexSession session,
            final FutureCallback<IndexableMessage> callback)
        {
            super(session.session());
            this.session = session;
            this.callback = callback;
        }

        @Override
        public void completed(final List<JsonObject> result) {
            UpdateMessage message;
            try {
                message = new UpdateMessage(
                    session.chatId(),
                    session.messageId(),
                    session.messageType(),
                    result);
                int urlsSize = session.rcaUrls().size();
                for (int i = 0; i < result.size() && i < urlsSize; i++) {
                    rcaCache.put(session.rcaUrls().get(i), result.get(i));
                }
            } catch (JsonException je) {
                callback.failed(je);
                return;
            }

            callback.completed(message);
        }

        @Override
        public void failed(Exception e) {
            if (e instanceof BadResponseException && ((BadResponseException) e).statusCode() == HttpStatus.SC_BAD_REQUEST) {
                session.logger().warning("got 400 code from rca, skipping urls " + session.rcaUrls());
                JsonMap map = new JsonMap(BasicContainerFactory.INSTANCE);
                map.put("rca_error", new JsonString(e.getMessage()));
                JsonList list = new JsonList(BasicContainerFactory.INSTANCE);
                list.add(map);
                completed(list);
                return;
            }
            super.failed(e);
        }
    }

    private static class UpdateMessage extends PrefixedIndexableMessage {
        private final String chatId;
        private final String messageId;
        private final String messageType;
        private final JsonObject rcaData;
        private final String rcaSnippet;
        private final String rcaTitle;

        //CSOFF: ParameterNumber
        UpdateMessage(
            final String chatId,
            final String messageId,
            final String messageType,
            final List<JsonObject> rcaData)
            throws JsonException
        {
            super(new StringPrefix(chatId), false);
            this.chatId = chatId;
            this.messageId = messageId;
            this.messageType = messageType;
            this.rcaData = new JsonList(BasicContainerFactory.INSTANCE);
            this.rcaData.asList().addAll(rcaData);
            if (!rcaData.isEmpty()) {
                this.rcaSnippet = rcaData.get(0).asMap().getString("snippet", null);
                this.rcaTitle = rcaData.get(0).asMap().getString("title", null);
            } else {
                this.rcaSnippet = null;
                this.rcaTitle = null;
            }
        }
        //CSON: ParameterNumber

        @Override
        public String id() {
            return messageId;
        }

        @Override
        public String type() {
            return messageType;
        }

        @Override
        protected void writeDocumentFields(final Utf8JsonWriter writer)
            throws IOException
        {
            writer.key(MESSAGE_RCA_DATA);
            writer.value(JsonType.NORMAL.toString(rcaData));
            writer.key("message_first_link_snippet");
            writer.value(rcaSnippet);
            writer.key("message_first_link_title");
            writer.value(rcaTitle);
        }

        @Override
        protected void writeGetFields(
            final Utf8JsonWriter writer,
            final Set<String> fields)
            throws IOException
        {
        }

        @Override
        public String uri(final String args) {
            return "/update?chat-id=" + chatId + args;
        }
    }
}
