package ru.yandex.search.messenger.indexer;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.text.ParseException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Set;
import java.util.logging.Level;

import NMessengerProtocol.Search.TDocument;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.HttpStatusPredicates;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.server.UpstreamStater;
import ru.yandex.http.util.server.UpstreamStaterFutureCallback;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.Utf8JsonWriter;
import ru.yandex.ps.search.messenger.UserFields;
import ru.yandex.search.messenger.indexer.v2org.OrgUserInfo;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.prefix.Prefix;

@SuppressWarnings("FutureReturnValueIgnored")
public class MessengerUsersHandler
    extends MessengerIndexHandlerBase<UserIndexSession, IndexableMessage>
{
    public static final String USER_UID = "user_uid";
    public static final Collection<String> PRESERVE_FIELDS =
        Collections.unmodifiableList(
            Arrays.asList(
                UserFields.HIDDEN_PVP_CHATS.stored(),
                UserFields.SEARCH_PRIVACY.stored(),
                UserFields.BLACKLISTED_USERS.stored(),
                UserFields.CHATS.stored()));

    public static final int USER_ORGANIZATIONS_LIMIT = 200;

    public static final byte[] REQUEST_RAVNO =
        "request=".getBytes(StandardCharsets.UTF_8);
    public static final String USER_ID = "user_id";
    public static final String GUID = "guid";
    public static final String DATA = "data";
    public static final String NONE_STATUS = "NONE_STATUS";
    public static final String IS_ROBOT = "is_robot";
    public static final String IS_DISMISSED = "is_dismissed";
    public static final String ORGANIZATIONS = "organizations";
    public static final String STATUS = "status";
    public static final String ORGANIZATION_ID = "organization_id";
    public static final String USER_WITH_ID = "User with id: ";

    private final String uri;
    private final HttpHost host;
    private UpstreamStater upstreamStater;

    public MessengerUsersHandler(
        final Malo malo,
        final UpstreamStater producerStater)
    {
        super(malo, malo.config().usersService(), producerStater);
        uri = malo.config().users().uri().getPath();
        host = malo.config().users().host();
    }

    @Override
    public UpstreamStater upstreamStater(final long metricsTimeFrame) {
        upstreamStater = new UpstreamStater(metricsTimeFrame, "meta-api-users");
        return upstreamStater;
    }

    @Override
    public UserIndexSession indexSession(final MaloRequest request)
        throws HttpException, IOException
    {
        final String userId = request.params().getString("user-id");
        request.session().logger().info("input CGI user-id: guid: " + userId);
        return new UserIndexSession(request, userId);
    }

    @Override
    public UserIndexSession postIndexSession(final PostRequestPart post)
        throws HttpException, IOException
    {
        TDocument inputMessage = TDocument.parseFrom(post.body());
        final String userId = inputMessage.getUuid();
        post.session().logger().info("input TDocument: uuid: " + userId);
        return new UserIndexSession(post, userId);
    }

    @Override
    public void handle(
        final UserIndexSession indexSession,
        final FutureCallback<IndexableMessage> callback)
        throws HttpException, IOException
    {
        final ProxySession session = indexSession.session();
        final String userId = indexSession.userId();
        AsyncClient client = malo.chatsClient().adjust(session.context());

        final byte[] postData;
        final ByteArrayOutputStream baos = baosTls.get();
        baos.reset();
        try (
            Utf8JsonWriter writer = JsonType.NORMAL.create(baos))
        {
            baos.write(REQUEST_RAVNO);
            writer.startObject();
            writer.key("method");
            writer.value("get_user");
            writer.key("params");
            writer.startObject();
            writer.key(GUID);
            writer.value(userId);
            writer.endObject();
            writer.endObject();
            writer.flush();
            postData = baos.toByteArray();
        }

        final BasicAsyncRequestProducerGenerator post =
            new BasicAsyncRequestProducerGenerator(
                uri,
                postData,
                ContentType.APPLICATION_FORM_URLENCODED);
        client.execute(
            host,
            post,
            new StatusCheckAsyncResponseConsumerFactory<JsonObject>(
                HttpStatusPredicates.NON_PROTO_FATAL,
                JsonAsyncTypesafeDomConsumerFactory.INSTANCE),
            session.listener().createContextGeneratorFor(client),
            new UpstreamStaterFutureCallback<>(
                new UsersResponseCallback(indexSession, callback),
                upstreamStater));
    }

    private class UsersResponseCallback
        extends AbstractFilterFutureCallback<JsonObject, IndexableMessage>
    {
        private final String userId;
        private final UserIndexSession session;

        UsersResponseCallback(
            final UserIndexSession session,
            final FutureCallback<IndexableMessage> callback)
        {
            super(callback);
            this.session = session;
            this.userId = session.userId();
        }

        @Override
        public void completed(final JsonObject response) {
            try {
                UserMessage message = parseResponseJson(response);
                callback.completed(message);
            } catch (ParseException e) {
                session.logger().log(
                    Level.SEVERE,
                    "Parse Error with user id: " + userId
                        + ". temporary skipping user ",
                    e);
                callback.completed(null);
            } catch (IOException | JsonException | HttpException e) {
                failed(e);
            }
//            session.response(HttpStatus.SC_BAD_REQUEST);
        }

        private UserMessage parseResponseJson(final JsonObject response)
            throws HttpException, IOException, JsonException, ParseException
        {
            session.logger().info("Users response: "
                + JsonType.HUMAN_READABLE.toString(response)
                + "/malo:" + malo);
            JsonMap map = response.asMap();
            String status = map.getOrNull(STATUS);
            UserMessage message = null;
            if ("ok".equals(status)) {
                JsonMap data = map.getMap(DATA);
                if (data.containsKey("organizations_v2")) {
                    message = new OrgUserInfo(data, malo.config().v2OrgUserService());
                } else {
                    message = new UserInfo(data);
                }
                if (message.userId == null) {
                    malo.badResponse(uri, "Empty user-id");
                }
            } else if ("error".equals(status)) {
                JsonMap data = map.getMap(DATA);
                String code = data.getOrNull("code");
                if ("user_does_not_exist".equals(code)
                    || "user_not_found".equals(code))
                {
                    session.logger().info(USER_WITH_ID + userId
                        + " was not found. Will send remove request");
                    message = new DeleteUser(new LongPrefix(0), userId);
                } else if ("unhandled".equals(code) || "bad_request".equals(code)) {
                    //TODO: should be fixed with another status code
                    session.logger().info(USER_WITH_ID + userId
                        + " has no active users. Will send remove request");
                    message = new DeleteUser(new LongPrefix(0), userId);
                } else if ("unprocessable_entity".equals(code)) {
                    //TODO: should be fixed with another status code
                    session.logger().info(USER_WITH_ID + userId
                        + " has invalid id format. Will send remove request.");
                    message = new DeleteUser(new LongPrefix(0), userId);
                } else {
                    malo.badResponse(
                        uri,
                        "Unhandled meta_api error code: " + code
                            + ". Expecting: user_does_not_exist|unhandled");
                }
            } else {
                malo.badResponse(
                    uri,
                    "Unhandled meta_api status: " + status
                        + ". Expecting: ok|error");
            }
            return message;
        }
    }

    private static class DeleteUser extends UserMessage {
        DeleteUser(final Prefix prefix, final String userId) {
            super(prefix, userId);
        }

        @Override
        protected void writeDocumentFields(final Utf8JsonWriter writer)
            throws IOException
        {
        }

        @Override
        protected void writeGetFields(
            final Utf8JsonWriter writer,
            final Set<String> fields)
            throws IOException
        {
        }

        @Override
        public String uri(final String args) {
            return "/delete?user-id=" + userId + args;
        }
    }

}
