package ru.yandex.search.messenger.indexer;

import java.io.IOException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.Utf8JsonWriter;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.util.string.StringUtils;

public class UserInfo extends UserMessage {
    protected final long uid;
    protected final String gender;
    protected final String defaultDisplayName;
    protected final String defaultNickname;
    protected final String defaultPosition;
    protected final String defaultWebsite;
    //        private final String defaultEmail;
    protected final String defaultDepartmentName;
    protected final boolean isRobot;
    protected final boolean isDismissed;
    protected final boolean isHomeworker;
    protected final boolean isDisplayRestricted;
    protected final long version;
    //        private final List<String> contacts;
    protected final double longitude;
    protected final double latitude;
    protected final boolean geo;
    protected final JsonMap json;
    protected final List<String> status;
    protected final List<Long> organizations;
    protected final Map<Long, String> displayNames = new HashMap<>();
    protected final Map<Long, String> nicknames = new HashMap<>();
    protected final Map<Long, String> positions = new HashMap<>();
    protected final Map<Long, String> websites = new HashMap<>();
    protected final Map<Long, String> departments = new HashMap<>();
    //        private final Map<String, String> emails;
    protected final long orgId;
    protected final Long service;
    protected final String affiliation;

    public UserInfo(final JsonMap json)
        throws JsonException, ParseException {
        super(new LongPrefix(0), json.getOrNull(MessengerUsersHandler.GUID));
        this.json = json;
        version = json.getLong("version");
        defaultDisplayName = json.getOrNull("passport_display_name");
        gender = json.getOrNull("gender");
        affiliation = json.getOrNull("affiliation");
        orgId = json.getLong("org_id");
        String defaultNickname;
        // https://st.yandex-team.ru/PS-3650
        if (orgId == 34L) {
            defaultNickname = json.getOrNull("mssngr_nickname");
        } else {
            defaultNickname = json.getOrNull("nickname");
        }
        this.defaultNickname = defaultNickname;
        defaultWebsite = json.getOrNull("website");

        uid = json.getLong("uid", Long.valueOf(-1));
        isHomeworker = json.getBoolean("is_homeworker", false);
        isDisplayRestricted = json.getBoolean("is_display_restricted", false);
        boolean isRobot = json.getBoolean(MessengerUsersHandler.IS_ROBOT, false);
        boolean isDismissed = json.getBoolean(MessengerUsersHandler.IS_DISMISSED, false);
        JsonList statuses = json.getListOrNull(MessengerUsersHandler.STATUS);
        if (statuses != null) {
            this.status = new ArrayList<>();
            for (JsonObject status : statuses) {
                String statusStr = status.asString();
                if (MessengerUsersHandler.IS_ROBOT.equals(statusStr)) {
                    isRobot = true;
                } else if (MessengerUsersHandler.IS_DISMISSED.equals(statusStr)) {
                    isDismissed = true;
                }
                this.status.add(statusStr);
            }
            if (this.status.size() == 0) {
                this.status.add(MessengerUsersHandler.NONE_STATUS);
            }
        } else {
            if (isRobot || isDismissed) {
                this.status = new ArrayList<>();
                if (isRobot) {
                    this.status.add(MessengerUsersHandler.IS_ROBOT);
                }
                if (isDismissed) {
                    this.status.add(MessengerUsersHandler.IS_DISMISSED);
                }
            } else {
                this.status = Collections.singletonList(MessengerUsersHandler.NONE_STATUS);
            }
        }
        this.organizations = new ArrayList<>();
        JsonList jsonOrgsList = json.getListOrNull(MessengerUsersHandler.ORGANIZATIONS);
        if (jsonOrgsList != null) {
            for (JsonObject org : jsonOrgsList) {
                organizations.add(org.asLong());
            }
        }
        this.isRobot = isRobot;
        this.isDismissed = isDismissed;

        JsonObject service = json.get("service");
        if (service != null && service.type() != JsonObject.Type.NULL) {
            this.service = service.asLong();
        } else {
            this.service = null;
        }

        EmployeeInfo employeeInfo =
            parseEmployeeInfo(json.getMapOrNull("employee_info"));
        defaultPosition = employeeInfo.position();
        defaultDepartmentName = employeeInfo.department();

        JsonList rawEmployeeInfos = json.getListOrNull("raw_employee_info");
        if (rawEmployeeInfos != null) {
            for (JsonObject rei : rawEmployeeInfos) {
                employeeInfo = parseEmployeeInfo(rei.asMap());
                if (employeeInfo.orgId() != -1) {
                    positions.put(
                        employeeInfo.orgId(),
                        employeeInfo.position());
                    departments.put(
                        employeeInfo.orgId(),
                        employeeInfo.department());
                }
            }
        }

        JsonList rawDisplayNames = json.getListOrNull("raw_display_name");
        if (rawDisplayNames != null) {
            for (JsonObject rdn : rawDisplayNames) {
                JsonMap dn = rdn.asMap();
                String displayName = dn.getOrNull("display_name");
                long orgId = dn.getLong(MessengerUsersHandler.ORGANIZATION_ID, -1L);
                if (orgId != -1 && displayName != null) {
                    displayNames.put(orgId, displayName);
                }
            }
        }

        JsonMap location = json.getMapOrNull("location");
        if (location != null) {
            JsonObject lon = location.get("longitude");
            JsonObject lat = location.get("latitude");
            if ((lon != null && lon.type() != JsonObject.Type.NULL)
                    && (lat != null && lat.type() != JsonObject.Type.NULL)) {
                longitude = lon.asDouble();
                latitude = lat.asDouble();
                geo = true;
            } else {
                geo = false;
                longitude = 0.;
                latitude = 0.;
            }
        } else {
            geo = false;
            longitude = 0.;
            latitude = 0.;
        }
    }

    private EmployeeInfo parseEmployeeInfo(final JsonMap employeeInfo)
        throws JsonException {
        EmployeeInfo ret = new EmployeeInfo();
        if (employeeInfo != null) {
            ret.position(employeeInfo.getOrNull("position"));
            JsonMap department = employeeInfo.getMapOrNull("department");
            if (department != null) {
                ret.department(department.getOrNull("name"));
            }
            ret.orgId(employeeInfo.getLong(MessengerUsersHandler.ORGANIZATION_ID, -1L));
        }
        return ret;
    }

    @Override
    public boolean multiMessage() {
        return true;
    }

    @Override
    public Collection<IndexableMessage> subMessages() {
        int limit =
            Math.min(organizations.size(), MessengerUsersHandler.USER_ORGANIZATIONS_LIMIT);
        List<IndexableMessage> messages =
            new ArrayList<>(limit + 1);
        messages.add(new SubMessage(0));
        for (long org : organizations) {
            messages.add(new SubMessage(org));
            if (messages.size() > limit) {
                break;
            }
        }
        return messages;
    }

    @Override
    protected void writeDocumentFields(final Utf8JsonWriter writer)
        throws IOException {
        throw new IOException(
            "writeDocumentFields called on root multimessage");
    }

    @Override
    protected void writeGetFields(
        final Utf8JsonWriter writer,
        final Set<String> fields)
        throws IOException {
        for (String field : fields) {
            switch (field) {
                case MessengerUsersHandler.USER_UID:
                    writer.key(MessengerUsersHandler.USER_UID);
                    writer.value(uid);
                    break;
                default:
                    break;
            }
        }
    }

    @Override
    public String uri(final String args) {
        return "/modify?user-id=" + userId + args;
    }

    public class SubMessage extends UserMessage {
        private final long orgId;

        public SubMessage(final long orgId) {
            super(
                new LongPrefix(orgId),
                UserInfo.this.userId,
                true);
            this.orgId = orgId;
        }

        @Override
        protected void writeDocumentFields(final Utf8JsonWriter writer)
            throws IOException {
            //leave this first, making this fields apears in index
            //first, so FieldsVisitor breaks early
            writer.key("user_geo");
            writer.value(geo);
            if (geo) {
                writer.key("user_latitude");
                writer.value(latitude);
                writer.key("user_longitude");
                writer.value(longitude);
                writer.key("user_latitude_raw");
                writer.value(latitude);
                writer.key("user_longitude_raw");
                writer.value(longitude);
                writer.key("geo_shard");
                //CSOFF: MagicNumber
                int geoShard =
                    (int) (((longitude + 180) * 180
                                + (latitude + 90)) * 10);
                writer.value(geoShard);
                //CSON: MagicNumber
            }
            writer.key(MessengerUsersHandler.USER_ID);
            writer.value(userId);
            //leave
            writer.key("user_version");
            writer.value(version);

            String displayName =
                getOrDefault(displayNames, defaultDisplayName);
            writer.key("user_display_name");
            writer.value(displayName);

            writer.key("user_gender");
            writer.value(gender);

            writer.key("user_affiliation");
            writer.value(affiliation);

            writer.key("user_affiliation_nn");
            if (affiliation != null) {
                writer.value(affiliation);
            } else {
                writer.value("na");
            }

            String nickname =
                getOrDefault(nicknames, defaultNickname);
            writer.key("user_nickname");
            writer.value(nickname);

            String position =
                getOrDefault(positions, defaultPosition);
            writer.key("user_position");
            writer.value(position);

            String website =
                getOrDefault(websites, defaultWebsite);
            writer.key("user_website");
            writer.value(website);

            String department =
                getOrDefault(departments, defaultDepartmentName);
            writer.key("user_department_name");
            writer.value(department);

            writer.key("user_is_robot");
            writer.value(isRobot);

            writer.key(ru.yandex.ps.search.messenger.UserFields.DISPLAY_RESTRICTED.stored());
            writer.value(isDisplayRestricted);

            writer.key("user_is_dismissed");
            writer.value(isDismissed);

            writer.key("user_is_homeworker");
            writer.value(isHomeworker);

            writer.key("user_status");
            writer.value(StringUtils.join(status, '\n'));

            if (organizations.size() > 0) {
                writer.key("user_organizations");
                StringBuilder sb = new StringBuilder();
                String sep = "";
                for (Long org : organizations) {
                    sb.append(sep);
                    sep = "\n";
                    sb.append(org);
                }
                writer.value(new String(sb));
            }

            writer.key("user_org_id");
//                if (orgId == 0) {
//                    writer.value(defaultOrgId);
//                } else {
            writer.value(UserInfo.this.orgId);
//                }

            writer.key(MessengerUsersHandler.USER_UID);
            writer.value(uid);

            writer.key("user_has_service");
            if (service != null) {
                writer.value(true);
                writer.key("user_service");
                writer.value(service);
            } else {
                writer.value(false);
            }
            //whole blob
            writer.key("user_data");
            writer.value(JsonType.NORMAL.toString(json));
        }

        protected String getOrDefault(
            final Map<Long, String> map,
            final String defaultValue) {
            String value = map.get(orgId);
            if (value == null) {
                value = defaultValue;
            }
            if (value == null && map.size() > 0) {
                value = map.values().iterator().next();
            }
            return value;
        }

        @Override
        protected void writeGetFields(
            final Utf8JsonWriter writer,
            final Set<String> fields)
            throws IOException {
            throw new IOException("writeGetFields is not implemented");
        }

        @Override
        public Collection<String> preserveFields() {
            return MessengerUsersHandler.PRESERVE_FIELDS;
        }

        @Override
        public String uri(final String args) {
            return "/update?sub&org_version=1&user-id=" + userId + args;
        }
    }

}
