package ru.yandex.search.messenger.indexer.user.settings;

import java.io.IOException;
import java.util.Map;
import java.util.Set;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.Utf8JsonWriter;
import ru.yandex.ps.search.messenger.UserFields;
import ru.yandex.search.messenger.indexer.IndexableMessage;
import ru.yandex.search.messenger.indexer.UserMessage;
import ru.yandex.search.messenger.indexer.user.settings.privacy.SearchBucketIndexSession;
import ru.yandex.search.prefix.LongPrefix;

public class HiddenUserChatsHandler implements SettingsSubTypeHandler {
    private final SettingsUpdateHandler handler;

    public HiddenUserChatsHandler(
        final SettingsUpdateHandler handler)
    {
        this.handler = handler;
    }

    @Override
    public void handle(
        final SearchBucketIndexSession session,
        final FutureCallback<IndexableMessage> indexCallback)
        throws HttpException, IOException
    {
        handler.sendBucketRequest(
            session,
            new HiddenChatsMetaCallback(session, indexCallback));
    }

    private static class HiddenChatsMetaCallback
        extends AbstractFilterFutureCallback<JsonObject, IndexableMessage>
    {
        private final SearchBucketIndexSession session;

        public HiddenChatsMetaCallback(
            final SearchBucketIndexSession session,
            final FutureCallback<? super IndexableMessage> callback)
        {
            super(callback);

            this.session = session;
        }

        @Override
        public void completed(final JsonObject meta) {
            try {
                UserMessage message = parseHiddenChats(meta);
                callback.completed(message);
            } catch (JsonException e) {
                failed(e);
            }
        }

        /*
            https://st.yandex-team.ru/PS-3508

         */
        private UserMessage parseHiddenChats(final JsonObject meta) throws JsonException {
            session.logger().info("MetaResponse " + JsonType.HUMAN_READABLE.toString(meta));
            JsonMap bucket = meta.asMap().getMap("data").getMap("bucket");
            String name = bucket.getString("bucket_name");
            JsonMap values = bucket.getMap("bucket_value");

            StringBuilder hiddenChats = new StringBuilder();
            for (Map.Entry<String, JsonObject> entry: values.entrySet()) {
                hiddenChats.append(entry.getKey());
                hiddenChats.append("\t");
                hiddenChats.append(entry.getValue().asLong());
                hiddenChats.append("\n");
            }

            if (hiddenChats.length() > 0) {
                hiddenChats.setLength(hiddenChats.length() - 1);
            }

            return new HiddenChatsUpdateMessage(
                session.userId(),
                hiddenChats.toString());
        }
    }

    private static class HiddenChatsUpdateMessage extends UserMessage {
        private final String hiddenChats;

        public HiddenChatsUpdateMessage(
            final String userId,
            final String hiddenChats)
        {
            super(new LongPrefix(0L), userId, false);

            this.hiddenChats = hiddenChats;
        }

        @Override
        protected void writeGetFields(
            final Utf8JsonWriter writer,
            final Set<String> fields)
            throws IOException
        {
        }

        @Override
        public String updateRequest() {
            return "user_id:" + userId;
        }

        @Override
        public boolean writeIdField() {
            return false;
        }

        @Override
        protected void writeDocumentFields(
            final Utf8JsonWriter writer)
            throws IOException
        {
            writer.key(UserFields.HIDDEN_PVP_CHATS.stored());
            writer.value(hiddenChats);
        }

        @Override
        public String uri(String args) {
            return "/update?hiddenChats&user-id=" + userId + args;
        }
    }
}
