package ru.yandex.search.messenger.indexer.user.settings;

import java.io.IOException;
import java.util.Set;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.Utf8JsonWriter;
import ru.yandex.search.messenger.UserFields;
import ru.yandex.search.messenger.indexer.IndexableMessage;
import ru.yandex.search.messenger.indexer.UserMessage;
import ru.yandex.search.messenger.indexer.user.settings.privacy.SearchBucketIndexSession;
import ru.yandex.search.prefix.LongPrefix;

public class RestrictionsUpdateHandler implements SettingsSubTypeHandler {
    private final SettingsUpdateHandler handler;

    public RestrictionsUpdateHandler(
        final SettingsUpdateHandler handler)
    {
        this.handler = handler;
    }

    @Override
    public void handle(
        final SearchBucketIndexSession session,
        final FutureCallback<IndexableMessage> indexCallback)
        throws HttpException, IOException
    {
        handler.sendBucketRequest(
            session,
            new RestrictionsMetaCallback(session, indexCallback));
    }

    private static class RestrictionsMetaCallback
        extends AbstractFilterFutureCallback<JsonObject, IndexableMessage>
    {
        private final SearchBucketIndexSession session;

        public RestrictionsMetaCallback(
            final SearchBucketIndexSession session,
            final FutureCallback<? super IndexableMessage> callback) {
            super(callback);

            this.session = session;
        }

        @Override
        public void completed(final JsonObject meta) {
            try {
                UserMessage message = parseRestrictions(meta);
                callback.completed(message);
            } catch (JsonException e) {
                failed(e);
            }
        }

        /*
            https://st.yandex-team.ru/PS-3543
            {
              "data": {
                "bucket": {
                  "bucket_name": "privacy",
                  "bucket_value": {
                    "calls": 0,
                    "invites": 1,
                    "online_status": 0,
                    "private_chats": 1,
                    "search": 0
                  },
                  "version": 1586280411370482
                }
              },
              "status": "ok"
            }
         */
        private UserMessage parseRestrictions(final JsonObject meta) throws JsonException {
            session.logger().info("MetaResponse " + JsonType.HUMAN_READABLE.toString(meta));
            JsonMap bucket = meta.asMap().getMap("data").getMap("bucket");
            String name = bucket.getString("bucket_name");
            JsonMap values = bucket.getMap("bucket_value");

            JsonList whitelist = values.getListOrNull("whitelist");
            JsonList blacklist = values.getListOrNull("blacklist");

            StringBuilder whitelistSb = new StringBuilder();
            if (whitelist != null) {
                for (JsonObject jo: whitelist) {
                    whitelistSb.append(jo.asString());
                    whitelistSb.append("\n");
                }
            }

            StringBuilder blacklistSb = new StringBuilder();
            if (blacklist != null) {
                for (JsonObject jo: blacklist) {
                    blacklistSb.append(jo.asString());
                    blacklistSb.append("\n");
                }
            }
            return new RestrictionsUpdateMessage(
                session.userId(),
                whitelistSb.toString(),
                blacklistSb.toString());
        }
    }

    private static class RestrictionsUpdateMessage extends UserMessage {
        private final String blacklist;
        private final String whitelist;

        public RestrictionsUpdateMessage(
            final String userId,
            final String whitelist,
            final String blacklist)
        {
            super(new LongPrefix(0), userId, false);

            this.blacklist = blacklist;
            this.whitelist = whitelist;
        }

        @Override
        protected void writeGetFields(
            final Utf8JsonWriter writer,
            final Set<String> fields)
            throws IOException
        {
        }

        @Override
        public String updateRequest() {
            return "user_id:" + userId;
        }

        @Override
        public boolean writeIdField() {
            return false;
        }

        @Override
        protected void writeDocumentFields(
            final Utf8JsonWriter writer)
            throws IOException
        {
            writer.key(UserFields.BLACKLISTED_USERS.stored());
            writer.value(blacklist);
            writer.key(UserFields.WHITELISTED_USERS.stored());
            writer.value(whitelist);
        }

        @Override
        public String uri(String args) {
            return "/update?restrictions&user-id=" + userId + args;
        }
    }
}
