package ru.yandex.search.messenger.indexer.user.settings;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;

import NMessengerProtocol.Search;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;

import ru.yandex.http.util.HttpStatusPredicates;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.server.UpstreamStater;
import ru.yandex.http.util.server.UpstreamStaterFutureCallback;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.Utf8JsonWriter;
import ru.yandex.search.messenger.indexer.IndexableMessage;
import ru.yandex.search.messenger.indexer.Malo;
import ru.yandex.search.messenger.indexer.MaloRequest;
import ru.yandex.search.messenger.indexer.MessengerIndexHandlerBase;
import ru.yandex.search.messenger.indexer.PostRequestPart;
import ru.yandex.search.messenger.indexer.user.settings.privacy.SearchBucketIndexSession;
import ru.yandex.search.messenger.indexer.user.settings.privacy.UserSearchPrivacyUpdateHandler;

public class SettingsUpdateHandler extends MessengerIndexHandlerBase<SearchBucketIndexSession, IndexableMessage> {
    private static final String PRIVACY_SUBTYPE = "privacy";
    private static final byte[] REQUEST_RAVNO =
        "request=".getBytes(StandardCharsets.UTF_8);

    protected final ThreadLocal<ByteArrayOutputStream> baosTls =
        ThreadLocal.<ByteArrayOutputStream>withInitial(
            () -> new ByteArrayOutputStream());
    private final String uri;
    private final HttpHost host;
    private UpstreamStater upstreamStater;
    private final Map<SearchBucketSubType, SettingsSubTypeHandler> subTypeHandlers;

    public SettingsUpdateHandler(final Malo malo, final UpstreamStater producerStater) {
        super(malo, malo.config().usersService(), producerStater);

        uri = malo.config().users().uri().getPath();
        host = malo.config().users().host();

        // don't forget to add fields to preserves in user/chat handlers
        Map<SearchBucketSubType, SettingsSubTypeHandler> subTypeMap = new LinkedHashMap<>();
        subTypeMap.put(SearchBucketSubType.PRIVACY, new UserSearchPrivacyUpdateHandler(this));
        subTypeMap.put(SearchBucketSubType.RESTRICTIONS, new RestrictionsUpdateHandler(this));
        subTypeMap.put(SearchBucketSubType.HIDDEN_PRIVATE_CHATS, new HiddenUserChatsHandler(this));

        this.subTypeHandlers = Collections.unmodifiableMap(subTypeMap);
    }

    @Override
    public UpstreamStater upstreamStater(final long metricsTimeFrame) {
        upstreamStater = new UpstreamStater(metricsTimeFrame, "meta-api-privacy");
        return upstreamStater;
    }

    @Override
    public SearchBucketIndexSession indexSession(final MaloRequest request)
        throws HttpException, IOException
    {
        String userId = request.params().getString("user-id");
        String subTypeStr = request.params().getString("subtype");
        SearchBucketSubType subType = SearchBucketSubType.parse(subTypeStr);
        if (subType == null) {
            throw new HttpException("Unknown subtype " + subTypeStr);
        }

        return new SearchBucketIndexSession(request, subType, userId);
    }

    @Override
    public SearchBucketIndexSession postIndexSession(final PostRequestPart post)
        throws HttpException, IOException
    {
        Search.TDocument document = Search.TDocument.parseFrom(post.body());
        final String userId = document.getUuid();
        post.session().logger().info("input TDocument: uuid: " + userId);
        SearchBucketSubType subType = SearchBucketSubType.parse(document.getSubType());
        if (subType == null) {
            post.session().logger().warning("Unknown subtype " + document.getSubType());
        }

        return new SearchBucketIndexSession(post, subType, document.getUuid());
    }

    @Override
    public void handle(
        final SearchBucketIndexSession session,
        final FutureCallback<IndexableMessage> indexCallback)
        throws HttpException, IOException
    {
        SettingsSubTypeHandler handler = null;
        if (session.subType() != null) {
            handler = subTypeHandlers.get(session.subType());
        }

        if (handler == null) {
            session.logger().warning(
                "Skipping message with unknown subtype " + session.subType().bucketName());
            session.session().response(HttpStatus.SC_OK);
            return;
        }

        session.logger().info("Handling " + session.subType().bucketName());
        handler.handle(session, indexCallback);
    }

    public void sendBucketRequest(
        final SearchBucketIndexSession session,
        final FutureCallback<JsonObject> callback)
        throws HttpException, IOException
    {
        AsyncClient client = malo.chatsClient().adjust(session.session().context());

        final byte[] postData;
        final ByteArrayOutputStream baos = baosTls.get();
        baos.reset();
        try (Utf8JsonWriter writer = JsonType.NORMAL.create(baos)) {
            baos.write(REQUEST_RAVNO);
            writer.startObject();
            writer.key("method");
            writer.value("get_bucket");
            writer.key("params");
            writer.startObject();
            writer.key("guid");
            writer.value(session.userId());
            writer.key("bucket_name");
            writer.value(session.subType().bucketName());
            writer.endObject();
            writer.endObject();
            writer.flush();
            postData = baos.toByteArray();
        }

        final BasicAsyncRequestProducerGenerator post =
            new BasicAsyncRequestProducerGenerator(
                uri,
                postData,
                ContentType.APPLICATION_FORM_URLENCODED);
        post.addHeader("X-Version", "6");
        FutureCallback<JsonObject> metaApiCallback = callback;
        if (upstreamStater != null) {
            metaApiCallback = new UpstreamStaterFutureCallback<>(callback, upstreamStater);
        }

        client.execute(
            host,
            post,
            new StatusCheckAsyncResponseConsumerFactory<>(
                HttpStatusPredicates.NON_PROTO_FATAL,
                JsonAsyncTypesafeDomConsumerFactory.INSTANCE),
            session.session().listener().createContextGeneratorFor(client),
            metaApiCallback);
    }
}
