package ru.yandex.search.messenger.indexer.user.settings.privacy;

import java.io.IOException;
import java.util.Set;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.Utf8JsonWriter;
import ru.yandex.search.messenger.SearchPrivacy;
import ru.yandex.search.messenger.UserFields;
import ru.yandex.search.messenger.indexer.IndexableMessage;
import ru.yandex.search.messenger.indexer.UserMessage;
import ru.yandex.search.messenger.indexer.user.settings.SettingsSubTypeHandler;
import ru.yandex.search.messenger.indexer.user.settings.SettingsUpdateHandler;
import ru.yandex.search.prefix.LongPrefix;

public class UserSearchPrivacyUpdateHandler
        implements SettingsSubTypeHandler
{
    private final SettingsUpdateHandler handler;
    public UserSearchPrivacyUpdateHandler(final SettingsUpdateHandler handler) {
        this.handler = handler;
    }

    @Override
    public void handle(
            final SearchBucketIndexSession session,
            final FutureCallback<IndexableMessage> indexCallback)
            throws HttpException, IOException
    {
        handler.sendBucketRequest(
            session,
            new PrivacyMetaCallback(session, indexCallback));
    }

    private static class PrivacyMetaCallback
        extends AbstractFilterFutureCallback<JsonObject, IndexableMessage>
    {
        private final SearchBucketIndexSession session;

        public PrivacyMetaCallback(
                final SearchBucketIndexSession session,
                final FutureCallback<? super IndexableMessage> callback) {
            super(callback);
            this.session = session;
        }

        @Override
        public void completed(final JsonObject meta) {
            try {
                UserMessage message = parsePrivacyBans(meta);
                callback.completed(message);
            } catch (JsonException e) {
                failed(e);
            }
        }

        /*
            https://st.yandex-team.ru/PS-3543
            {
              "data": {
                "bucket": {
                  "bucket_name": "privacy",
                  "bucket_value": {
                    "calls": 0,
                    "invites": 1,
                    "online_status": 0,
                    "private_chats": 1,
                    "search": 0
                  },
                  "version": 1586280411370482
                }
              },
              "status": "ok"
            }
         */
        private UserMessage parsePrivacyBans(final JsonObject meta) throws JsonException {
            session.logger().info("MetaResponse " + JsonType.HUMAN_READABLE.toString(meta));
            JsonMap bucket = meta.asMap().getMap("data").getMap("bucket");
            String name = bucket.getString("bucket_name");
            JsonMap values = bucket.getMap("bucket_value");

            // "search": 0 - all; 1 - contacts; 2 - nobody
            int policyInteger = values.getInt("search", 0);
            SearchPrivacy policy = SearchPrivacy.parse(policyInteger);
            if (policy == null) {
                throw new JsonException("Failed to parse unknown search policy " + policyInteger);
            }
            return new PolicyUpdateUserMessage(session.userId(), policy);
        }
    }

    private static class PolicyUpdateUserMessage extends UserMessage {
        private final SearchPrivacy searchPrivacy;

        public PolicyUpdateUserMessage(
            final String userId,
            final SearchPrivacy searchPrivacy)
        {
            super(new LongPrefix(0), userId, false);

            this.searchPrivacy = searchPrivacy;
        }

        @Override
        protected void writeGetFields(
                final Utf8JsonWriter writer,
                final Set<String> fields)
                throws IOException
        {
        }

        @Override
        public String updateRequest() {
            return "user_id:" + userId;
        }

        @Override
        public boolean writeIdField() {
            return false;
        }

        @Override
        protected void writeDocumentFields(
            final Utf8JsonWriter writer)
            throws IOException
        {
            writer.key(UserFields.SEARCH_PRIVACY.stored());
            writer.value(searchPrivacy.value());
        }

        @Override
        public String uri(String args) {
            return "/update?privacy&user-id=" + userId + args;
        }
    }
}
