package ru.yandex.search.messenger.indexer.v2org;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;

import NMessengerProtocol.Search;
import com.googlecode.protobuf.format.JsonFormat;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;

import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.HttpStatusPredicates;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.server.UpstreamStater;
import ru.yandex.http.util.server.UpstreamStaterFutureCallback;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.Utf8JsonWriter;
import ru.yandex.search.messenger.indexer.IndexableMessage;
import ru.yandex.search.messenger.indexer.Malo;
import ru.yandex.search.messenger.indexer.MaloRequest;
import ru.yandex.search.messenger.indexer.MessengerIndexHandlerBase;
import ru.yandex.search.messenger.indexer.PostRequestPart;

public class DepartmentsHandler extends MessengerIndexHandlerBase<DepartmentIndexSession, IndexableMessage> {
    private static final byte[] REQUEST_RAVNO =
        "request=".getBytes(StandardCharsets.UTF_8);

    protected final ThreadLocal<ByteArrayOutputStream> baosTls =
        ThreadLocal.<ByteArrayOutputStream>withInitial(
            () -> new ByteArrayOutputStream());
    private final String uri;
    private final HttpHost host;
    private UpstreamStater upstreamStater;

    public DepartmentsHandler(final Malo malo, final UpstreamStater producerStater) {
        super(malo, malo.config().v2OrgUserService(), producerStater);

        uri = malo.config().users().uri().getPath();
        host = malo.config().users().host();
    }

    @Override
    public UpstreamStater upstreamStater(final long metricsTimeFrame) {
        upstreamStater = new UpstreamStater(metricsTimeFrame, "meta-api-privacy");
        return upstreamStater;
    }

    @Override
    public DepartmentIndexSession indexSession(final MaloRequest request)
        throws HttpException, IOException
    {
        Long orgId = request.params().getLong("org-id");
        Long depId = request.params().getLong("dep-id");

        return new DepartmentIndexSession(request, orgId, depId);
    }

    @Override
    public DepartmentIndexSession postIndexSession(final PostRequestPart post)
        throws HttpException, IOException
    {
        Search.TDocument document = Search.TDocument.parseFrom(post.body());
        final String userId = document.getUuid();
        Long depId;
        try {
            depId = Long.parseLong(document.getUuid());
        } catch (NumberFormatException nfe) {
            throw new BadRequestException("Expecting long dep id", nfe);
        }
        post.session().logger().info("input TDocument: uuid: " + userId + " doc " + new JsonFormat().printToString(document));
        return new DepartmentIndexSession(post, document.getOrganizationId(), depId);
    }

    @Override
    public void handle(
        final DepartmentIndexSession session,
        final FutureCallback<IndexableMessage> indexCallback)
        throws HttpException, IOException
    {
        session.logger().info("Handling department");
        sendDepRequest(
            session,
            new DepartmentCallback(session, indexCallback));
    }

    public void sendDepRequest(
        final DepartmentIndexSession session,
        final FutureCallback<JsonObject> callback)
        throws HttpException, IOException
    {
        AsyncClient client = malo.chatsClient().adjust(session.session().context());

        final byte[] postData;
        final ByteArrayOutputStream baos = baosTls.get();
        baos.reset();
        try (Utf8JsonWriter writer = JsonType.NORMAL.create(baos)) {
            baos.write(REQUEST_RAVNO);
            writer.startObject();
            writer.key("method");
            writer.value("get_department");
            writer.key("params");
            writer.startObject();
            writer.key("organization_id");
            writer.value(session.orgId());
            writer.key("department_id");
            writer.value(session.depId());
            writer.endObject();
            writer.endObject();
            writer.flush();
            postData = baos.toByteArray();
        }

        session.logger().info("Deps request " + new String(postData, StandardCharsets.UTF_8));
        final BasicAsyncRequestProducerGenerator post =
            new BasicAsyncRequestProducerGenerator(
                uri,
                postData,
                ContentType.APPLICATION_FORM_URLENCODED);
        post.addHeader("X-Version", "6");
        FutureCallback<JsonObject> metaApiCallback = callback;
        if (upstreamStater != null) {
            metaApiCallback = new UpstreamStaterFutureCallback<>(callback, upstreamStater);
        }

        client.execute(
            host,
            post,
            new StatusCheckAsyncResponseConsumerFactory<>(
                HttpStatusPredicates.NON_PROTO_FATAL,
                JsonAsyncTypesafeDomConsumerFactory.INSTANCE),
            session.session().listener().createContextGeneratorFor(client),
            metaApiCallback);
    }

    private static class DepartmentCallback extends AbstractFilterFutureCallback<JsonObject, IndexableMessage> {
        private final DepartmentIndexSession session;

        public DepartmentCallback(
            final DepartmentIndexSession session,
            final FutureCallback<? super IndexableMessage> callback) {
            super(callback);

            this.session = session;
        }

        @Override
        public void completed(final JsonObject meta) {
            try {
                JsonMap map = meta.asMap();
                String status = map.getOrNull("status");
                if ("ok".equals(status)) {
                    DepartmentInfo message = parseDepartment(meta);
                    callback.completed(message);
                    session.logger().info(String.valueOf(message));
                } else if ("error".equals(status)) {
                    JsonMap data = map.getMap("data");
                    String code = data.getOrNull("code");
                    if ("department_not_found".equals(code)) {
                        session.logger().info("Department with  id: " + session.depId() + " for org " + session.orgId()
                                                  + " did not found. Will send remove request");
                        DeleteDepartmentMessage message = new DeleteDepartmentMessage(session.orgId(), session.depId());
                        callback.completed(message);
                    } else {
                        callback.failed(new Exception("Failed to handle departmnet " + JsonType.NORMAL.toString(meta)));
                    }
                }
                //callback.failed(new ServerException(HttpStatus.SC_INTERNAL_SERVER_ERROR, "this is planned"));
            } catch (JsonException e) {
                failed(e);
            }
        }

        private DepartmentInfo parseDepartment(final JsonObject meta) throws JsonException {
            session.logger().info("MetaResponse " + JsonType.HUMAN_READABLE.toString(meta));
            JsonMap data = meta.asMap().getMap("data");

            return new DepartmentInfo(data);
        }
    }
}
