package ru.yandex.search.messenger.indexer.v2org;

import java.io.IOException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.Utf8JsonWriter;
import ru.yandex.search.messenger.indexer.ChatInfo;
import ru.yandex.search.messenger.indexer.ChatMessage;
import ru.yandex.search.messenger.indexer.IndexableMessage;
import ru.yandex.search.messenger.indexer.MessengerChatsHandler;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.util.string.StringUtils;

public class OrgChatInfo extends ChatInfo {
    private static final int CHAT_ORG_LIMIT = 100;
    private final List<ChatOrgInfo> organizations;
    private final long[] orgIds;
    private final String service;

    public OrgChatInfo(final JsonMap json, final String service) throws JsonException, ParseException {
        super(json);

        this.service = service;
        JsonList orgsList = json.getList("organizations_v2");
        List<ChatOrgInfo> orgs = new ArrayList<>(orgsList.size());
        orgIds = new long[orgsList.size()];
        for (int i = 0; i < orgsList.size(); i++) {
            ChatOrgInfo orgInfo = new ChatOrgInfo(orgsList.get(i).asMap());
            orgIds[i] = orgInfo.orgId();
            orgs.add(orgInfo);
        }

        this.organizations = Collections.unmodifiableList(orgs);
    }

    public long[] orgIds() {
        return orgIds;
    }

    @Override
    public String uri(final String args) {
        return "/update?db=v2org&chat-id=" + chatId + args;
    }

    @Override
    public boolean multiMessage() {
        return true;
    }

    @Override
    public Collection<IndexableMessage> subMessages() {
        int limit =
            Math.min(organizations.size(), CHAT_ORG_LIMIT);
        List<IndexableMessage> messages =
            new ArrayList<>(limit + 1);

        for (ChatOrgInfo org: organizations) {
            messages.add(new OrgChatSubMessage(chatId, org));
            if (messages.size() > limit) {
                break;
            }
        }
        return messages;
    }

    @Override
    public String service() {
        return service;
    }

    private static class ChatOrgInfo {
        private final Long orgId;
        private final List<Long> groups;
        private final List<Long> departments;


        public ChatOrgInfo(final JsonMap map) throws JsonException {
            orgId = map.getLong("organization_id");
            JsonList groupList = map.getList("group_ids");
            List<Long> groups = new ArrayList<>(groupList.size());
            for (JsonObject groupId: groupList) {
                groups.add(groupId.asLong());
            }
            this.groups = Collections.unmodifiableList(groups);


            JsonList depsList = map.getList("department_ids");
            List<Long> departments = new ArrayList<>(depsList.size());
            for (JsonObject depId: depsList) {
                departments.add(depId.asLong());
            }
            this.departments = Collections.unmodifiableList(departments);
        }

        public Long orgId() {
            return orgId;
        }

        public List<Long> groups() {
            return groups;
        }

        public List<Long> departments() {
            return departments;
        }
    }

    private class OrgChatSubMessage extends ChatMessage {
        private final ChatOrgInfo orgInfo;
        private final LongPrefix prefix;
        public OrgChatSubMessage(final String chatId, final ChatOrgInfo orgInfo) {
            super(chatId, true);

            this.orgInfo = orgInfo;
            this.prefix = new LongPrefix(orgInfo.orgId());
        }

        @Override
        protected void writeGetFields(
            final Utf8JsonWriter writer,
            final Set<String> fields)
            throws IOException {
            for (String field : fields) {
                switch (field) {
                    case MessengerChatsHandler.CHAT_MEMBERS:
                        writer.key(MessengerChatsHandler.CHAT_MEMBERS);
                        writer.value(members);
                        break;
                    default:
                        break;
                }
            }
        }

        @Override
        public String service() {
            return service;
        }

        @Override
        public String uri(final String args) {
            return "/update?db=v2org&chat-id=" + chatId + args;
        }

        @Override
        protected void writeDocumentFields(final Utf8JsonWriter writer) throws IOException {
            writer.key(MessengerChatsHandler.CHAT_ID);
            writer.value(chatId);
            //leave
            writer.key("chat_version");
            writer.value(version);
            writer.key("chat_create_timestamp");
            writer.value(createTimestamp);
            writer.key("chat_avatar_id");
            writer.value(avatarId);
            writer.key("chat_private");
            writer.value(privateChat);
            writer.key("chat_public");
            writer.value(publicChat);
            writer.key("chat_invite_hash");
            writer.value(inviteHash);
            writer.key("chat_avatar");
            writer.value(avatar);
            writer.key("chat_name");
            writer.value(name);
            writer.key(ru.yandex.ps.search.messenger.ChatFields.ALIAS.stored());
            writer.value(alias);
            writer.key("chat_icon_id");
            writer.value(iconId);
            writer.key("chat_description");
            writer.value(description);

//            writer.key(CHAT_MEMBERS);
//            writer.value(StringUtils.join(members, '\n'));

            writer.key("chat_permissions_users");
            writer.value(StringUtils.join(usersPermissions, '\n'));
            writer.key("chat_permissions_groups");
            writer.value(StringUtils.join(groupsPermissions, '\n'));
            writer.key("chat_permissions_departments");
            writer.value(StringUtils.join(departmentsPermissions, '\n'));

            writer.key("chat_roles_admin");
            writer.value(StringUtils.join(adminRoles, '\n'));

            writer.key("chat_org_id");
            writer.value(orgInfo.orgId());


            writer.key("chat_groups");
            writer.value(
                orgInfo.groups().stream().map(String::valueOf).collect(Collectors.joining("\n")));

            writer.key("chat_departments");
            writer.value(
                orgInfo.departments().stream().map(String::valueOf).collect(Collectors.joining("\n")));

            if (geoId != null) {
                writer.key("chat_geo_id");
                writer.value(geoId);
            }

            writer.key("chat_geo_type");
            writer.value(geoType);

            writer.key("chat_entity_id");
            writer.value(entityId);

            writer.key("chat_subservice");
            writer.value(subservice);

            writer.key("chat_namespace");
            writer.value(namespace);

            writer.key("chat_parent_url");
            writer.value(parentUrl);

            writer.key("chat_show_on_morda");
            writer.value(showOnMorda);

            writer.key(ru.yandex.ps.search.messenger.ChatFields.CHANNEL.stored());
            writer.value(channel);

            //whole blob
            writer.key("chat_data");
            writer.value(JsonType.NORMAL.toString(json));
        }

        @Override
        public String prefixHash() {
            return Long.toString(prefix.hash());
        }

        @Override
        public String prefix() {
            return prefix.toString();
        }
    }
}
