package ru.yandex.search.messenger.indexer.v2org;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;

import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.Utf8JsonWriter;
import ru.yandex.search.messenger.indexer.ChatMembersMessage;
import ru.yandex.search.messenger.indexer.IndexableMessage;
import ru.yandex.search.messenger.indexer.PvpChatPredicate;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.util.string.StringUtils;

public class OrgChatMembersMessage extends ChatMembersMessage {
    private final List<String> removeMembers;
    private final List<String> addMembers;
    private final long version;
    private final String service;
    private final boolean reindex;
    private final int memberCount;
    private final Collection<IndexableMessage> subMessages;

    public OrgChatMembersMessage(
        final String chatId,
        final long[] orgs,
        final String service,
        final JsonMap json,
        final boolean reindex,
        final int prevMemberCount)
        throws JsonException
    {
        super(chatId);
        this.service = service;
        this.reindex = reindex;
        JsonList removedList = json.getListOrNull("removed");
        int memberCount = prevMemberCount;
        if (removedList != null && removedList.size() > 0) {
            removeMembers = new ArrayList<>(removedList.size());
            for (JsonObject o: removedList) {
                removeMembers.add(o.asString());
            }
            memberCount -= removeMembers.size();
        } else {
            removeMembers = null;
        }

        JsonList addedList = json.getListOrNull("added");
        if (addedList != null && addedList.size() > 0) {
            addMembers = new ArrayList<>(addedList.size());
            for (JsonObject o: addedList) {
                addMembers.add(o.asString());
            }
            memberCount += addMembers.size();
        } else {
            addMembers = null;
        }
        this.memberCount = memberCount;
        version = json.getLong("version", 0L);

        subMessages = new ArrayList<>();
        for (int i = 0; i < orgs.length; i++) {
            subMessages.add(
                new OrgChatMembersInfoMessage(
                    chatId,
                    orgs[i],
                    removeMembers,
                    addMembers,
                    version,
                    reindex,
                    memberCount));
            if (addMembers != null) {
                for (String userId: addMembers) {
                    subMessages.add(
                        new PerUserOrgChatMembersMessage(chatId, userId, service, orgs[i], false));
                }
            }
            if (removeMembers != null) {
                for (String userId: removeMembers) {
                    subMessages.add(
                        new PerUserOrgChatMembersMessage(chatId, userId, service, orgs[i], true));
                }
            }
        }
    }

    @Override
    public String service() {
        return service;
    }

    public List<String> removeMembers() {
        return removeMembers;
    }

    public List<String> addMembers() {
        return addMembers;
    }

    @Override
    public boolean multiMessage() {
        return true;
    }

    @Override
    public Collection<IndexableMessage> subMessages() {
        return subMessages;
    }

    @Override
    protected void writeDocumentFields(final Utf8JsonWriter writer)
        throws IOException
    {
        throw new IOException(
            "writeDocumentFields called on root multimessage");
    }

    @Override
    protected void writeGetFields(
        final Utf8JsonWriter writer,
        final Set<String> fields)
        throws IOException
    {
    }

    @Override
    public String uri(final String args) {
        return "/update?batch&db=v2org&chat-members&chat-id=" + chatId + args;
    }

    private class OrgChatMembersInfoMessage extends ChatMembersMessage {
        private final List<String> removeMembers;
        private final List<String> addMembers;
        private final long version;
        private final boolean reindex;
        private final int memberCount;
        private final LongPrefix prefix;

        OrgChatMembersInfoMessage(
            final String chatId,
            final long orgId,
            final List<String> removeMembers,
            final List<String> addMembers,
            final long version,
            final boolean reindex,
            final int memberCount)
        {
            super(chatId);
            this.prefix = new LongPrefix(orgId);
            this.removeMembers = removeMembers;
            this.addMembers = addMembers;
            this.version = version;
            this.reindex = reindex;
            this.memberCount = memberCount;
        }

        @Override
        public String service() {
            return service;
        }

        @Override
        protected void writeDocumentFields(final Utf8JsonWriter writer)
            throws IOException
        {
            //update only if stored version less than new
            /*
            "chat_members_version":{
                "max":[
                    {"get":["chat_members_version"]},
                    VERSION
                ]
            }
            */
            JsonObject checkVersion;
            if (reindex) {
                checkVersion = new JsonString("1");
            } else {
                checkVersion =
                    jsonFunction(
                        "gt",
                        Long.toString(version),
                        jsonFunction(
                            "get",
                            "chat_members_version"));
            }

            //members jopa
            //if (stored.membersVersion < membersVersion) {
            //    members.addAll(addedMembers);
            //    members.removeAll(removedMembers);
            //}
            if (!chatId.startsWith("1/")) {
                if (reindex) {
                    writer.key("chat_members");
                    if (addMembers != null) {
                        writer.value(StringUtils.join(addMembers, '\n'));
                    } else {
                        writer.value("");
                    }
                } else if (addMembers != null || removeMembers != null) {
                    writer.key("chat_members");
                    JsonObject membersFunc = jsonFunction("get", "chat_members");
                    if (removeMembers != null) {
                        membersFunc =
                            jsonFunction(
                                "remove_set",
                                StringUtils.join(removeMembers, '\n'),
                                membersFunc);
                    }
                    if (addMembers != null) {
                        membersFunc =
                            jsonFunction(
                                "make_set",
                                membersFunc,
                                StringUtils.join(addMembers, '\n'));
                    }

                    jsonFunction(
                        "set",
                        membersFunc,
                        checkVersion)
                        .writeValue(writer);

                }
            } else {
                writer.key("chat_members");
                writer.value("");
            }

            if (PvpChatPredicate.INSTANCE.test(chatId)) {
                // on pvp chat create, we have only chat members notify; pvp chats always have 0 namespace
                writer.key(ru.yandex.ps.search.messenger.ChatFields.NAMESPACE.stored());
                writer.value(0);
            }

            // order of keys in json makes sense, we should check version for members
            // before with set one
            writer.key("chat_members_version");
            jsonFunction(
                "set",
                Long.toString(version),
                checkVersion)
                .writeValue(writer);

            writer.key("chat_members_count");
            writer.value(memberCount);

            writer.key("chat_id");
            writer.value(chatId);
        }

        @Override
        protected void writeGetFields(
            final Utf8JsonWriter writer,
            final Set<String> fields)
            throws IOException
        {
        }

        @Override
        public String uri(final String args) {
            return "/update?db=v2org&chat-members-info&chat-id=" + chatId + "&org-id=" + prefix.prefix() + args;
        }

        @Override
        public String prefixHash() {
            return Long.toString(prefix.hash());
        }

        @Override
        public String prefix() {
            return prefix.toStringFast();
        }
    }
}
