package ru.yandex.search.messenger.indexer.v2org;


import java.io.IOException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.Utf8JsonWriter;
import ru.yandex.search.messenger.indexer.IndexableMessage;
import ru.yandex.search.messenger.indexer.MessengerUsersHandler;
import ru.yandex.search.messenger.indexer.UserInfo;
import ru.yandex.util.string.StringUtils;

public class OrgUserInfo extends UserInfo {
    private final List<UserOrgInfo> v2Orgs;
    private final String service;
    private final UserInfo noV2Info;
    public OrgUserInfo(final JsonMap json, final String service) throws JsonException, ParseException {
        super(json);

        this.service = service;
        JsonList orgsList = json.getList("organizations_v2");
        List<UserOrgInfo> orgs = new ArrayList<>(orgsList.size());
        for (JsonObject item: orgsList) {
            orgs.add(new UserOrgInfo(item.asMap()));
        }

        this.noV2Info = new UserInfo(json);
        this.v2Orgs = Collections.unmodifiableList(orgs);
    }

    @Override
    public Collection<IndexableMessage> subMessages() {
        int limit =
            Math.min(v2Orgs.size(), MessengerUsersHandler.USER_ORGANIZATIONS_LIMIT);
        List<IndexableMessage> messages =
            new ArrayList<>(limit + 1);
        for (UserOrgInfo org : v2Orgs) {
            messages.add(new OrgSubMessage(org));
            if (messages.size() > limit) {
                break;
            }
        }
        messages.addAll(noV2Info.subMessages());
        return messages;
    }

    @Override
    public String service() {
        return service;
    }

    private static class UserOrgInfo {
        private final Long orgId;
        private final String displayName;
        private final List<Long> groups;
        private final List<Long> departments;
        private final boolean deleted;

        public UserOrgInfo(final JsonMap map) throws JsonException {
            this.orgId = map.getLong("organization_id");
            this.deleted = map.getBoolean("deleted", false);
            List<JsonObject> groupList = map.getListOrNull("group_ids");
            if (groupList == null) {
                groupList = Collections.emptyList();
            }
            List<Long> groups = new ArrayList<>(groupList.size());
            for (JsonObject groupId: groupList) {
                groups.add(groupId.asLong());
            }
            this.groups = Collections.unmodifiableList(groups);

            List<JsonObject> depsList = map.getListOrNull("department_ids");

            if (depsList == null) {
                depsList = Collections.emptyList();
            }
            List<Long> departments = new ArrayList<>(depsList.size());
            for (JsonObject depId: depsList) {
                departments.add(depId.asLong());
            }
            this.departments = Collections.unmodifiableList(departments);
            this.displayName = map.getOrNull("display_name");
        }

        public Long orgId() {
            return orgId;
        }

        public String displayName() {
            return displayName;
        }

        public List<Long> groups() {
            return groups;
        }

        public List<Long> departments() {
            return departments;
        }

        public boolean deleted() {
            return deleted;
        }
    }

    private class OrgSubMessage extends SubMessage {
        private final UserOrgInfo orgInfo;
        public OrgSubMessage(final UserOrgInfo orgInfo) {
            super(orgInfo.orgId());

            this.orgInfo = orgInfo;
        }

        @Override
        public String service() {
            return service;
        }

        @Override
        public String uri(final String args) {
            return "/update?db=v2org&org-id=" + orgInfo.orgId() + "&user-id=" + userId + args;
        }

//        @Override
//        public String id() {
//            return "user_" + userId + '@' + prefix.toString();
//        }

        @Override
        protected void writeDocumentFields(final Utf8JsonWriter writer) throws IOException {
            writer.key(MessengerUsersHandler.USER_ID);
            writer.value(userId);
            //leave
            writer.key("user_version");
            writer.value(version);

            writer.key("user_display_name");
            String displayName;
            if (orgInfo.displayName() == null) {
                displayName = getOrDefault(displayNames, defaultDisplayName);
            } else {
                displayName = orgInfo.displayName();
            }
            writer.value(displayName);

            writer.key("user_gender");
            writer.value(gender);

            writer.key("user_affiliation");
            writer.value(affiliation);

            writer.key("user_affiliation_nn");
            if (affiliation != null) {
                writer.value(affiliation);
            } else {
                writer.value("na");
            }

            String nickname =
                getOrDefault(nicknames, defaultNickname);
            writer.key("user_nickname");
            writer.value(nickname);

            String position =
                getOrDefault(positions, defaultPosition);
            writer.key("user_position");
            writer.value(position);

            String website =
                getOrDefault(websites, defaultWebsite);
            writer.key("user_website");
            writer.value(website);

            String department =
                getOrDefault(departments, defaultDepartmentName);
            writer.key("user_department_name");
            writer.value(department);

            writer.key("user_is_robot");
            writer.value(isRobot);

            writer.key(ru.yandex.ps.search.messenger.UserFields.DISPLAY_RESTRICTED.stored());
            writer.value(isDisplayRestricted);

            writer.key("user_is_dismissed");
            writer.value(isDismissed || orgInfo.deleted());

            writer.key("user_is_homeworker");
            writer.value(isHomeworker);

            writer.key("user_status");
            writer.value(StringUtils.join(status, '\n'));

            if (organizations.size() > 0) {
                writer.key("user_organizations");
                StringBuilder sb = new StringBuilder();
                String sep = "";
                for (Long org : organizations) {
                    sb.append(sep);
                    sep = "\n";
                    sb.append(org);
                }
                writer.value(new String(sb));
            }

            writer.key("user_org_id");
//                if (orgId == 0) {
//                    writer.value(defaultOrgId);
//                } else {
            writer.value(orgInfo.orgId());
//                }

            writer.key(MessengerUsersHandler.USER_UID);
            writer.value(uid);

            writer.key("user_has_service");
            if (service != null) {
                writer.value(true);
                writer.key("user_service");
                writer.value(service);
            } else {
                writer.value(false);
            }
            //whole blob
            writer.key("user_data");
            writer.value(JsonType.NORMAL.toString(json));
        }
    }
}
