package ru.yandex.search.messenger.proxy;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.Weigher;
import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.function.BasicGenericConsumer;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.parser.StackContentHandler;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.messenger.proxy.config.ImmutableMoxyConfig;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.proxy.universal.PlainUniversalSearchProxyRequestContext;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.util.string.StringUtils;

public class CMNTInfoHandler implements ProxyRequestHandler {
    private static final int OBJ_SIZE = 64;
    private static final int DEFAULT_WEIGHT = 256 * 1024;
    private static final long MAX_CACHE_SIZE = 1024 * 1024 * 1024;
    private static final String GET = "get";
    private static final String PREFIX = "prefix";
    private static final String SERVICE = "service";
    private static final String ZERO = "0";
    private static final String TEXT = "text";
    private static final String HITS_ARRAY = "hitsArray";
    private static final String CHAT_DATA = "chat_data";
    private static final String CHAT_TEASER_DATA = "chat_teaser_data";
    private static final JsonObject FAKE_RESPONSE = JsonNull.INSTANCE;

    private static final CollectionParser<
        String,
        Set<String>,
        Exception>
        SET_PARSER = new CollectionParser<>(
            NonEmptyValidator.INSTANCE,
            LinkedHashSet::new);
    private static final boolean ALLOW_LAGGING_HOSTS = true;

    private final Moxy moxy;
    private final String chatsService;
    private final Cache<String, ChatInfo> chatsCache;
    private final TimeFrameQueue<Long> cmntEntitiesCount;
    private final TimeFrameQueue<Long> cmntCacheHits;

    public CMNTInfoHandler(
        final Moxy moxy,
        final TimeFrameQueue<Long> cmntEntitiesCount,
        final TimeFrameQueue<Long> cmntCacheHits)
    {
        this.moxy = moxy;
        this.cmntEntitiesCount = cmntEntitiesCount;
        this.cmntCacheHits = cmntCacheHits;
        chatsService = moxy.config().chatsService();
        chatsCache = CacheBuilder.newBuilder()
            .maximumWeight(MAX_CACHE_SIZE)
            .expireAfterWrite(
                moxy.config().cmntCacheTime(),
                TimeUnit.MILLISECONDS)
            .weigher(new ChatInfoWeigher())
            .build();
    }

    @Override
    public void handle(final ProxySession session) throws HttpException {
        final CMNTInfoRequestContext cmntInfoContext =
            new CMNTInfoRequestContext(session, moxy.config());

        cmntInfoContext.resolveCached();

        if (!cmntInfoContext.empty()) {
            sendRequest(
                cmntInfoContext,
                session,
                new CMNTInfoCallback(session, cmntInfoContext));
        } else {
            new CMNTInfoCallback(session, cmntInfoContext)
                .completed(FAKE_RESPONSE);
        }
    }

    public void sendRequest(
        final CMNTInfoRequestContext context,
        final ProxySession session,
        final FutureCallback<JsonObject> callback)
        throws HttpException
    {
        AsyncClient client = moxy.searchClient().adjust(session.context());
        UniversalSearchProxyRequestContext requestContext =
            new PlainUniversalSearchProxyRequestContext(
                context.user(),
                null,
                context.allowLaggingHosts,
                client,
                session.logger());
        QueryConstructor query = context.query();
        query.append(SERVICE, context.service());
        moxy.sequentialRequest(
            session,
            requestContext,
            new BasicAsyncRequestProducerGenerator(query.toString()),
            context.failoverDelay,
            context.localityShuffle,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.listener().createContextGeneratorFor(client),
            callback);
    }

    @Override
    public String toString() {
        return "Commentator chats info handler: "
            + "https://wiki.yandex-team.ru/ps/documentation/"
            + "moxy#cmntinfo";
    }

    private class CMNTInfoRequestContext {
        private final Collection<String> get;
        private final boolean localityShuffle;
        private final boolean allowLaggingHosts;
        private final long failoverDelay;
        private final Collection<String> entityIds;
        private final Collection<String> chatIds;
        private final int serviceId;
        private List<ChatInfo> chatInfos = null;

        CMNTInfoRequestContext(
            final ProxySession session,
            final ImmutableMoxyConfig config)
            throws HttpException
        {
            this.get = session.params().getAll(
                GET,
                Collections.emptySet(),
                SET_PARSER);
            this.localityShuffle = session.params().getBoolean(
                "locality-shuffle",
                config.topPostsLocalityShuffle());
            this.failoverDelay = session.params().getLong(
                "failover-delay",
                config.topPostsFailoverDelay());
            this.allowLaggingHosts = session.params().getBoolean(
                "allow-lagging-hosts",
                ALLOW_LAGGING_HOSTS);
            this.entityIds = session.params().getAll(
                "entity-id",
                Collections.emptySet(),
                SET_PARSER);
            this.chatIds = session.params().getAll(
                "chat-id",
                Collections.emptySet(),
                SET_PARSER);
            if (!entityIds.isEmpty()) {
                this.serviceId = session.params().getInt("service-id");
            } else {
                this.serviceId = -1;
            }
            for (String chatId: chatIds) {
                if (!chatId.startsWith("0/14/")) {
                    throw new BadRequestException(
                        "permission denied for chat-id: " + chatId);
                }
            }
            if (entityIds.isEmpty() && chatIds.isEmpty()) {
                throw new BadRequestException(
                    "no chat-id nor entity-id is set");
            }
        }

        public void addChatInfo(final ChatInfo info) {
            if (chatInfos == null) {
                chatInfos = new ArrayList<>();
            }
            chatInfos.add(info);
        }

        public List<ChatInfo> chatInfos() {
            if (chatInfos == null) {
                return Collections.emptyList();
            }
            return chatInfos;
        }

        public String cacheKey(final ChatInfo info) {
            if (info.entityId != null) {
                return entityCacheKey(info.entityId);
            } else {
                return chatCacheKey(info.chatId);
            }
        }

        public String entityCacheKey(final String entityId) {
            return "entity_id:" + entityId
                + '@' + Integer.toString(serviceId);
        }

        public String chatCacheKey(final String chatId) {
            return "chat_id:" + chatId;
        }

        public void resolveCached() {
            cmntEntitiesCount.accept(
                (long) (chatIds.size() + entityIds.size()));
            long cached = 0;
            if (!entityIds.isEmpty()) {
                Iterator<String> iter = entityIds.iterator();
                while (iter.hasNext()) {
                    String entityId = iter.next();
                    String key = entityCacheKey(entityId);
                    ChatInfo info = chatsCache.getIfPresent(key);
                    if (info != null) {
                        addChatInfo(info);
                        iter.remove();
                        cached++;
                    }
                }
            }
            if (!chatIds.isEmpty()) {
                Iterator<String> iter = entityIds.iterator();
                while (iter.hasNext()) {
                    String chatId = iter.next();
                    String key = chatCacheKey(chatId);
                    ChatInfo info = chatsCache.getIfPresent(key);
                    if (info != null) {
                        addChatInfo(info);
                        iter.remove();
                        cached++;
                    }
                }
            }
            cmntCacheHits.accept(cached);
        }

        public boolean empty() {
            return entityIds.isEmpty() && chatIds.isEmpty();
        }

        public Collection<String> get() {
            return get;
        }

        public Collection<String> entityIds() {
            return entityIds;
        }

        public Collection<String> chatIds() {
            return chatIds;
        }

        public int serviceId() {
            return serviceId;
        }

        protected String service() {
            return chatsService;
        }

        protected QueryConstructor query() throws HttpException {
            return createQuery();
        }

        protected QueryConstructor createQuery() throws HttpException {
            QueryConstructor query = new QueryConstructor(
                "/search-cmnt-info?json-type=dollar&IO_PRIO=0"
                + "&get=chat_id,chat_name,chat_entity_id,chat_subservice"
                + ",chat_namespace,chat_total_message_count"
                + ",chat_hidden_message_count,chat_message_count"
                + ",chat_total_indexed_message_count,chat_parent_url");
            query.append(
                TEXT,
                searchText());
            query.append(PREFIX, ZERO);
            if (get.size() > 0) {
                query.append(GET, StringUtils.join(get, ','));
            }
            return query;
        }

        private String searchText() {
            final StringBuilder sb = new StringBuilder();
            if (!entityIds.isEmpty()) {
                sb.append("chat_entity_id:(");
                joinIdStrings(sb, entityIds);
                sb.append(") AND chat_subservice:");
                sb.append(serviceId);
            }
            if (!chatIds.isEmpty()) {
                if (sb.length() > 0) {
                    sb.append(" OR ");
                }
                sb.append("chat_id:(");
                joinIdStrings(sb, chatIds);
                sb.append(')');
            }
            return new String(sb);
        }

        protected User user() {
            return new User(chatsService, new LongPrefix(0));
        }

        private void joinIdStrings(
            final StringBuilder sb,
            final Collection<String> ids)
        {
            String sep = "";
            for (String id: ids) {
                sb.append(sep);
                sb.append(SearchRequestText.fullEscape(id, false));
                sep = " ";
            }
        }
    }

    private class CMNTInfoCallback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final JsonType jsonType;
        private final CMNTInfoRequestContext context;

        CMNTInfoCallback(
            final ProxySession session,
            final CMNTInfoRequestContext context)
            throws HttpException
        {
            super(session);
            this.context = context;
            this.jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
        }

        @Override
        public void completed(final JsonObject result) {
            long startTime = System.currentTimeMillis();

            final BasicGenericConsumer<JsonObject, JsonException> consumer =
                new BasicGenericConsumer<>();
            final JsonParser jsonParser = new JsonParser(
                new StackContentHandler(
                    new TypesafeValueContentHandler(
                        consumer)));

            StringBuilderWriter sbw = new StringBuilderWriter();

            if (result != FAKE_RESPONSE) {
                try {
                    JsonMap chatsResult = result.asMap();
                    JsonList hits = chatsResult.getList(HITS_ARRAY);
                    for (JsonObject obj: hits) {
                        JsonMap hit = obj.asMap();
                        String id = hit.getString("chat_entity_id", null);
                        String entityId = null;
                        String chatId = null;
                        if (id != null && context.entityIds.contains(id)) {
                            entityId = id;
                        } else {
                            id = hit.getString("chat_id", null);
                            if (id == null) {
                                continue;
                            }
                            if (context.chatIds.contains(id)) {
                                chatId = id;
                            } else {
                                continue;
                            }
                        }
                        jsonReformat(hit, jsonParser, consumer);
                        ChatInfo info = new ChatInfo(
                            entityId,
                            context.serviceId,
                            chatId,
                            hit);
                        context.addChatInfo(info);
                        chatsCache.put(context.cacheKey(info), info);
                    }
                } catch (JsonException e) {
                    session.logger().log(Level.SEVERE, "JsonParse exception", e);
                    failed(e);
                    return;
                } finally {
                    session.logger().info("Callback parse time: "
                        + (System.currentTimeMillis() - startTime));
                }
            }
            startTime = System.currentTimeMillis();

            try (JsonWriter writer = jsonType.create(sbw)) {
                writer.startObject();
                for (ChatInfo info: context.chatInfos()) {
                    writer.key(info.id());
                    writer.value(info.value());
                }
                writer.endObject();
                session.logger().info(
                    "Total chats: " + context.chatInfos().size()
                    + ", time: " + (System.currentTimeMillis() - startTime));
                startTime = System.currentTimeMillis();

                HttpEntity entity =
                    new NStringEntity(
                        sbw.toString(),
                        ContentType.APPLICATION_JSON.withCharset(
                            session.acceptedCharset()));
                session.logger().info("Generate response time: "
                    + (System.currentTimeMillis() - startTime));
                session.response(HttpStatus.SC_OK, entity);
            } catch (IOException e) {
                session.logger().log(Level.SEVERE, "JsonWrite exception", e);
                failed(e);
            } finally {
                session.logger().info("Response generate time: "
                    + (System.currentTimeMillis() - startTime));
            }
        }

        private void jsonReformat(
            final JsonMap doc,
            final JsonParser jsonParser,
            final BasicGenericConsumer<JsonObject, JsonException> consumer)
        {
            try {
                String jsonString = doc.get(CHAT_DATA).asStringOrNull();
                if (jsonString != null) {
                    jsonParser.parse(jsonString);
                    JsonObject obj = consumer.get();
                    doc.put(CHAT_DATA, obj);
                }
                jsonString = doc.get(CHAT_TEASER_DATA).asStringOrNull();
                if (jsonString != null) {
                    jsonParser.parse(jsonString);
                    JsonObject obj = consumer.get();
                    doc.put(CHAT_TEASER_DATA, obj);
                }
            } catch (JsonException e) { // skip, obj is null
            }
        }
    }

    private static final class ChatInfo {
        private final String entityId;
        private final int serviceId;
        private final String chatId;
        private JsonObject value;
        private int weight = -1;

        ChatInfo(
            final String entityId,
            final int serviceId,
            final String chatId,
            final JsonObject value)
        {
            this.entityId = entityId;
            this.serviceId = serviceId;
            this.chatId = chatId;
            this.value = value;
        }

        public String id() {
            if (entityId != null) {
                return entityId;
            }
            return chatId;
        }

        public JsonObject value() {
            return value;
        }

        public int weight() {
            if (weight == -1) {
                calcWeight();
            }
            return weight;
        }

        public void calcWeight() {
            int weight = 100;
            if (entityId != null) {
                weight += entityId.length() << 1;
            }
            if (chatId != null) {
                weight += chatId.length() << 1;
            }
            try {
                weight += jsonWeight(value);
            } catch (JsonException e) {
                e.printStackTrace();
                weight += DEFAULT_WEIGHT;
            }
            this.weight = weight;
        }

        private int jsonWeight(final JsonObject obj) throws JsonException {
            switch(obj.type()) {
                case NULL:
                    return OBJ_SIZE;
                case BOOLEAN:
                    return OBJ_SIZE + 1;
                case LONG:
                    return OBJ_SIZE + 8;
                case DOUBLE:
                    return OBJ_SIZE + 8;
                case STRING:
                    return (obj.asString().length() << 1) + OBJ_SIZE;
                case LIST: {
                    int sum = OBJ_SIZE;
                    for (JsonObject o: obj.asList()) {
                        sum += jsonWeight(o);
                    }
                    return sum;
                }
                case MAP: {
                    int sum = OBJ_SIZE;
                    for (Map.Entry<String, JsonObject> e:
                        obj.asMap().entrySet())
                    {
                        sum += OBJ_SIZE + e.getKey().length() << 1;
                        sum += jsonWeight(e.getValue());
                    }
                    return sum;
                }
                default:
                    return OBJ_SIZE;
            }
        }
    }

    private static class ChatInfoWeigher
        implements Weigher<String, ChatInfo>
    {
        @Override
        public int weigh(final String key, final ChatInfo value) {
            return key.length() * 2 + 100
                + value.weight();
        }
    }
}

