package ru.yandex.search.messenger.proxy;

import java.util.Collections;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.function.Supplier;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.ErrorSuppressingFutureCallback;
import ru.yandex.http.util.IdempotentFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.search.messenger.proxy.suggest.AcceptAllUsersFilter;
import ru.yandex.search.messenger.proxy.suggest.BasicSuggestRequestContext;
import ru.yandex.search.messenger.proxy.suggest.HtmlHighlighter;
import ru.yandex.search.messenger.proxy.suggest.SuggestItem;
import ru.yandex.search.messenger.proxy.suggest.SuggestPrinter;
import ru.yandex.search.messenger.proxy.suggest.SuggestResult;
import ru.yandex.search.messenger.proxy.suggest.SuggestType;
import ru.yandex.search.messenger.proxy.suggest.rules.MergeResultsRule;
import ru.yandex.search.messenger.proxy.suggest.rules.ResolveChatRule;
import ru.yandex.search.messenger.proxy.suggest.rules.ResolveContactsUsersRule;
import ru.yandex.search.messenger.proxy.suggest.rules.UsersSuggestRule;
import ru.yandex.search.messenger.proxy.suggest.rules.providers.SuggestRequestContextRequestAllFilterProvider;
import ru.yandex.search.messenger.proxy.suggest.rules.providers.SuggestRequestContextRequestChatsFilterProvider;
import ru.yandex.search.messenger.proxy.suggest.rules.providers.SuggestRequestContextRequestsChatsFilterProvider;
import ru.yandex.search.rules.pure.ChainedSearchRule;
import ru.yandex.search.rules.pure.SearchRule;
import ru.yandex.search.rules.pure.TranslitRule;
import ru.yandex.search.rules.pure.TranslitRule2;

public class ChatMembersHandler implements ProxyRequestHandler {
    private static final long TIMEOUT_OVERHEAD = 10L;
    private static final long MIN_TIMEOUT = 50L;
    private static final TranslitRule.TableSelector TRANSLIT_SELECTOR =
        new TranslitSelector123();
    private static final Supplier<List<SuggestItem>> NULL_SUPPLIER =
        () -> null;
    private final Timer timer = new Timer("Chat-Members-Timer", true);
    private final SearchRule<BasicSuggestRequestContext, List<SuggestItem>> userRule;
    private final SearchRule<BasicSuggestRequestContext, List<SuggestItem>> contactsRule;

    private final Moxy proxy;

    public ChatMembersHandler(final Moxy proxy) {
        this.proxy = proxy;
        this.userRule =
            new ResolveChatRule<>(
                new ChainedSearchRule<>(
                    new TranslitRule<>(
                        new ChainedSearchRule<>(
                            new MergeResultsRule<>(
                                new ChainedSearchRule<>(
                                    new UsersSuggestRule<>(proxy, SuggestType.USERS, false, false),
                                    (input, request) ->
                                        new SuggestRequestContextRequestAllFilterProvider(
                                            input.suggestRequestContext(),
                                            request,
                                            AcceptAllUsersFilter.INSTANCE,
                                            input.chatsFilter()))),
                            (input, requests) ->
                                new SuggestRequestContextRequestsChatsFilterProvider(
                                    input.suggestRequestContext(),
                                    input.chatsFilter(),
                                    requests)),
                        true,
                        TRANSLIT_SELECTOR),
                    (input, filter) ->
                        new SuggestRequestContextRequestChatsFilterProvider(
                            input.suggestRequestContext(),
                            input.request(),
                            filter)));
        this.contactsRule =
            new ResolveChatRule<>(
                new ChainedSearchRule<>(
                    new TranslitRule<>(
                        new ChainedSearchRule<>(
                            new MergeResultsRule<>(
                                new ChainedSearchRule<>(
                                    new ResolveContactsUsersRule<>(
                                        proxy,
                                        new ChainedSearchRule<>(
                                            new UsersSuggestRule<>(proxy, SuggestType.CONTACTS, false, true),
                                            (input, filter) ->
                                                new SuggestRequestContextRequestAllFilterProvider(
                                                    input.suggestRequestContext(),
                                                    input.request(),
                                                    filter,
                                                    input.chatsFilter()))),
                                    (input, request) ->
                                        new SuggestRequestContextRequestAllFilterProvider(
                                            input.suggestRequestContext(),
                                            request,
                                            AcceptAllUsersFilter.INSTANCE,
                                            input.chatsFilter()))),
                            (input, requests) ->
                                new SuggestRequestContextRequestsChatsFilterProvider(
                                    input.suggestRequestContext(),
                                    input.chatsFilter(),
                                    requests)),
                        true,
                        TRANSLIT_SELECTOR),
                    (input, filter) ->
                        new SuggestRequestContextRequestChatsFilterProvider(
                            input.suggestRequestContext(),
                            input.request(),
                            filter)));
    }

    @Override
    public void handle(final ProxySession session) throws HttpException {
        session.params().replace(
            "suggest-types",
            SuggestType.USERS.toString() + "," + SuggestType.CONTACTS.toString());
        BasicSuggestRequestContext context =
            new BasicSuggestRequestContext(proxy, session);
        Map<SuggestType, List<SuggestItem>> responses =
            new EnumMap<>(SuggestType.class);
        FutureCallback<Object> printer =
            new IdempotentFutureCallback<>(
                new SuggestResultCallback(
                    new SuggestPrinter(context, HtmlHighlighter.INSTANCE),
                    responses));
        MultiFutureCallback<Object> callback =
            new MultiFutureCallback<>(printer);
        userRule.execute(
            context,
            new ResponseStoringCallback(
                callback.newCallback(),
                context,
                SuggestType.USERS,
                responses));
        contactsRule.execute(
            context,
            new ResponseStoringCallback(
                callback.newCallback(),
                context,
                SuggestType.CONTACTS,
                responses));
        callback.done();
        Long timeoutBoxed = context.timeout();
        if (timeoutBoxed != null) {
            long now = System.currentTimeMillis();
            long timeElapsed = now - session.connection().requestStartTime();
            long timeout = timeoutBoxed - timeElapsed;
            timeout = Math.max(timeout - TIMEOUT_OVERHEAD, MIN_TIMEOUT);
            timer.schedule(new TimeoutTask(printer, context.logger()), timeout);
        }
    }

    private static class ResponseStoringCallback
        extends ErrorSuppressingFutureCallback<List<SuggestItem>>
    {
        private final BasicSuggestRequestContext context;
        private final SuggestType type;
        private final Map<SuggestType, List<SuggestItem>> responses;

        // CSOFF: ParameterNumber
        ResponseStoringCallback(
            final FutureCallback<Object> callback,
            final BasicSuggestRequestContext context,
            final SuggestType type,
            final Map<SuggestType, List<SuggestItem>> responses)
        {
            super(callback, NULL_SUPPLIER);
            this.context = context;
            this.type = type;
            this.responses = responses;
        }
        // CSON: ParameterNumber

        @Override
        public void failed(final Exception e) {
            context.logger().log(
                Level.WARNING,
                "Failed to get results for chat members " + type,
                e);
            super.failed(e);
        }

        @Override
        public void completed(final List<SuggestItem> response) {
            if (response != null) {
                if (context.debug()) {
                    context.logger().info(
                        "Results for chat members " + type + ": " + response);
                } else {
                    context.logger().info(
                        "Got " + response.size()
                            + " results for chat members " + type);
                }
                synchronized (responses) {
                    List<SuggestItem> alreadyFound =
                        responses.get(type);
                    if (alreadyFound != null) {
                        alreadyFound.addAll(response);
                        Collections.sort(alreadyFound);
                    } else {
                        responses.put(type, response);
                    }
                }
            }
            callback.completed(null);
        }
    }

    private static class SuggestResultCallback
        extends AbstractFilterFutureCallback<Object, SuggestResult>
    {
        private final Map<SuggestType, List<SuggestItem>> responses;

        SuggestResultCallback(
            final FutureCallback<? super SuggestResult> callback,
            final Map<SuggestType, List<SuggestItem>> responses)
        {
            super(callback);
            this.responses = responses;
        }

        @Override
        public void completed(final Object response) {
            SuggestResult result;
            synchronized (responses) {
                result = new SuggestResult(responses);
            }
            callback.completed(result);
        }
    }

    private static class TimeoutTask extends TimerTask {
        private final FutureCallback<?> callback;
        private final Logger logger;

        TimeoutTask(
            final FutureCallback<?> callback,
            final Logger logger)
        {
            this.callback = callback;
            this.logger = logger;
        }

        @Override
        public void run() {
            logger.info("Chat members handler timeout");
            callback.completed(null);
        }
    }

    private static final class TranslitSelector123
        implements TranslitRule.TableSelector, TranslitRule2.TableSelector
    {
        @Override
        public boolean useTable(final String request, final String table) {
            return !table.startsWith("translit") || request.length() >= 2;
        }
    }
}
