package ru.yandex.search.messenger.proxy;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.parser.searchmap.SearchMapHost;
import ru.yandex.parser.searchmap.SearchMapShard;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.StringPrefix;

public class ContactsOnlySearchPrivacyFilter {
    private final Moxy moxy;

    public ContactsOnlySearchPrivacyFilter(final Moxy moxy) {
        this.moxy = moxy;
    }

    public void filter(
        final ProxySession session,
        final String requestUser,
        final Set<String> users,
        final FutureCallback<Set<String>> callback)
    {
        try {
            filterWithException(session, requestUser, users, callback);
        } catch (HttpException e) {
            callback.failed(e);
        }
    }

    public void filterWithException(
        final ProxySession session,
        final String requestUser,
        final Set<String> users,
        final FutureCallback<Set<String>> callback)
        throws HttpException
    {
        Map<SearchMapShard, List<String>> map = new LinkedHashMap<>();
        for (String userStr: users) {
            User user = new User(
                moxy.config().messagesService(),
                new StringPrefix(userStr));

            SearchMapShard shard = moxy.searchMap().apply(user);
            map.computeIfAbsent(shard, (k) -> new ArrayList<>()).add(userStr);
        }

        AsyncClient searchClient = moxy.searchClient().adjust(session.context());

        session.logger().info(
            "Filter by contacts Privacy " + users.size()
                + " for " + requestUser
                + " shards: " + map.size());
        MultiFutureCallback<JsonObject> mfcb =
            new MultiFutureCallback<>(new Callback(session, callback));

        for (Map.Entry<SearchMapShard, List<String>> entry: map.entrySet()) {
            List<HttpHost> hosts = new ArrayList<>(entry.getKey().size());
            for (SearchMapHost host: entry.getKey()) {
                hosts.add(host.searchHost());
            }
            StringBuilder textsb = new StringBuilder();
            for (String user: entry.getValue()) {
                textsb.append("(contact_user_id:");
                textsb.append(user);
                textsb.append(" AND contact_id:");
                textsb.append(requestUser);
                textsb.append(')');
                textsb.append(" OR ");
            }
            textsb.setLength(textsb.length() - 4);
            QueryConstructor qc = new QueryConstructor("/search?check_cntct_policy");
            qc.append("text", textsb.toString());
            qc.append("get", "contact_user_id");

            searchClient.execute(
                hosts,
                new BasicAsyncRequestProducerGenerator(qc.toString()),
                JsonAsyncTypesafeDomConsumerFactory.OK,
                session.listener().createContextGeneratorFor(searchClient),
                mfcb.newCallback());
        }
        mfcb.done();
    }

    private final class Callback
        extends AbstractFilterFutureCallback<List<JsonObject>, Set<String>>
    {
        private final ProxySession session;

        public Callback(
            final ProxySession session,
            final FutureCallback<? super Set<String>> callback)
        {
            super(callback);

            this.session = session;
        }

        @Override
        public void completed(final List<JsonObject> results) {
            Set<String> matched = new LinkedHashSet<>(results.size());

            try {
                for (JsonObject response: results) {
                    for (JsonObject hit: response.asMap().getList("hitsArray")) {
                        String userId = hit.asMap().getOrNull("contact_user_id");
                        if (userId != null) {
                            matched.add(userId);
                        }
                    }
                }
            } catch (JsonException e) {
                failed(e);
                return;
            }

            session.logger().info("Privacy left " + matched.size());

            callback.completed(matched);
        }
    }
}
