package ru.yandex.search.messenger.proxy;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.function.BasicGenericConsumer;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.parser.StackContentHandler;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.proxy.universal.PlainUniversalSearchProxyRequestContext;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;
import ru.yandex.util.string.StringUtils;

public class GeoSearchHandler implements ProxyRequestHandler {
    private static final int DEFAULT_LENGTH = 10000;
    private static final int DEFAULT_RADIUS = 2000;
    private static final double METTERS_PER_LAT_DEGREE = 111000;
    private static final double MAX_METTERS_PER_LON_DEGREE = 111300;
    private static final double MAF100_PRECISION = 1000;
    private static final double MAF1000_PRECISION = 100;
    private static final String TO = " TO ";
    private static final String DP = "dp";
    private static final String PREFIX = "prefix";
    private static final String SERVICE = "service";
    private static final String ZERO = "0";
    private static final String DPFCONST = "&dp=fconst(";
    private static final String CHAT_DATA = "chat_data";
    private static final String USER_DATA = "user_data";

    private final Moxy moxy;
    private final String chatsService;
    private final String usersService;
    private final long failoverDelay;
    private final boolean localityShuffle;
    private boolean allowLaggingHosts = true;

    public GeoSearchHandler(final Moxy moxy) {
        this.moxy = moxy;
        chatsService = moxy.config().chatsService();
        usersService = moxy.config().usersService();
        failoverDelay = moxy.config().geoFailoverDelay();
        localityShuffle = moxy.config().geoLocalityShuffle();
    }

    @Override
    public void handle(final ProxySession session) throws HttpException {
        final GeoRequestContext geoContext = new GeoRequestContext(session);
        final RequestContext chatsContext =
            new ChatsRequestContext(geoContext, session);
        final RequestContext usersContext =
            new UsersRequestContext(geoContext, session);

        boolean jsonReformat = session.params().getBoolean(
            "data_json_reformat",
            true);
        final DoubleFutureCallback<JsonObject, JsonObject> mergedCallback =
            new DoubleFutureCallback<>(
                new ChatsAndUsersCallback(session, jsonReformat));

        sendRequest(chatsContext, session, mergedCallback.first());
        sendRequest(usersContext, session, mergedCallback.second());
    }

    public void sendRequest(
        final RequestContext context,
        final ProxySession session,
        final FutureCallback<JsonObject> callback)
        throws HttpException
    {
        Prefix prefix = new LongPrefix(0);
        AsyncClient client = moxy.searchClient().adjust(session.context());
        UniversalSearchProxyRequestContext requestContext =
            new PlainUniversalSearchProxyRequestContext(
                new User(context.service(), prefix),
                null,
                context.allowLaggingHosts,
                client,
                session.logger());
        QueryConstructor query = context.query();
        query.append(PREFIX, ZERO);
        query.append(SERVICE, context.service());
        moxy.sequentialRequest(
            session,
            requestContext,
            new BasicAsyncRequestProducerGenerator(query.toString()),
            context.failoverDelay,
            context.localityShuffle,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.listener().createContextGeneratorFor(client),
            callback);
    }

    //CSOFF: ParameterNumber
    private String searchText(
        final String fieldPrefix,
        final String filter,
        final GeoRequestContext geoContext)
    {
        final double cLat = geoContext.cLat;
        final double cLon = geoContext.cLon;
        final int distance = geoContext.distance;
        final StringBuilder sb = new StringBuilder();
        if (filter != null) {
            sb.append('(');
            sb.append(filter);
            sb.append(") AND ");
        }
        sb.append(fieldPrefix);
        if (distance >= MAF100_PRECISION) {
            sb.append("_latitude_maf100:[");
        } else if (distance >= MAF1000_PRECISION) {
            sb.append("_latitude_maf1000:[");
        } else {
            sb.append("_latitude:[");
        }
        final double distanceLat = (distance / METTERS_PER_LAT_DEGREE) * 2;
        final double bottomLat = cLat - distanceLat;
        final double upperLat = cLat + distanceLat;
        sb.append(bottomLat);
        sb.append(TO);
        sb.append(upperLat);
        sb.append("] AND ");
        final double shrinkKoef = Math.abs(Math.cos(cLat));
        final double mettersPerLonDegree = MAX_METTERS_PER_LON_DEGREE
            * shrinkKoef;
        sb.append(fieldPrefix);
        if (distance >= MAF100_PRECISION * shrinkKoef) {
            sb.append("_longitude_maf100:[");
        } else if (distance >= MAF1000_PRECISION * shrinkKoef) {
            sb.append("_longitude_maf1000:[");
        } else {
            sb.append("_longitude:[");
        }
        final double distanceLon = (distance / mettersPerLonDegree) * 2;
        final double leftLon = cLon - distanceLon;
        final double rightLon = cLon + distanceLon;
        sb.append(leftLon);
        sb.append(TO);
        sb.append(rightLon);
        sb.append(']');
        return new String(sb);
    }
    //CSON: ParameterNumber

    @Override
    public String toString() {
        return "Messenger chats & users GEO search handler: "
            + "https://wiki.yandex-team.ru/ps/documentation/"
            + "moxy#geo";
    }

    private class ChatsRequestContext extends RequestContext {
        ChatsRequestContext(
            final GeoRequestContext geoContext,
            final ProxySession session)
            throws HttpException
        {
            super(geoContext, session);
        }

        @Override
        protected String fieldsPrefix() {
            return "chat";
        }

        @Override
        protected String service() {
            return chatsService;
        }

        @Override
        protected String defaultGetFields() {
            return "chat_id,chat_name,distance";
        }
    }

    private class UsersRequestContext extends RequestContext {
        UsersRequestContext(
            final GeoRequestContext geoContext,
            final ProxySession session)
            throws HttpException
        {
            super(geoContext, session);
        }

        @Override
        protected String fieldsPrefix() {
            return "user";
        }

        @Override
        protected String service() {
            return usersService;
        }

        @Override
        protected String defaultGetFields() {
            return "user_id,user_display_name,distance";
        }
    }

    private abstract class RequestContext {
        private final GeoRequestContext geoContext;
        private final int length;
        private final String get;
        private final String sort;
        private final String asc;
        private final List<String> dps;
        private final List<String> postfilters;
        private final String filter;
        private final boolean localityShuffle;
        private final boolean allowLaggingHosts;
        private final long failoverDelay;
        private final QueryConstructor query;
        private final boolean useRawFields;

        @SuppressWarnings("StringSplitter")
        RequestContext(
            final GeoRequestContext geoContext,
            final ProxySession session)
            throws HttpException
        {
            this.geoContext = geoContext;
            String fieldsPrefix = fieldsPrefix();
            int length = session.params().getInt(fieldsPrefix + "_length", -1);
            if (length == -1) {
                length = session.params().getInt("length", DEFAULT_LENGTH);
            }
            this.length = length;
            final String defaultGet = defaultGetFields();
            String get =
                session.params().getString(fieldsPrefix + "_get", null);
            if (get == null) {
                get = session.params().getString(
                    "get",
                    null);
                if (get != null) {
                    String[] gets = get.split(",");
                    List<String> getArray = new ArrayList<>(gets.length);
                    for (String singleGet: gets) {
                        if (singleGet.startsWith(fieldsPrefix)) {
                            getArray.add(singleGet);
                        }
                    }
                    if (getArray.size() > 0) {
                        get = StringUtils.join(getArray, ',');
                    } else {
                        get = defaultGet;
                    }
                } else {
                    get = defaultGet;
                }
            }
            this.get = get;
            sort =
                session.params().getString(fieldsPrefix + "_sort", "distance");
            asc = session.params().getString(fieldsPrefix + "_asc", "true");
            List<String> dps = session.params().getAll(fieldsPrefix + "_dp");
            if (dps == null || dps.size() == 0) {
                dps = session.params().getAll(DP);
            }
            this.dps = dps;
            this.postfilters =
                session.params().getAll(fieldsPrefix + "_postfilter");

            filter = session.params().getString(fieldsPrefix + "_filter", null);
            localityShuffle = session.params().getBoolean(
                "locality-shuffle",
                GeoSearchHandler.this.localityShuffle);
            failoverDelay = session.params().getLong(
                "failover-delay",
                GeoSearchHandler.this.failoverDelay);
            allowLaggingHosts = session.params().getBoolean(
                "allow-lagging-hosts",
                GeoSearchHandler.this.allowLaggingHosts);
            useRawFields =
                session.params().getBoolean(
                    fieldsPrefix + "_use_raw_fields",
                    true);

            query = createQuery();
        }

        protected abstract String fieldsPrefix();

        protected abstract String service();

        protected abstract String defaultGetFields();

        private QueryConstructor createQuery() throws HttpException {
            final String geoFields;
            if (useRawFields) {
                geoFields = fieldsPrefix() + "_latitude_raw"
                    + ',' + fieldsPrefix() + "_longitude_raw";
            } else {
                geoFields = fieldsPrefix() + "_latitude"
                    + ',' + fieldsPrefix() + "_longitude";
            }
            QueryConstructor query = new QueryConstructor(
                "/search?json-type=dollar&IO_PRIO=0&sort=" + sort + "&asc="
                    + asc
                + DPFCONST + geoContext.cLat + "+clat)"
                + DPFCONST + geoContext.cLon + "+clon)"
                + "&dp=geo_distance(clat,clon," + geoFields + "+distance)"
                + "&postfilter=distance+<=+" + geoContext.distance
                + "&length=" + length + "&get=" + get);
            query.append(
                "text",
                searchText(
                    fieldsPrefix(),
                    filter,
                    geoContext));
            if (dps != null && dps.size() > 0) {
                for (String dp: dps) {
                    query.append(DP, dp);
                }
            }
            if (postfilters != null && postfilters.size() > 0) {
                for (String pf: postfilters) {
                    query.append("postfilter", pf);
                }
            }
            return query;
        }

        public QueryConstructor query() {
            return query;
        }
    }

    private static class GeoRequestContext {
        private final double cLat;
        private final double cLon;
        private final int distance;

        GeoRequestContext(final ProxySession session) throws HttpException {
            cLat = session.params().getDouble("clat");
            cLon = session.params().getDouble("clon");
            distance = session.params().getInt("radius", DEFAULT_RADIUS);
        }
    }

    private static class ChatsAndUsersCallback
        extends AbstractProxySessionCallback<Map.Entry<JsonObject, JsonObject>>
    {
        private final JsonType jsonType;
        private final BasicGenericConsumer<JsonObject, JsonException> consumer;
        private final JsonParser jsonParser;
        private final boolean jsonReformat;

        ChatsAndUsersCallback(
            final ProxySession session,
            final boolean jsonReformat)
            throws HttpException
        {
            super(session);
            this.jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
            this.jsonReformat = jsonReformat;
            if (jsonReformat) {
                consumer = new BasicGenericConsumer<>();
                jsonParser = new JsonParser(
                    new StackContentHandler(
                        new TypesafeValueContentHandler(
                            consumer)));
            } else {
                jsonParser = null;
                consumer = null;
            }
        }

        @Override
        public void completed(final Map.Entry<JsonObject, JsonObject> result) {
            try {
                StringBuilderWriter sbw = new StringBuilderWriter();
                try (JsonWriter writer = jsonType.create(sbw)) {
                    writer.startObject();

                    writer.key("chats");
                    writer.startObject();
                    writeResults(result.getKey(), writer, CHAT_DATA);
                    writer.endObject();

                    writer.key("users");
                    writer.startObject();
                    writeResults(result.getValue(), writer, USER_DATA);
                    writer.endObject();

                    writer.endObject();
                }
                session.response(
                    HttpStatus.SC_OK,
                    new NStringEntity(
                        sbw.toString(),
                        ContentType.APPLICATION_JSON
                            .withCharset(session.acceptedCharset())));
            } catch (IOException | JsonException e) {
                failed(e);
            }
        }

        private void writeResults(
            final JsonObject results,
            final JsonWriter writer,
            final String dataKey)
            throws IOException, JsonException
        {
            JsonMap map = results.asMap();
            writer.key("total_found");
            writer.value(map.get("hitsCount"));

            writer.key("hits");
            writer.startArray();
            JsonList hits = map.getList("hitsArray");
            for (JsonObject hit: hits) {
                if (jsonReformat && hit instanceof JsonMap) {
                    jsonReformat(hit.asMap(), dataKey);
                }
                hit.writeValue(writer);
            }
            writer.endArray();
        }

        private void jsonReformat(final JsonMap doc, final String dataKey) {
            try {
                final String jsonString = doc.get(dataKey).asStringOrNull();
                if (jsonString != null) {
                    jsonParser.parse(jsonString);
                    JsonObject obj = consumer.get();
                    doc.put(dataKey, obj);
                }
            } catch (JsonException e) { // skip, obj is null
            }
        }
    }
}

