package ru.yandex.search.messenger.proxy;

import java.io.IOException;
import java.io.InputStream;
import java.text.ParseException;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import NMessengerProtocol.Client;
import NMessengerProtocol.Message;
import com.google.protobuf.CodedOutputStream;
import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;

import ru.yandex.cityhash.CityHashingArrayOutputStream;
import ru.yandex.collection.IntPair;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.HttpStatusPredicates;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.NByteArrayEntityAsyncConsumerFactory;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.StatusCodeAsyncConsumerFactory;

public class LastSeenInfoProvider {
    public static final int VERSION = 2;

    private static final ThreadLocal<CityHashingArrayOutputStream> OUT_TLS =
            ThreadLocal.<CityHashingArrayOutputStream>withInitial(
                    () -> new CityHashingArrayOutputStream());

    private final Moxy moxy;

    public LastSeenInfoProvider(final Moxy moxy) {
        this.moxy = moxy;
    }

    public void get(
            final ProxySession session,
            final Set<String> guids,
            final FutureCallback<Map<String, Long>> callback)
            throws IOException, HttpException
    {
        get(session, guids, false, callback);
    }

    public void get(
            final ProxySession session,
            final Set<String> guids,
            final boolean foreground,
            final FutureCallback<Map<String, Long>> callback)
            throws IOException, HttpException
    {
        session.logger().info(
            "Requesting online status for guids size " + guids.size()
                + ", foreground: " + foreground);
        Message.TLastSeenRequest request = Message.TLastSeenRequest.newBuilder()
            .addAllGuids(guids)
            .setForeground(foreground)
            .build();

        CityHashingArrayOutputStream out = OUT_TLS.get();
        out.reset();
        CodedOutputStream googleOut = CodedOutputStream.newInstance(out);

        request.writeTo(googleOut);
        googleOut.flush();

        byte[] postData =
                out.toByteArrayWithVersion(VERSION);

        final BasicAsyncRequestProducerGenerator post =
                new BasicAsyncRequestProducerGenerator(
                        "/last_seen",
                        postData,
                        ContentType.DEFAULT_BINARY);

        moxy.mssngrRouterClient().execute(
                moxy.config().mssngrRouterConfig().host(),
                post,
                new StatusCheckAsyncResponseConsumerFactory<>(
                        HttpStatusPredicates.NON_PROTO_FATAL,
                        new StatusCodeAsyncConsumerFactory<>(
                                NByteArrayEntityAsyncConsumerFactory.INSTANCE)),
                session.listener().createContextGeneratorFor(moxy.mssngrRouterClient()),
                new LastSeenCallback(session, callback));
    }

    private static class LastSeenCallback extends AbstractProxySessionCallback<IntPair<HttpEntity>> {
        private final FutureCallback<Map<String, Long>> callback;

        public LastSeenCallback(
                final ProxySession session,
                final FutureCallback<Map<String, Long>> callback)
        {
            super(session);
            this.callback = callback;
        }

        @Override
        public void completed(final IntPair<HttpEntity> response) {
            HttpEntity entity = response.second();
            int code = response.first();
            try {
                switch (code) {
                    case HttpStatus.SC_OK:
                        parseResponse(entity);
                        break;
                    default:
                        failed(
                                new HttpException("Last seen request failed with code "
                                        + code + " " + CharsetUtils.toString(entity)));
                        return;
                }
            } catch (HttpException | IOException | ParseException e) {
                failed(e);
            }
        }

        private void parseResponse(final HttpEntity entity)
                throws IOException, HttpException, ParseException
        {
            InputStream is = entity.getContent();
            long skipped =
                    is.skip(CityHashingArrayOutputStream.THEADER_SIZE);
            if (skipped != CityHashingArrayOutputStream.THEADER_SIZE) {
                callback.failed(new HttpException("Header skip failed"));
                return;
            }
            Message.TLastSeenResponse response =
                    Message.TLastSeenResponse.parseFrom(is);
            if (response.getStatus() != Message.EGenericResponseStatus.Success) {
                callback.failed(
                    new HttpException(
                        "Router request failed " + response.getStatus()
                            + " " + response.getErrorInfos(0)));
                return;
            }
            List<Client.TLastSeenInfo> infos = response.getLastSeenInfosList();
            Map<String, Long> result = new LinkedHashMap<>();
            for (Client.TLastSeenInfo info: infos) {
                result.put(info.getUser().getGuid(), info.getTimestamp());
            }
            session.logger().info("LastSeenResult size " + result.size());
            callback.completed(result);
        }
    }
}
