package ru.yandex.search.messenger.proxy;

import java.io.IOException;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Set;

import javax.xml.parsers.ParserConfigurationException;

import org.apache.http.concurrent.FutureCallback;
import org.xml.sax.SAXException;

import ru.yandex.collection.Pattern;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.erratum.ErratumClient;
import ru.yandex.erratum.ImmutableErratumConfig;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.FilterFutureCallback;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.search.messenger.SearchPrivacy;
import ru.yandex.search.messenger.UserFields;
import ru.yandex.search.messenger.proxy.chatmedia.ChatMediaHandler;
import ru.yandex.search.messenger.proxy.config.ImmutableMoxyConfig;
import ru.yandex.search.messenger.proxy.forward.ForwardSuggestHandler;
import ru.yandex.search.messenger.proxy.recchannels.CachingRecommendedChannelsHandler;
import ru.yandex.search.messenger.proxy.recchannels.RecommendedChannelsHandler;
import ru.yandex.search.messenger.proxy.recents.DefaultRecentsHandler;
import ru.yandex.search.messenger.proxy.recents.RecentsCache;
import ru.yandex.search.messenger.proxy.suggest.ChatsFilter;
import ru.yandex.search.messenger.proxy.suggest.SuggestHandler;
import ru.yandex.search.messenger.proxy.suggest.SuggestRequestContext;
import ru.yandex.search.messenger.proxy.suggest.UsersFilter;
import ru.yandex.search.messenger.proxy.topposts.TopPostsHandler;
import ru.yandex.search.proxy.universal.UniversalSearchProxy;
import ru.yandex.stater.CountAggregatorFactory;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.MaxAggregatorFactory;
import ru.yandex.stater.MinAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;
import ru.yandex.stater.TriplexStaterFactory;

public class Moxy extends UniversalSearchProxy<ImmutableMoxyConfig> {
    private static final String USER_ID = "user_id";
    private static final String CHAT_ID = "chat_id";
    private static final String MESSAGE_CHAT_ID = "message_chat_id";
    private static final String MESSAGE_ID = "message_id";

    private final Synonyms synonyms;
    private final TimeFrameQueue<Long> topChatsPostsCount;
    private final TimeFrameQueue<Long> cmntEntitiesCount;
    private final TimeFrameQueue<Long> cmntCacheHits;
    private final AsyncClient mssngrRouterClient;
    private final AsyncClient selfMoxyClient;
    private final ErratumClient misspellClient;
    private final LastSeenInfoProvider lastSeenInfoProvider;
    private final ContactsOnlySearchPrivacyFilter privacyFilter;
    private final TimeFrameQueue<Integer> recentsResponces;
    private final TimeFrameQueue<Integer> forwardRespones;
    private final RecentsCache recentsCache;

    public Moxy(final ImmutableMoxyConfig config)
        throws IOException, SAXException, ParserConfigurationException
    {
        super(config);
        synonyms = new Synonyms();

        if (config.recentsCache()) {
            recentsCache = new RecentsCache(this);
        } else {
            recentsCache = null;
        }

        mssngrRouterClient = client("MssngrRouter", config.mssngrRouterConfig());
        selfMoxyClient = client("SelfMoxyClient", config.moxyConfig());
        ImmutableErratumConfig misspellConfig = config.misspellConfig();
        if (misspellConfig == null) {
            misspellClient = null;
        } else {
            misspellClient = new ErratumClient(reactor, misspellConfig);
            registerClient("ErratumClient", misspellClient, misspellConfig);
        }
        lastSeenInfoProvider = new LastSeenInfoProvider(this);

        topChatsPostsCount =
            new TimeFrameQueue<>(config.metricsTimeFrame());
        cmntEntitiesCount =
            new TimeFrameQueue<>(config.metricsTimeFrame());
        cmntCacheHits =
            new TimeFrameQueue<>(config.metricsTimeFrame());

        privacyFilter = new ContactsOnlySearchPrivacyFilter(this);

        register(
            new Pattern<>("/api/search/messenger/geo", false),
            new GeoSearchHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/search/messenger/suggest", false),
            new SuggestHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/search/messenger/recents", false),
            new DefaultRecentsHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/search/messenger/forward/suggest", false),
            new ForwardSuggestHandler(this),
            //new RecentsHandler(this),
            RequestHandlerMapper.GET);
        ProxyRequestHandler recChannelsHandler;
        if (config.recommendedChannelsConfig().cacheEnabled()) {
            recChannelsHandler = new CachingRecommendedChannelsHandler(this);
        } else {
            recChannelsHandler = new RecommendedChannelsHandler(this);
        }
        register(
            new Pattern<>("/api/search/messenger/recommended-channels", false),
            recChannelsHandler,
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/search/messenger/top-posts", false),
            new TopPostsHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/search/messenger/user-messages", false),
            new UserMessagesHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/search/messenger/chat-media", false),
            new ChatMediaHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/search/messenger/cmnt-info", false),
            new CMNTInfoHandler(this, cmntEntitiesCount, cmntCacheHits),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/search/messenger/chat-members", false),
            new ChatMembersHandler(this),
            RequestHandlerMapper.GET);

        // special for manokk
        if (config.producerClientConfig() != null) {
            register(
                new Pattern<>("/api/search/messenger/clear-chat-messages", false),
                new Drop26NamespaceChatHistory(this),
                RequestHandlerMapper.GET);
        }

        registerStater(
            new PassiveStaterAdapter<>(
                topChatsPostsCount,
                new TriplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "top-chats-posts-count_ammm",
                        IntegralSumAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        "top-chats-posts-count_annn",
                        new MinAggregatorFactory()),
                    new NamedStatsAggregatorFactory<>(
                        "top-chats-posts-count_axxx",
                        new MaxAggregatorFactory(0)))));
        registerStater(
            new PassiveStaterAdapter<>(
                cmntEntitiesCount,
                new TriplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "cmnt-info-entity-count-count_ammm",
                        IntegralSumAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        "cmnt-info-entity-count-count_axxx",
                        new MinAggregatorFactory()),
                    new NamedStatsAggregatorFactory<>(
                        "cmnt-info-entity-count-count_annn",
                        new MaxAggregatorFactory(0)))));
        registerStater(
            new PassiveStaterAdapter<>(
                cmntCacheHits,
                new TriplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "cmnt-info-cache-hits_ammm",
                        IntegralSumAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        "cmnt-info-cache-hits_axxx",
                        new MinAggregatorFactory()),
                    new NamedStatsAggregatorFactory<>(
                        "cmnt-info-cache-hits_annn",
                        new MaxAggregatorFactory(0)))));

        recentsResponces = new TimeFrameQueue<>(config().metricsTimeFrame());
        registerStater(
            new PassiveStaterAdapter<>(
                recentsResponces,
                new NamedStatsAggregatorFactory<>(
                    "recents-empty-responses_ammm",
                    CountAggregatorFactory.INSTANCE)));
        forwardRespones = new TimeFrameQueue<>(config().metricsTimeFrame());
        registerStater(
            new PassiveStaterAdapter<>(
                forwardRespones,
                new NamedStatsAggregatorFactory<>(
                    "forwards-empty-responses_ammm",
                    CountAggregatorFactory.INSTANCE)));
    }

    public void consumePostCount(final long count) {
        topChatsPostsCount.accept(count);
    }

    public Synonyms synonyms() {
        return synonyms;
    }

    public AsyncClient selfMoxyClient() {
        return selfMoxyClient;
    }

    public AsyncClient mssngrRouterClient() {
        return mssngrRouterClient;
    }

    public ErratumClient misspellClient() {
        return misspellClient;
    }

    public LastSeenInfoProvider lastSeenInfoProvider() {
        return lastSeenInfoProvider;
    }

    public ContactsOnlySearchPrivacyFilter privacyFilter() {
        return privacyFilter;
    }

    //CSOFF: ParameterNumber
    public void filterUserResources(
        final SuggestRequestContext context,
        final String userGuid,
        final JsonList hits,
        final UsersFilter filter,
        final FutureCallback<? super Set<String>> callback)
        throws BadRequestException, JsonException
    {
        Set<String> resourcesWithPrivacy = null;
        Set<String> resources = new HashSet<>(hits.size() << 1);
        int nobody = 0;
        for (JsonObject hit: hits) {
            String resourceId = hit.get(USER_ID).asStringOrNull();
            if (resourceId != null && (filter == null || filter.testUser(resourceId))) {
                SearchPrivacy searchPrivacy =
                    SearchPrivacy.parse(
                        hit.get(UserFields.SEARCH_PRIVACY.stored()).asStringOrNull());

                if (searchPrivacy == SearchPrivacy.CONTACTS) {
                    if (userGuid == null) {
                        continue;
                    }

                    if (resourcesWithPrivacy == null) {
                        resourcesWithPrivacy = new LinkedHashSet<>();
                    }

                    resourcesWithPrivacy.add(resourceId);
                } else if (searchPrivacy != SearchPrivacy.NOBODY) {
                    resources.add(resourceId);
                } else {
                    nobody += 1;
                }
            }
        }

        if (nobody > 0) {
            logger.info("Filtered by nobody filter " + nobody);
        }

        if (resourcesWithPrivacy != null) {
            logger.info("Contacts needs to be validated against privacy " + resourcesWithPrivacy);
            privacyFilter.filter(
                context.session(),
                userGuid,
                resourcesWithPrivacy,
                new PrivacyMergeCallback(callback, resources));
        } else {
            callback.completed(resources);
        }
    }

    private static class PrivacyMergeCallback extends FilterFutureCallback<Set<String>> {
        private final Set<String> resources;

        public PrivacyMergeCallback(
            final FutureCallback<? super Set<String>> callback,
            final Set<String> resources)
        {
            super(callback);
            this.resources = resources;
        }

        @Override
        public void completed(final Set<String> result) {
            this.resources.addAll(result);
            super.completed(this.resources);
        }
    }
    //CSON: ParameterNumber

    //CSOFF: ParameterNumber
    public void filterChatResources(
        final SuggestRequestContext context,
        final JsonList hits,
        final FutureCallback<? super Set<String>> callback)
        throws BadRequestException, JsonException
    {
        Set<String> resources = new HashSet<>(hits.size() << 1);
        for (JsonObject hit: hits) {
            String resourceId = hit.get(CHAT_ID).asStringOrNull();
            if (resourceId != null) {
                resources.add(resourceId);
            }
        }
        callback.completed(resources);
    }
    //CSON: ParameterNumber

    //CSOFF: ParameterNumber
    public void filterMessagesResources(
        final SuggestRequestContext context,
        final JsonList hits,
        final ChatsFilter filter,
        final FutureCallback<? super Set<String>> callback)
        throws BadRequestException, JsonException
    {
        Set<String> resources = new HashSet<>(hits.size() << 1);
        for (JsonObject hit: hits) {
            String resourceId = hit.get(MESSAGE_ID).asStringOrNull();
            String chatId = hit.get(MESSAGE_CHAT_ID).asStringOrNull();
            if (resourceId != null && chatId != null
                && filter.testChat(chatId))
            {
                resources.add(resourceId);
            } else {
                context.logger().info("Filtering out " + resourceId + " " + chatId + " " + filter);
            }
        }
        callback.completed(resources);
    }
    //CSON: ParameterNumber

    public void emptyRecentResponse() {
        recentsResponces.accept(1);
    }

    public void emptyForwardResponse() {
        forwardRespones.accept(1);
    }

    public RecentsCache recentsCache() {
        return recentsCache;
    }
}
