package ru.yandex.search.messenger.proxy;

import java.io.IOException;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedQueue;

import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.function.BasicGenericConsumer;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.parser.StackContentHandler;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.messenger.proxy.config.ImmutableMoxyConfig;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.prefix.StringPrefix;
import ru.yandex.search.proxy.universal.PlainUniversalSearchProxyRequestContext;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;
import ru.yandex.util.string.StringUtils;
//import ru.yandex.parser.string.NonEmptyValidator;

public class RecentContactsHandler implements ProxyRequestHandler {
    private static final int DEFAULT_LENGTH = 20;
    private static final String GET = "get";
    private static final String DP = "dp";
    private static final String PREFIX = "prefix";
    private static final String SERVICE = "service";
    private static final String ZERO = "0";
    private static final String TEXT = "text";
    private static final String HITS_ARRAY = "hitsArray";
    private static final String USER_ID = "user_id";
    private static final String CHAT_ID = "chat_id";
    private static final String USER_DATA = "user_data";
    private static final String CHAT_DATA = "chat_data";
    private static final String CHAT_NAME = "chat_name";
    private static final String CHAT_DESCRIPTION = "chat_description";
    private static final String CHAT_INVITE_HASH = "chat_invite_hash";
    private static final String CHAT_AVATAR_ID = "chat_avatar_id";
    private static final String CHAT_MEMBER_COUNT = "chat_member_count";
    private static final String CHAT_MEMBERS = "chat_members";
    private static final String MESSAGE_DATA = "message_data";
    private static final String MESSAGE_ID = "message_id";
    private static final String MESSAGE_CHAT_ID = "message_chat_id";
    private static final String MESSAGE_SEQ_NO = "message_seq_no";
    private static final String MESSAGE_RCA_DATA = "message_rca_data";
    private static final String MESSAGE_TIMESTAMP = "message_timestamp";
    private static final String GET_FIELDS = CHAT_MEMBERS
        + ",chat_last_message_timestamp,chat_message_count,user_id,user_data";

    private static final CollectionParser<
        String,
        Set<String>,
        Exception>
        SET_PARSER = new CollectionParser<>(
            String::trim,
//            NonEmptyValidator.INSTANCE,
            LinkedHashSet::new);
    private static final ConcurrentLinkedQueue<SimpleJsonParser> JSON_PARSERS =
        new ConcurrentLinkedQueue<>();
    private static final boolean ALLOW_LAGGING_HOSTS = true;

    private final Moxy moxy;
    private final String messagesService;
    private final String chatsService;
//    private final String usersService;

    public RecentContactsHandler(final Moxy moxy) {
        this.moxy = moxy;
        messagesService = moxy.config().messagesService();
        chatsService = moxy.config().chatsService();
//        usersService = moxy.config().usersService();
    }

    public static SimpleJsonParser getJsonParser() {
        SimpleJsonParser parser = JSON_PARSERS.poll();
        if (parser == null) {
            parser = new SimpleJsonParser();
        }
        return parser;
    }

    public static void freeJsonParser(final SimpleJsonParser parser) {
        JSON_PARSERS.add(parser);
    }

    @Override
    public void handle(final ProxySession session) throws HttpException {
        RecentRequestContext topContext =
            new RecentRequestContext(session, moxy.config());
        ChatsRequestContext chatsContext =
            new ChatsRequestContext(session, topContext);
        ContactsRequestContext contactsContext =
            new ContactsRequestContext(session, topContext);
        sendRequest(
            chatsContext,
            session,
            new ChatsCallback(session, contactsContext));
    }

    public void sendRequest(
        final RequestContext context,
        final ProxySession session,
        final FutureCallback<JsonObject> callback)
        throws HttpException
    {
        AsyncClient client = moxy.searchClient().adjust(session.context());
        UniversalSearchProxyRequestContext requestContext =
            new PlainUniversalSearchProxyRequestContext(
                context.user(),
                null,
                context.context.allowLaggingHosts,
                client,
                session.logger());
        QueryConstructor query = context.query();
        query.append(SERVICE, context.service());
        moxy.sequentialRequest(
            session,
            requestContext,
            new BasicAsyncRequestProducerGenerator(query.toString()),
            context.context.failoverDelay,
            context.context.localityShuffle,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.listener().createContextGeneratorFor(client),
            callback);
    }

    @Override
    public String toString() {
        return "Recents handler: "
            + "https://wiki.yandex-team.ru/ps/documentation/"
            + "moxy#recents";
    }

    private static class RecentRequestContext {
        private final int length;
        private final String get;
        private final boolean localityShuffle;
        private final boolean allowLaggingHosts;
        private final long failoverDelay;
        private final String guid;
        private final Map<String, UserInfo> foundUsers;
        private final JsonType jsonType;
        private final ProxySession session;

        RecentRequestContext(
            final ProxySession session,
            final ImmutableMoxyConfig config)
            throws HttpException
        {
            this.session = session;
            this.length = session.params().getInt("length", DEFAULT_LENGTH);
            String get =
                session.params().getString(GET, null);
            this.get = get;
            this.localityShuffle = session.params().getBoolean(
                "locality-shuffle",
                config.topPostsLocalityShuffle());
            this.failoverDelay = session.params().getLong(
                "failover-delay",
                config.topPostsFailoverDelay());
            this.allowLaggingHosts = session.params().getBoolean(
                "allow-lagging-hosts",
                ALLOW_LAGGING_HOSTS);
            this.guid = session.params().getString("guid");
            this.foundUsers = new LinkedHashMap<>();
            this.jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
        }

        public void addUser(final String guid, final UserInfo user) {
            foundUsers.put(guid, user);
        }

        public int length() {
            return length;
        }

        public String get() {
            return get;
        }

        public String guid() {
            return guid;
        }

        public void submitResponse() throws IOException {
            StringBuilderWriter sbw = new StringBuilderWriter();

            try (JsonWriter writer = jsonType.create(sbw)) {
                writer.startObject();
                writer.key("recents");
                writer.startArray();
                for (Map.Entry<String, UserInfo> entry: foundUsers.entrySet()) {
                    final String guid = entry.getKey();
                    final UserInfo user = entry.getValue();
//                    writer.key(guid);
                    user.json.writeValue(writer);
                }
                writer.endArray();
                writer.endObject();
                HttpEntity entity =
                    new NStringEntity(
                        sbw.toString(),
                        ContentType.APPLICATION_JSON.withCharset(
                            session.acceptedCharset()));
                session.response(HttpStatus.SC_OK, entity);
            }
        }
    }

    private static abstract class RequestContext {
        protected RecentRequestContext context;

        RequestContext(
            final ProxySession session,
            final RecentRequestContext context)
            throws HttpException
        {
            this.context = context;
        }

        protected abstract User user();

        protected abstract String service();

        protected QueryConstructor query() throws HttpException {
            return createQuery(context);
        }

        protected abstract QueryConstructor createQuery(
            final RecentRequestContext context)
            throws HttpException;
    }

    private class ChatsRequestContext extends RequestContext {
        private final User user;

        ChatsRequestContext(
            final ProxySession session,
            final RecentRequestContext context)
            throws HttpException
        {
            super(session, context);
            user = new User(chatsService, new LongPrefix(0));
        }

        @Override
        protected String service() {
            return chatsService;
        }

        @Override
        protected QueryConstructor createQuery(
            final RecentRequestContext context)
            throws HttpException
        {
            QueryConstructor query =
                new QueryConstructor(
                    "/search?IO_PRIO=0&json-type=dollar");
            query.append("get", GET_FIELDS);
            query.append(PREFIX, ZERO);
            query.append("service", user.service());
            if (context.get() != null) {
                query.append("get", context.get());
            }
            query.append("text", "chat_members:" + context.guid());
            query.append("dp", "contains(chat_id,/ slash)");
            query.append("dp", "filter_cmp(slash,!=,1)");
            query.append(
                "dp",
                "replace_all(chat_id,(" + context.guid + "|_), guid)");
            query.append("dp", "equals(guid,,  empty)");
            query.append("dp", "filter_cmp(empty,==,0)");
            query.append(
                "dp",
                "left_join(guid,user_id,,user_data user_data,user_id user_id)");
            query.append("postfilter", "slash != 1");
            query.append("outer", "deduplicate:chat_members");
            query.append("length", context.length);

            return query;
        }

        @Override
        protected User user() {
            return user;
        }
    }

    private class ContactsRequestContext extends RequestContext {
        private final User user;

        ContactsRequestContext(
            final ProxySession session,
            final RecentRequestContext context)
            throws HttpException
        {
            super(session, context);
            user =
                new User(
                    messagesService,
                    new StringPrefix(context.guid));
        }

        @Override
        protected String service() {
            return messagesService;
        }

        @Override
        protected QueryConstructor createQuery(
            final RecentRequestContext context)
            throws HttpException
        {
            QueryConstructor query =
                new QueryConstructor(
                    "/search?IO_PRIO=0&json-type=dollar"
                        + "&skip-nulls"
                        + "&scorer=lucene&sort=multi(%23score,contact_name)"
                        + "&dp=fallback(contact_id+id)");
            query.append("get", "contact_id,contact_name");
            query.append("prefix", user.prefix().toString());
            query.append("service", user.service());
            query.append(
                "text",
                StringUtils.join(
                    context.foundUsers.keySet(),
                    ' ',
                    "contact_id_p:(",
                    ")"));
            return query;
        }

        @Override
        protected User user() {
            return user;
        }
    }

    private class ChatsCallback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final ContactsRequestContext context;

        ChatsCallback(
            final ProxySession session,
            final ContactsRequestContext context)
            throws HttpException
        {
            super(session);
            this.context = context;
        }

        @Override
        public void completed(final JsonObject result) {
            SimpleJsonParser jsonParser = getJsonParser();
            try {
                JsonList hits = result.asMap().getList("hitsArray");
                for (JsonObject o: hits) {
                    JsonMap hit = o.asMap();
                    String userId = hit.getString("user_id", null);
                    if (userId != null) {
                        jsonReformat(hit, jsonParser);
                        context.context.addUser(
                            userId,
                            new UserInfo(userId, hit));
                    }
                }
                FutureCallback<JsonObject> contactsCallback =
                    new ContactsCallback(session, context);
                if (context.context.foundUsers.size() > 0) {
                    sendRequest(context, session, contactsCallback);
                } else {
                    context.context.submitResponse();
                }
            } catch (HttpException | JsonException | IOException e) {
                failed(e);
            } finally {
                freeJsonParser(jsonParser);
            }
        }

        private void jsonReformat(
            final JsonMap doc,
            final SimpleJsonParser jsonParser)
        {
            try {
                String jsonString = doc.get(CHAT_DATA).asStringOrNull();
                if (jsonString != null) {
                    JsonObject obj = jsonParser.parse(jsonString);
                    doc.put(CHAT_DATA, obj);
                }
                jsonString = doc.get(USER_DATA).asStringOrNull();
                if (jsonString != null) {
                    JsonObject obj = jsonParser.parse(jsonString);
                    doc.put(USER_DATA, obj);
                }
            } catch (JsonException e) { // skip, obj is null
            }
        }
    }

    private class ContactsCallback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final ContactsRequestContext context;

        ContactsCallback(
            final ProxySession session,
            final ContactsRequestContext context)
            throws HttpException
        {
            super(session);
            this.context = context;
        }

        @Override
        public void completed(final JsonObject result) {
            try {
                JsonList hits = result.asMap().getList("hitsArray");
                for (JsonObject o: hits) {
                    JsonMap hit = o.asMap();
                    String contactName = hit.getString("contact_name", null);
                    String contactId = hit.getString("contact_id", null);
                    if (contactId != null && contactName != null) {
                        UserInfo user =
                            context.context.foundUsers.get(contactId);
                        if (user != null) {
                            user.json.put(
                                "contact_name",
                                new JsonString(contactName));
                        }
                    }
                }
                context.context.submitResponse();
            } catch (IOException | JsonException e) {
                failed(e);
            }
        }
    }

    private static final class UserInfo {
        private final String guid;
        private final JsonMap json;

        UserInfo(final String guid, final JsonMap json) {
            this.guid = guid;
            this.json = json;
        }
    }

    private static final class SimpleJsonParser {
        private final JsonParser parser;
        private final BasicGenericConsumer<JsonObject, JsonException> consumer;

        SimpleJsonParser() {
            consumer = new BasicGenericConsumer<>();
            parser =
                new JsonParser(new StackContentHandler(
                    new TypesafeValueContentHandler(consumer)));
        }

        public JsonObject parse(final String json) throws JsonException {
            parser.parse(json);
            return consumer.get();
        }
    }
}

