package ru.yandex.search.messenger.proxy;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParserFactory;

import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

public class Synonyms {
    private static final String KEY = "key";
    private static final String ARRAY = "array";
    private static final String STRING = "string";

    private final TreeMap<String, String[]> synonymsMap = new TreeMap<>();

    public Synonyms()
        throws IOException, SAXException, ParserConfigurationException
    {
        try (Reader reader = new InputStreamReader(
                Synonyms.class.getResourceAsStream("humannames.plist"),
                StandardCharsets.UTF_8))
        {
            SAXParserFactory.newInstance().newSAXParser().parse(
                new InputSource(reader),
                new PlistParser(synonymsMap));
        }
    }

    private static class PlistParser extends DefaultHandler {
        private final List<String> values = new ArrayList<>();
        private final StringBuilder sb = new StringBuilder();
        private final Map<String, String[]> synonymsMap;
        private boolean isKey = false;
        private boolean isArray = false;
        private boolean isValue = false;
        private String key;

        PlistParser(final Map<String, String[]> synonymsMap) {
            this.synonymsMap = synonymsMap;
        }

        //CSOFF: ParameterNumber
        @Override
        public void startElement(
            final String uri,
            final String localName,
            final String qName,
            final Attributes attributes)
        {
            if (qName.equalsIgnoreCase(KEY)) {
                isKey = true;
                isArray = false;
                isValue = false;
                key = null;
            } else if (qName.equalsIgnoreCase(ARRAY)) {
                isKey = false;
                isArray = true;
                isValue = false;
                values.clear();
            } else if (qName.equalsIgnoreCase(STRING)) {
                isValue = true;
                sb.setLength(0);
            }
        }

        @Override
        public void endElement(
            final String uri,
            final String localName,
            final String qName)
        {
            if (isArray
                && qName.equalsIgnoreCase(ARRAY)
                && !values.isEmpty()
                && key != null)
            {
                values.add(key);
                String[] array = values.toArray(new String[values.size()]);
                values.clear();
                for (String value: array) {
                    String[] old = synonymsMap.put(value, array);
                    if (old != null) {
                        Set<String> set = new LinkedHashSet<>(
                            (array.length + old.length) << 1);
                        for (String str: old) {
                            set.add(str);
                        }
                        for (String str: array) {
                            set.add(str);
                        }
                        synonymsMap.put(
                            value,
                            set.toArray(new String[values.size()]));
                    }
                }
                isArray = false;
                isKey = false;
                key = null;
            } else if (isValue && qName.equalsIgnoreCase(STRING)) {
                if (sb.length() > 0) {
                    values.add(new String(sb));
                    sb.setLength(0);
                }
                isValue = false;
            } else if (isKey && qName.equalsIgnoreCase(KEY)) {
                if (sb.length() > 0) {
                    key = new String(sb);
                    sb.setLength(0);
                }
                isKey = false;
            }
        }

        @Override
        public void characters(
            final char[] ch,
            final int start,
            final int length)
            throws SAXException
        {
            if (isKey || isValue) {
                sb.append(ch, start, length);
            }
        }
    }

    public List<String> suggest(final String str, final boolean isPrefix) {
        List<String> synonyms = new ArrayList<>();
        if (isPrefix) {
            Map<String, String[]> subMap =
                synonymsMap.subMap(str, str + Character.MAX_VALUE);
            for (String[] value: subMap.values()) {
                for (String syn: value) {
                    synonyms.add(syn);
                }
            }
        } else {
            String[] value = synonymsMap.get(str);
            if (value != null) {
                synonyms.addAll(Arrays.asList(value));
            }
        }
        return synonyms;
    }

    public List<String> suggest(final String str) {
        Map<String, String[]> tailMap = synonymsMap.tailMap(str);
        List<String> synonyms = new ArrayList<>();
        for (Map.Entry<String, String[]> entry: tailMap.entrySet()) {
            if (!entry.getKey().startsWith(str)) {
                break;
            }
            synonyms.addAll(Arrays.asList(entry.getValue()));
        }
        return synonyms;
    }

    public String[] synonyms(final String str) {
        return synonymsMap.get(str);
    }
}
