package ru.yandex.search.messenger.proxy;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;

import NMessengerProtocol.Client.TServerMessage;
import NMessengerProtocol.Message.TOutMessage;
import NMessengerProtocol.Search.TPostPreviewInfo;
import NMessengerProtocol.Search.TTopPosts;
import NMessengerProtocol.Search.TTopPostsResponse;
import com.google.protobuf.CodedInputStream;
import com.google.protobuf.CodedOutputStream;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.LooseProcessors;
import io.netty.util.HashedWheelTimer;
import io.netty.util.Timeout;
import io.netty.util.TimerTask;
import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NByteArrayEntity;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.cityhash.CityHashingArrayOutputStream;
import ru.yandex.function.BasicGenericConsumer;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.parser.StackContentHandler;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.searchmap.SearchMapShard;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.messenger.ChatFields;
import ru.yandex.search.messenger.UserChats;
import ru.yandex.search.messenger.proxy.config.ImmutableMoxyConfig;
import ru.yandex.search.messenger.proxy.suggest.rules.ProtoUtils;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.StringPrefix;
import ru.yandex.search.proxy.universal.PlainUniversalSearchProxyRequestContext;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;
import ru.yandex.util.string.StringUtils;
import ru.yandex.util.string.UnhexStrings;
//import ru.yandex.parser.string.NonEmptyValidator;

public class TopPostsHandler implements ProxyRequestHandler {
    private static final long MS_PER_MICRO = 1000;
    private static final long MILLIS_PER_DAY = 86400 * 1000;
    private static final long EXPIRE_THRESHOLD = 10;
    private static final long GC_THRESHOLD = 50;
    private static final int DEFAULT_LENGTH = 20;
    private static final int DEFAULT_BYTE_ARRAY_LENGTH = 1024;
    private static final int VERSION = 2;
    private static final int MIN_IMAGE_SIZE = 240;
    private static final String GET = "get";
    private static final String DP = "dp";
    private static final String PREFIX = "prefix";
    private static final String SERVICE = "service";
    private static final String ZERO = "0";
    private static final String TEXT = "text";
    private static final String HITS_ARRAY = "hitsArray";
    private static final String USER_ID = "user_id";
    private static final String CHAT_ID = "chat_id";
    private static final String CHAT_NAME = "chat_name";
    private static final String CHAT_DESCRIPTION = "chat_description";
    private static final String CHAT_INVITE_HASH = "chat_invite_hash";
    private static final String CHAT_AVATAR_ID = "chat_avatar_id";
    private static final String CHAT_MEMBER_COUNT = "chat_member_count";
    private static final String CHAT_MEMBERS = "chat_members";
    private static final String MESSAGE_DATA = "message_data";
    private static final String MESSAGE_ID = "message_id";
    private static final String MESSAGE_CHAT_ID = "message_chat_id";
    private static final String MESSAGE_SEQ_NO = "message_seq_no";
    private static final String MESSAGE_RCA_DATA = "message_rca_data";
    private static final String MESSAGE_TIMESTAMP = "message_timestamp";
    private static final String SNIPPET = "snippet";
    private static final String TOP_MESSAGES = "chat_top_messages";

    private static final HashedWheelTimer DELAYED_EXECUTOR =
        newHashedWheelTimer();

    private static HashedWheelTimer newHashedWheelTimer() {
        HashedWheelTimer timer =
            new HashedWheelTimer(1 + 2 + 2, TimeUnit.MILLISECONDS);
        timer.start();
        return timer;
    }

    private static final CollectionParser<
        String,
        Set<String>,
        Exception>
        SET_PARSER = new CollectionParser<>(
            String::trim,
//            NonEmptyValidator.INSTANCE,
            LinkedHashSet::new);
    private static final Set<String> DEFAULT_MESSAGE_TYPES =
        new LinkedHashSet<>(
            Arrays.asList(
                new String[] {
                    "text_message",
//                    "image_message",
                    "gallery_message"}));
    private static final ConcurrentLinkedQueue<SimpleJsonParser> JSON_PARSERS =
        new ConcurrentLinkedQueue<>();
    private static final boolean ALLOW_LAGGING_HOSTS = true;
    private static final ThreadLocal<CityHashingArrayOutputStream> OUT_TLS =
        ThreadLocal.<CityHashingArrayOutputStream>withInitial(
            () -> new CityHashingArrayOutputStream());
    private static final ThreadLocal<ReusableCodedInputStream> CODED_INPUT_TLS =
        ThreadLocal.<ReusableCodedInputStream>withInitial(
            () -> new ReusableCodedInputStream());

    private final Moxy moxy;
    private final String messagesService;
    private final String chatsService;
    private final ThreadPoolExecutor executor;
//    private final ConcurrentHashMap<String, CachedResponse> cache;
    private final ConcurrentHashMap<String, ChatsTopMessages> cache;
    private final long cacheUpdateInterval;

    public TopPostsHandler(final Moxy moxy) {
        this.moxy = moxy;
        LooseProcessors.init();
        messagesService = moxy.config().messagesService();
        chatsService = moxy.config().chatsService();
        cache = new ConcurrentHashMap<>();
        cacheUpdateInterval = moxy.config().topPostsCacheTime();
        executor =
            new ThreadPoolExecutor(
                moxy.config().workers(),
                moxy.config().workers(),
                1,
                TimeUnit.DAYS,
                new LinkedBlockingQueue<>());
        DELAYED_EXECUTOR.newTimeout(
            new CacheGC(),
            cacheUpdateInterval * GC_THRESHOLD,
            TimeUnit.MILLISECONDS);
    }

    public static SimpleJsonParser getJsonParser() {
        SimpleJsonParser parser = JSON_PARSERS.poll();
        if (parser == null) {
            parser = new SimpleJsonParser();
        }
        return parser;
    }

    public static void freeJsonParser(final SimpleJsonParser parser) {
        JSON_PARSERS.add(parser);
    }

    public void dropCacheEntry(final String key) {
        cache.remove(key);
    }

    @Override
    public void handle(final ProxySession session) throws HttpException {
        String uri = session.request().getRequestLine().getUri();
        final TopPostsRequestContext topPostsContext =
            new TopPostsRequestContext(session, moxy.config());
        final ChatsRequestContext chatsContext =
            new ChatsRequestContext(session, topPostsContext);

        sendRequest(
            chatsContext,
            session,
            new ChatsCallback(
                session,
                topPostsContext));
    }

    private void procceedMessagesRequest(
        final ProxySession session,
        final TopPostsRequestContext topPostsContext,
        final JsonObject chatsResponse)
        throws HttpException
    {
        final MessagesRequestContext messagesContext =
            new MessagesRequestContext(session, topPostsContext);
        final DoubleFutureCallback<JsonObject, List<ChatsTopMessages>>
            mergedCallback =
                new DoubleFutureCallback<>(
                    new ChatsAndMessagesCallback(session, topPostsContext));

        mergedCallback.first().completed(chatsResponse);

        if (topPostsContext.skipMessagesRequest) {
            mergedCallback.second().completed(new ArrayList<>());
        } else {
            MultiFutureCallback<ChatsTopMessages> multiCallback =
                new MultiFutureCallback<>(mergedCallback.second());
            for (RequestContext context: messagesContext.subContexts()) {
                sendMessagesRequest(
                    context,
                    session,
                    multiCallback.newCallback());
            }
            multiCallback.done();
        }
    }

    public void sendMessagesRequest(
        final RequestContext context,
        final ProxySession session,
        final FutureCallback<ChatsTopMessages> callback)
        throws HttpException
    {
        String cacheKey = context.cacheKey();
        ChatsTopMessages topMessages = cache.get(cacheKey);
        if (topMessages != null
            && !topMessages.expired(cacheUpdateInterval))
        {
            topMessages.updateLastUsed();
            callback.completed(topMessages);
        } else {
            sendRequest(
                context,
                session,
                new JsonToTopMessagesCallback(
                    callback,
                    context,
                    session,
                    cacheKey));
        }
    }

    public void sendRequest(
        final RequestContext context,
        final ProxySession session,
        final FutureCallback<JsonObject> callback)
        throws HttpException
    {
        AsyncClient client = moxy.searchClient().adjust(session.context());
        UniversalSearchProxyRequestContext requestContext =
            new PlainUniversalSearchProxyRequestContext(
                context.user(),
                null,
                context.context.allowLaggingHosts,
                client,
                session.logger());
        QueryConstructor query = context.query();
        query.append(SERVICE, context.service());
        moxy.sequentialRequest(
            session,
            requestContext,
            new BasicAsyncRequestProducerGenerator(query.toString()),
            context.context.failoverDelay,
            context.context.localityShuffle,
//            JsonAsyncTypesafeDomConsumerFactory.OK,
            AsyncStringConsumerFactory.OK,
            session.listener().createContextGeneratorFor(client),
            new JsonParseCallback(
                callback,
                executor,
                session,
                context.getClass().getName()));
    }

    @Override
    public String toString() {
        return "Messenger top messages for selected chats handler: "
            + "https://wiki.yandex-team.ru/ps/documentation/"
            + "moxy#tophosts";
    }

    private static final ReusableCodedInputStream getTlsCodedInputStream(
        final int size)
        throws InvalidProtocolBufferException
    {
        ReusableCodedInputStream rcis = CODED_INPUT_TLS.get();
        rcis.reset();
        rcis.checkSize(size);
        return rcis;
    }

    private static class TopPostsRequestContext {
        private final int length;
        private final String get;
        private final String sort;
        private final String asc;
        private final boolean localityShuffle;
        private final boolean allowLaggingHosts;
        private final long failoverDelay;
        private final int daysDepth;
        private final boolean filterEmptyText;
        private final Collection<String> chatIds;
        private final Collection<String> types;
        private final Collection<String> moderationActions;
        private final boolean protobufOut;
        private final Long uid;
        private final boolean skipMessagesRequest;
        private int postCount = 0;

        TopPostsRequestContext(
            final ProxySession session,
            final ImmutableMoxyConfig config)
            throws HttpException
        {
            this.length = session.params().getInt("length2", DEFAULT_LENGTH);
            String get =
                session.params().getString(GET, null);
            this.get = get;
            sort =
                session.params().getString("sort", MESSAGE_SEQ_NO);
            asc = session.params().getString("asc", "true");
            this.uid = session.params().getLong("uid", null);
            this.localityShuffle = session.params().getBoolean(
                "locality-shuffle",
                config.topPostsLocalityShuffle());
            this.failoverDelay = session.params().getLong(
                "failover-delay",
                config.topPostsFailoverDelay());
            this.allowLaggingHosts = session.params().getBoolean(
                "allow-lagging-hosts",
                ALLOW_LAGGING_HOSTS);
            this.chatIds = session.params().getAll(
                "chat-ids",
                new HashSet<>(),
                SET_PARSER);
            this.types = session.params().getAll(
                "types",
                DEFAULT_MESSAGE_TYPES,
                SET_PARSER);
            this.moderationActions = session.params().getAll(
                "moderation-actions",
                Collections.singleton("keep"),
                SET_PARSER);
            this.skipMessagesRequest =
                session.params().getBoolean("skip-messages-request", false);
            this.daysDepth = session.params().getInt("days-depth", 5);
            this.protobufOut = session.params().getBoolean("proto", false);
            this.filterEmptyText = session.params().getBoolean(
                "filter-empty-text",
                true);
        }

        public int length() {
            return length;
        }

        public String sort() {
            return sort;
        }

        public String get() {
            return get;
        }

        public String asc() {
            return asc;
        }

        public Collection<String> chatIds() {
            return chatIds;
        }

        public Collection<String> types() {
            return types;
        }

        public int daysDepth() {
            return daysDepth;
        }

        public Collection<String> moderationActions() {
            return moderationActions;
        }

        public boolean proto() {
            return protobufOut;
        }
    }

    private static abstract class RequestContext {
        private TopPostsRequestContext context;

        RequestContext(
            final ProxySession session,
            final TopPostsRequestContext context)
            throws HttpException
        {
            this.context = context;
        }

        protected abstract String cacheKey();

        protected abstract User user();

        protected abstract String service();

        protected QueryConstructor query() throws HttpException {
            return createQuery(context);
        }

        protected abstract QueryConstructor createQuery(
            final TopPostsRequestContext context)
            throws HttpException;
    }

    private class MessagesRequestContext {
        private List<ShardMessagesRequestContext> subContexts;

        MessagesRequestContext(
            final ProxySession session,
            final TopPostsRequestContext context)
            throws HttpException
        {
            Map<SearchMapShard, List<String>> shardMap = new HashMap<>();
            Map<SearchMapShard, User> userMap = new HashMap<>();
            for (String chatId: context.chatIds()) {
                Prefix prefix = new StringPrefix(chatId);
                User user = new User(messagesService, prefix);
                SearchMapShard shard = moxy.searchMap().apply(user);
                List<String> chats = shardMap.get(shard);
                if (chats == null) {
                    chats = new ArrayList<>(2);
                    shardMap.put(shard, chats);
                    userMap.put(shard, user);
                }
                chats.add(chatId);
            }
            subContexts = new ArrayList<>(shardMap.size());
            for (Map.Entry<SearchMapShard, List<String>> entry
                : shardMap.entrySet())
            {
                User user = userMap.get(entry.getKey());
                session.logger().info("SubContext: prefixed, user: "
                    + user + ", shard:  " + entry.getKey());
                subContexts.add(
                    new ShardMessagesRequestContext(
                        session,
                        context,
                        user,
//                        entry.getKey(),
                        entry.getValue()));
            }
        }

        public List<ShardMessagesRequestContext> subContexts() {
            return subContexts;
        }
    }

    protected static void applyTimestampFilter(
        final String field,
        final StringBuilder sb,
        final int daysDepth)
    {
        long time = System.currentTimeMillis();
        sb.append(field);
        sb.append(":(");
        sb.append(time * MS_PER_MICRO);
        char sep = ' ';
        //we should select daysDepth + 1 days and than trim then by
        //docprocessor to emulate 24h limit
        for (int i = 0; i < daysDepth; i++) {
            sb.append(sep);
            time -= MILLIS_PER_DAY;
            sb.append(time * MS_PER_MICRO);
        }
        sb.append(')');
    }

    protected static void applyTimestampPostFilter(
        final QueryConstructor query,
        final String field,
        final int daysDepth)
        throws BadRequestException
    {
        long timeLimit = System.currentTimeMillis()
            - (daysDepth * MILLIS_PER_DAY);
        timeLimit *= MS_PER_MICRO;
        query.append(
            "postfilter",
            field + " >= " + timeLimit);
    }

    private class ShardMessagesRequestContext extends RequestContext {
        private final User user;
        private final List<String> chatIds;
        private final String cacheKey;

        //CSOFF: ParameterNumber
        ShardMessagesRequestContext(
            final ProxySession session,
            final TopPostsRequestContext context,
            final User user,
            final List<String> chatIds)
            throws HttpException
        {
            super(session, context);
            this.user = user;
            this.chatIds = chatIds;
            this.cacheKey =
                StringUtils.join(chatIds, ',')
                + context.sort()
                + "_" + context.asc()
                + "_" + context.daysDepth
                + "_" + context.length
                + "_" + context.filterEmptyText
                + "_" + StringUtils.join(context.types, ' ');
        }
        //CSON: ParameterNumber

        @Override
        protected String service() {
            return messagesService;
        }

        @Override
        protected String cacheKey() {
            return cacheKey;
        }

        @Override
        protected QueryConstructor createQuery(
            final TopPostsRequestContext context)
            throws HttpException
        {
            String get =
                MESSAGE_ID + ',' + MESSAGE_CHAT_ID
                + ',' + MESSAGE_DATA + ',' + MESSAGE_SEQ_NO
                + ',' + MESSAGE_TIMESTAMP
                + ',' + MESSAGE_RCA_DATA;
            if (context.get() != null) {
                get += ',' + context.get();
            }
            QueryConstructor query = new QueryConstructor(
                "/search?json-type=dollar&IO_PRIO=0&sort="
                    + context.sort()
                    + "&asc=" + context.asc()
//                    + "&length="
//                    + context.length()
                    + "&get=" + get);
            query.append(
                TEXT,
                searchText(context));
            for (String chatId: chatIds) {
                query.append(PREFIX, chatId);
            }
            query.append("collector", "sorted");
            query.append("group", MESSAGE_CHAT_ID);
            long timeLimit = System.currentTimeMillis()
                - (context.daysDepth * MILLIS_PER_DAY);
            timeLimit *= MS_PER_MICRO;
            query.append(
                "postfilter",
                MESSAGE_TIMESTAMP + " >= " + timeLimit);
            if (context.filterEmptyText) {
                query.append("dp", "equals(message_text,,  text_empty)");
                query.append("postfilter", "text_empty != 1");
            }
            return query;
        }

        private String searchText(
            final TopPostsRequestContext context)
        {
            final StringBuilder sb = new StringBuilder();
            long time = System.currentTimeMillis();
            sb.append("message_day_p:(");
            sb.append(time * MS_PER_MICRO);
            char sep = ' ';
            //we should select daysDepth + 1 days and than trim then by
            //docprocessor to emulate 24h limit
            for (int i = 0; i < context.daysDepth; i++) {
                sb.append(sep);
                time -= MILLIS_PER_DAY;
                sb.append(time * MS_PER_MICRO);
            }
            sb.append(')');
            if (context.types != null) {
                sb.append(" AND type_p:(");
                sb.append(StringUtils.join(context.types, ' '));
                sb.append(')');
            }
            if (context.moderationActions != null) {
                sb.append(" AND message_moderation_action:(");
                sb.append(StringUtils.join(context.moderationActions, ' '));
                sb.append(')');
            }
            return new String(sb);
        }

        @Override
        protected User user() {
            return user;
        }
    }

    private class ChatsRequestContext extends RequestContext {
        ChatsRequestContext(
            final ProxySession session,
            final TopPostsRequestContext context)
            throws HttpException
        {
            super(session, context);
        }

        @Override
        protected String service() {
            return chatsService;
        }

        @Override
        public String cacheKey() {
            return toString();
        }

        @Override
        protected QueryConstructor createQuery(
            final TopPostsRequestContext context)
            throws HttpException
        {
            QueryConstructor query = new QueryConstructor(
                "/search?json-type=dollar&IO_PRIO=0"
                + "&get=chat_id,chat_name,chat_description,chat_invite_hash"
                + ",chat_avatar_id,chat_members");
            query.append(
                TEXT,
                searchText(context));
            query.append(PREFIX, ZERO);
            if (context.uid != null) {
                query.append(DP, "const(" + context.uid + " user_uid)");
                query.append(
                    DP,
                    "left_join(user_uid,user_uid,,user_id user_id)");
                query.append(DP, "const(user_chats_ id_prefix)");
                query.append(DP, "concat(id_prefix,user_id user_chats_id)");
                query.append(DP, "left_join(user_chats_id,id,,user_chats user_chats)");
                query.append(GET, "user_id,user_chats");
            }
            applyTimestampPostFilter(
                query,
                ChatFields.LAST_MESSAGE_TIMESTAMP.global(),
                context.daysDepth);

            return query;
        }

        private String searchText(
            final TopPostsRequestContext context)
        {
            final StringBuilder sb = new StringBuilder();
            sb.append("chat_show_on_morda:true");
            sb.append(" AND ");
            applyTimestampFilter(
                ChatFields.LAST_MESSAGE_DAY.global(),
                sb,
                context.daysDepth);

            return new String(sb);
        }

        @Override
        protected User user() {
            return new User(chatsService, new LongPrefix(0));
        }
    }

    private class ChatsCallback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final TopPostsRequestContext context;

        ChatsCallback(
            final ProxySession session,
            final TopPostsRequestContext context)
            throws HttpException
        {
            super(session);
            this.context = context;
        }

        @Override
        public void completed(final JsonObject result) {
            try {
                JsonList hits = result.asMap().getList("hitsArray");
                Set<String> chatIds = new HashSet<>();
                for (JsonObject o: hits) {
                    JsonMap hit = o.asMap();
                    String chatId = hit.getString("chat_id");
                    chatIds.add(chatId);
                }
                if (chatIds.size() == 0) {
                    String msg =
                        "No chats found with show_on_morda flag";
                    session.logger().severe(msg);
                    session.response(HttpStatus.SC_NOT_FOUND, msg);
                } else {
                    context.chatIds.addAll(chatIds);
                    procceedMessagesRequest(
                        session,
                        context,
                        result);
                }
            } catch (HttpException|JsonException e) {
                failed(e);
            }
        }
    }

    private class ChatsAndMessagesCallback
        extends AbstractProxySessionCallback<
            Map.Entry<JsonObject, List<ChatsTopMessages>>>
    {
        private final JsonType jsonType;
        private final TopPostsRequestContext context;

        ChatsAndMessagesCallback(
            final ProxySession session,
            final TopPostsRequestContext context)
            throws HttpException
        {
            super(session);
            this.context = context;
            this.jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
        }

        @Override
        public void failed(final Exception e) {
            dropCacheEntry(session.request().getRequestLine().getUri());
            super.failed(e);
        }

        @Override
        public void completed(
            final Map.Entry<JsonObject, List<ChatsTopMessages>> result)
        {
            long startTime = System.currentTimeMillis();
            SimpleJsonParser jsonParser = getJsonParser();
            try {
                JsonMap chatsResult = result.getKey().asMap();
                JsonList hits = chatsResult.getList(HITS_ARRAY);
                Map<String, Chat> chatMap = new HashMap<>();
                for (JsonObject obj: hits) {
                    JsonMap hit = obj.asMap();
                    Chat chat = new Chat(hit);
                    chatMap.put(chat.chatId, chat);
                }
                for (ChatsTopMessages shardResult: result.getValue()) {
                    shardResult.updateLastUsed();
                    for (Map.Entry<String, List<Message>> entry:
                        shardResult.entrySet())
                    {
                        String chatId = entry.getKey();
                        Chat chat = chatMap.get(chatId);
                        if (chat == null) {
                            session.logger().info("Chat not found: " + chatId);
                            continue;
                        }
                        chat.addMessages(entry.getValue());
                    }
                }
                Iterator<Map.Entry<String, Chat>> iterator = chatMap.entrySet().iterator();
                int total = 0;
                while (iterator.hasNext()) {
                    Map.Entry<String, Chat> entry = iterator.next();
                    Chat chat = entry.getValue();
                    if (chat.messages.size() <= 0) {
                        session.logger().info("Skipping chat - no messages, " + chat.chatId);
                        iterator.remove();
                    } else {
                        session.logger().info("Chat "
                            + chat.chatId + ", messagesSelected: "
                            + chat.messages.size());
                        total += chat.messages.size();
                    }
                }

                session.logger().info("Total messages: " + total
                    + ", time: " + (System.currentTimeMillis() - startTime));
                startTime = System.currentTimeMillis();
                HttpEntity response;
                if (context.proto()) {
                    response = generateProtoResponse(chatMap);
                } else {
                    response = generateJsonResponse(chatMap);
                }
                session.logger().info("Generate response time: "
                    + (System.currentTimeMillis() - startTime));
                startTime = System.currentTimeMillis();
                session.response(HttpStatus.SC_OK, response);
                moxy.consumePostCount(context.postCount);
//            } catch (IOException e) {
            } catch (IOException | JsonException e) {
                session.logger().log(Level.SEVERE, "JsonParse exception", e);
                failed(e);
            } finally {
                freeJsonParser(jsonParser);
                session.logger().info("Callback time: "
                    + (System.currentTimeMillis() - startTime));
            }
        }

        private HttpEntity generateProtoResponse(
            final Map<String, Chat> chatMap)
            throws IOException
        {
            TTopPostsResponse.Builder builder =
                TTopPostsResponse.newBuilder();
            int[] postCount = new int[1];
            for (Chat chat: chatMap.values()) {
                builder.addTopPosts(chat.toProto(context.length, postCount));
            }
            context.postCount = postCount[0];
            CityHashingArrayOutputStream out = OUT_TLS.get();
            out.reset();
            CodedOutputStream googleOut = CodedOutputStream.newInstance(out);
            builder.build().writeTo(googleOut);
            googleOut.flush();

            byte[] postData = out.toByteArrayWithVersion(VERSION);
            return new NByteArrayEntity(postData);
        }

        private HttpEntity generateJsonResponse(
            final Map<String, Chat> chatMap)
            throws IOException, JsonException
        {
            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = jsonType.create(sbw)) {
                writer.startArray();
                for (Chat chat: chatMap.values()) {
                    context.postCount += chat.writeJson(writer, context.length);
                }
                writer.endArray();
            }
            return new NStringEntity(
                sbw.toString(),
                ContentType.APPLICATION_JSON.withCharset(
                    session.acceptedCharset()));
        }
    }

    private static class Chat {
        private final String chatId;
        private final String name;
        private final String description;
        private final String inviteHash;
        private final String avatarId;
        private final int memberCount;
        private final boolean isMember;
        private final List<Message> messages;

        Chat(final JsonMap json) throws JsonException {
            chatId = json.getOrNull(CHAT_ID);
            name = json.getOrNull(CHAT_NAME);
            description = json.getOrNull(CHAT_DESCRIPTION);
            inviteHash = json.getOrNull(CHAT_INVITE_HASH);
            avatarId = json.getOrNull(CHAT_AVATAR_ID);
            String members = json.getOrNull(CHAT_MEMBERS);
            String userChats = json.getOrNull(UserChats.CHATS.stored());
            String guid = json.getOrNull(USER_ID);
            if (members == null) {
                memberCount = 0;
                isMember = false;
            } else {
                String[] membersArray = members.split("\n");
                memberCount = membersArray.length;
                if (guid != null
                    && Arrays.stream(membersArray).anyMatch(guid::equals))
                {
                    isMember = true;
                } else if (userChats != null && !userChats.isEmpty() && chatId != null){
                    String[] userChatsArray = userChats.split("\n");
                    boolean isMemberByUserChats = false;
                    for (String chat: userChatsArray) {
                        if (chatId.equalsIgnoreCase(chat)) {
                            isMemberByUserChats = true;
                            break;
                        }
                    }

                    isMember = isMemberByUserChats;
                } else {
                    isMember = false;
                }
            }
            messages = new ArrayList<>();
        }

        public void addMessage(final Message message) {
            messages.add(message);
        }

        public void addMessages(final Collection<Message> messages) {
            this.messages.addAll(messages);
        }

        public TTopPosts toProto(final int length, final int[] postCount) {
            TTopPosts.Builder builder = TTopPosts.newBuilder();
            builder.setChatId(chatId);
            if (name != null) {
                builder.setName(name);
            }
            if (description != null) {
                builder.setDescription(description);
            }
            if (inviteHash != null) {
                builder.setInviteHash(inviteHash);
            }
            if (avatarId != null) {
                builder.setAvatarId(avatarId);
            }
            builder.setMemberCount(memberCount);
            builder.setIsMember(isMember);
            Collections.<Message>sort(messages);
            int i = 0;
            for (Message message: messages) {
                TServerMessage messageProto = message.serverMessageProto();
                if (messageProto != null) {
                    if (i++ >= length) {
                        break;
                    }
                    builder.addMessages(messageProto);
                    postCount[0]++;
                    if (message.hasImage) {
                        builder.addPreviews(
                            TPostPreviewInfo.newBuilder()
                                .setUrl(message.imageUrl)
                                .setWidth(message.imageWidth)
                                .setHeight(message.imageHeight)
                                .setMessageTimestamp(message.timestamp)
                                .build());
                    }
                }
            }
            return builder.build();
        }

        public int writeJson(final JsonWriter writer, final int length)
            throws JsonException, IOException
        {
            writer.startObject();

            writer.key(CHAT_ID);
            writer.value(chatId);

            writer.key(CHAT_NAME);
            writer.value(name);

            writer.key(CHAT_DESCRIPTION);
            writer.value(description);

            writer.key(CHAT_INVITE_HASH);
            writer.value(inviteHash);

            writer.key(CHAT_AVATAR_ID);
            writer.value(avatarId);

            writer.key(CHAT_MEMBER_COUNT);
            writer.value(memberCount);

            writer.key("user_is_member");
            writer.value(isMember);

            writer.key(TOP_MESSAGES);
            writer.startArray();
            Collections.sort(messages);
            int i = 0;
            for (Message message: messages) {
                if (i++ >= length) {
                    break;
                }
                message.writeJson(writer);
            }
            writer.endArray();
            writer.endObject();
            return i;
        }
    }

    private static class Message implements Comparable<Message> {
//        private final String messageId;
        private TOutMessage message;
        private final long seqNo;
        private final long timestamp;
        private final boolean hasImage;
        private final int imageWidth;
        private final int imageHeight;
        private final String imageUrl;
        private final String snippet;

        Message(
            final JsonMap json,
            final SimpleJsonParser parser)
            throws JsonException
        {
//            messageId = json.getString(MESSAGE_ID);
            String data = json.get(MESSAGE_DATA).asStringOrNull();
            seqNo = json.getLong(MESSAGE_SEQ_NO, 0L);
            timestamp = json.getLong(MESSAGE_TIMESTAMP, 0L);
            if (data != null) {
                try {
                    ReusableCodedInputStream rcis =
                        getTlsCodedInputStream(data.length() >> 1);
                    byte[] bytes = rcis.array;
                    UnhexStrings.unhex(data, bytes);
//                    message = TOutMessage.parseFrom(bytes);
//                    ByteString bs = ByteString.copyFrom(bytes);
//                    message = TOutMessage.parseFrom(bs);
                    message = TOutMessage.parseFrom(rcis.cis);
                } catch (IOException e) {
                    throw new JsonException(e);
                }
            } else {
                message = null;
            }
            boolean hasImage = false;
            int imageWidth = 0;
            int imageHeight = 0;
            String imageUrl = null;
            String snippet = null;
            String rcaData = json.get(MESSAGE_RCA_DATA).asStringOrNull();
            if (rcaData != null) {
                JsonObject rcaObject = parser.parse(rcaData);
                JsonMap rcaMap = rcaObject.asMap();
                JsonMap mainImage = rcaMap.getMapOrNull("main_image");
                if (mainImage != null) {
                    Integer height = mainImage.getInt("height", null);
                    Integer width = mainImage.getInt("width", null);
                    if (height != null && width != null
                        && Math.min(height, width) > MIN_IMAGE_SIZE)
                    {
                        hasImage = true;
                        imageUrl = mainImage.getString("src");
                        imageWidth = width;
                        imageHeight = height;
                        snippet = rcaMap.get(SNIPPET).asStringOrNull();
                    }
                }
            }
            this.hasImage = hasImage;
            this.imageWidth = imageWidth;
            this.imageHeight = imageHeight;
            this.imageUrl = imageUrl;
            this.snippet = snippet;
        }

        @Override
        public int hashCode() {
            return Long.hashCode(seqNo);
        }

        @Override
        public boolean equals(final Object o) {
            if (o instanceof Message) {
                Message other = (Message) o;
                return other.seqNo == seqNo;
            }
            return false;
        }

        @Override
        public int compareTo(final Message o) {
            return Long.compare(o.seqNo, seqNo);
        }

        public TServerMessage serverMessageProto() {
            if (message != null) {
                return message.getServerMessage();
            } else {
                return null;
            }
        }

        public void writeJson(final JsonWriter writer)
            throws JsonException, IOException
        {
            writer.startObject();
//            if (message != null) {
            writer.key("has_image");
            writer.value(hasImage);
            if (hasImage) {
                writer.key("image_height");
                writer.value(imageHeight);
                writer.key("image_width");
                writer.value(imageWidth);
                writer.key("image_url");
                writer.value(imageUrl);
                writer.key(SNIPPET);
                writer.value(snippet);
            }
            writer.key(MESSAGE_DATA);
            ProtoUtils.protoToJson(message).writeValue(writer);
//            }
            writer.endObject();
        }
    }

    private static final class SimpleJsonParser {
        private final JsonParser parser;
        private final BasicGenericConsumer<JsonObject, JsonException> consumer;

        SimpleJsonParser() {
            consumer = new BasicGenericConsumer<>();
            parser =
                new JsonParser(new StackContentHandler(
                    new TypesafeValueContentHandler(consumer)));
        }

        public JsonObject parse(final String json) throws JsonException {
            parser.parse(json);
            return consumer.get();
        }
    }

    private static final class JsonParseCallback
        extends AbstractFilterFutureCallback<String, JsonObject>
    {
        private final ThreadPoolExecutor executor;
        private final ProxySession session;
        private final String tag;

        JsonParseCallback(
            final FutureCallback<JsonObject> callback,
            final ThreadPoolExecutor executor,
            final ProxySession session,
            final String tag)
        {
            super(callback);
            this.executor = executor;
            this.session = session;
            this.tag = tag;
        }

        @Override
        public void completed(final String result) {
            session.logger().info("Response received: " + tag);
            executor.execute(new JsonParseTask(result, callback, session));
        }
    }

    private static final class JsonParseTask implements Runnable {
        private final FutureCallback<? super JsonObject> callback;
        private final String msg;
        private final ProxySession session;

        public JsonParseTask(
            final String msg,
            final FutureCallback<? super JsonObject> callback,
            final ProxySession session)
        {
            this.msg = msg;
            this.callback = callback;
            this.session = session;
        }

        @Override
        public void run() {
            SimpleJsonParser parser = getJsonParser();
            long time = System.currentTimeMillis();
            try {
                JsonObject json = parser.parse(msg);
                session.logger().info("Json parse time: "
                    + (System.currentTimeMillis() - time));
                callback.completed(json);
            } catch (Exception e) {
                callback.failed(e);
            } finally {
                freeJsonParser(parser);
            }
        }
    }

    private static final class ReusableCodedInputStream {
        private CodedInputStream cis;
        private LimitingByteArrayInputStream bis;
        private byte[] array;

        ReusableCodedInputStream() {
            allocate(DEFAULT_BYTE_ARRAY_LENGTH);
        }

        private void allocate(final int size) {
            array = new byte[size];
            bis = new LimitingByteArrayInputStream(array);
            bis.mark(0);
            cis = CodedInputStream.newInstance(bis);
        }

        public void checkSize(final int size)
            throws InvalidProtocolBufferException
        {
            if (array.length < size) {
                allocate(size << 1);
            }
            bis.limit(size);
            cis.setSizeLimit(size);
            cis.resetSizeCounter();
        }

        public void reset() {
            bis.reset();
            bis.mark(0);
        }
    }

    private static final class LimitingByteArrayInputStream
        extends ByteArrayInputStream
    {
        LimitingByteArrayInputStream(final byte[] array) {
            super(array);
        }

        public void limit(final int limit) {
            this.count = limit;
        }
    }

    private final class JsonToTopMessagesCallback
        extends AbstractFilterFutureCallback<JsonObject, ChatsTopMessages>
    {
        private final RequestContext context;
        private final ProxySession session;
        private final String cacheKey;

        JsonToTopMessagesCallback(
            final FutureCallback<ChatsTopMessages> callback,
            final RequestContext context,
            final ProxySession session,
            final String cacheKey)
        {
            super(callback);
            this.context = context;
            this.session = session;
            this.cacheKey = cacheKey;
        }

        @Override
        public void completed(final JsonObject response) {
            SimpleJsonParser jsonParser = getJsonParser();
            long startTime = System.currentTimeMillis();
            try {
                JsonList shardHits =
                    response.asMap().getList(HITS_ARRAY);
                ChatsTopMessages topMessages =
                    new ChatsTopMessages(
                        context,
                        session,
                        cacheKey,
                        cacheUpdateInterval);
                for (JsonObject obj: shardHits) {
                    JsonMap hit = obj.asMap();
                    Message message = new Message(hit, jsonParser);
                    String chatId = hit.getOrNull(MESSAGE_CHAT_ID);
                    if (chatId == null) {
                        continue;
                    }
                    topMessages.addMessage(chatId, message);
                    JsonList mergedList = hit.getListOrNull("merged_docs");
                    if (mergedList != null) {
                        for (JsonObject merged: mergedList) {
                            message = new Message(
                                merged.asMap(),
                                jsonParser);
                            topMessages.addMessage(chatId, message);
                        }
                    }
                }
                cache.put(cacheKey, topMessages);
                callback.completed(topMessages);
            } catch (HttpException | JsonException e) {
                session.logger().log(Level.SEVERE, "JsonParse exception", e);
                failed(e);
            } finally {
                freeJsonParser(jsonParser);
                session.logger().info("Messages callback time: "
                    + (System.currentTimeMillis() - startTime));
            }
        }
    }

    private class ChatsTopMessages
        extends HashMap<String, List<Message>>
        implements TimerTask, FutureCallback<ChatsTopMessages>
    {
        protected final long lastUpdate;
        protected final AtomicBoolean updateSent = new AtomicBoolean(false);
        private final RequestContext context;
        private final long updateInterval;
        private final ProxySession session;
        private final String cacheKey;
        private volatile long lastUsed;
        private final Logger logger;

        ChatsTopMessages(
            final RequestContext context,
            final ProxySession session,
            final String cacheKey,
            final long updateInterval)
            throws HttpException
        {
            this.context = context;
            this.updateInterval = updateInterval;
            this.session =
                new BasicProxySession(
                    moxy,
                    session.exchange(),
                    session.context());
            this.logger = session.logger().replacePrefix("CacheUpdate");
            this.cacheKey = cacheKey;
            lastUpdate = System.currentTimeMillis();
            DELAYED_EXECUTOR.newTimeout(
                this,
                updateInterval,
                TimeUnit.MILLISECONDS);
        }

        public void addMessage(final String chatId, final Message message) {
            List<Message> messages = get(chatId);
            if (messages == null) {
                messages = new ArrayList<>();
                put(chatId, messages);
            }
            messages.add(message);
        }

        public boolean expired(final long updateInterval) {
            return ((System.currentTimeMillis()
                    - (updateInterval * EXPIRE_THRESHOLD)) > lastUpdate);
        }

        public void updateLastUsed() {
            lastUsed = System.currentTimeMillis();
        }

        public boolean shouldGC() {
            long time = System.currentTimeMillis();
            return ((time - (updateInterval * GC_THRESHOLD) > lastUpdate)
                    || (time - (updateInterval * GC_THRESHOLD) > lastUsed));
        }

        @Override
        public void run(final Timeout timeout) {
            ChatsTopMessages cached = cache.get(cacheKey);
            if (cached != this) {
                logger.info("Entry replaced concurrently, discarding: "
                    + cacheKey);
                return;
            }
            final long time = System.currentTimeMillis();
            if (time >= lastUpdate + updateInterval) {
                if (lastUsed + (updateInterval * EXPIRE_THRESHOLD) < time) {
                    logger.info("Entry expired, removing: " + cacheKey);
                    cache.remove(cacheKey);
                } else {
                    logger.info("Updating entry: " + cacheKey);
                    try {
                        sendRequest(
                            context,
                            session,
                            new JsonToTopMessagesCallback(
                                this,
                                context,
                                session,
                                cacheKey));
                    } catch (HttpException e) {
                        failed(e);
                    }
                }
            } else {
                logger.info("Sporadic wakeup, skipping");
                DELAYED_EXECUTOR.newTimeout(
                    this,
                    lastUpdate + updateInterval - time,
                    TimeUnit.MILLISECONDS);
            }
        }

        @Override
        public void completed(final ChatsTopMessages newTopMessages) {
            logger.info("Entry updated, replacing: " + cacheKey);
            newTopMessages.lastUsed = lastUsed;
//            cache.put(cacheKey, newTopMessages);
        }

        @Override
        public void cancelled() {
            logger.info("Entry update cancelled, removing: " + cacheKey);
            cache.remove(cacheKey);
        }

        @Override
        public void failed(final Exception e) {
            final long time = System.currentTimeMillis();
            if (lastUsed + (updateInterval * EXPIRE_THRESHOLD) < time) {
                logger.info("Entry update failed & expired, removing: "
                    + cacheKey);
                cache.remove(cacheKey);
            } else {
                logger.info("Entry update failed, rescheduling: "
                    + cacheKey);
               DELAYED_EXECUTOR.newTimeout(
                    this,
                    updateInterval,
                    TimeUnit.MILLISECONDS);
            }
        }
    }

    private class CacheGC implements TimerTask {
        @Override
        public void run(final Timeout timeout) {
            Iterator<Map.Entry<String, ChatsTopMessages>> iter =
                cache.entrySet().iterator();
            while (iter.hasNext()) {
                Map.Entry<String, ChatsTopMessages> entry = iter.next();
                if (entry.getValue().shouldGC()) {
                    moxy.logger().info("CacheGC: removing orphaned entry: "
                        + entry.getValue().cacheKey);
                    iter.remove();
                }
            }
            DELAYED_EXECUTOR.newTimeout(
                this,
                cacheUpdateInterval * GC_THRESHOLD,
                TimeUnit.MILLISECONDS);
        }
    }
}

