package ru.yandex.search.messenger.proxy;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import NMessengerProtocol.Message.TOutMessage;
import com.google.protobuf.InvalidProtocolBufferException;
import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.searchmap.SearchMap;
import ru.yandex.parser.searchmap.SearchMapRow;
import ru.yandex.parser.searchmap.SearchMapShard;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.messenger.proxy.config.ImmutableMoxyConfig;
import ru.yandex.search.messenger.proxy.suggest.rules.ProtoUtils;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.ShardPrefix;
import ru.yandex.search.prefix.StringPrefix;
import ru.yandex.search.proxy.universal.PlainUniversalSearchProxyRequestContext;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;
import ru.yandex.util.string.StringUtils;
import ru.yandex.util.string.UnhexStrings;

public class UserMessagesHandler implements ProxyRequestHandler {
    private static final int DEFAULT_LENGTH = 3;
    private static final String MESSAGE_SEQ_NO = "message_seq_no";
    private static final String GET = "get";
    private static final String DP = "dp";
    private static final String PREFIX = "prefix";
    private static final String SERVICE = "service";
    private static final String ZERO = "0";
    private static final String TEXT = "text";
    private static final String HITS_ARRAY = "hitsArray";
    private static final String USER_ID = "user_id";
    private static final String CHAT_ID = "chat_id";
    private static final String CHAT_NAME = "chat_name";
    private static final String CHAT_DESCRIPTION = "chat_description";
    private static final String MESSAGE_DATA = "message_data";
    private static final String MESSAGE_ID = "message_id";
    private static final String MESSAGE_CHAT_ID = "message_chat_id";
    private static final String MESSAGE_TIMESTAMP = "message_timestamp";
    private static final String TYPE = "type";
    private static final String TOP_MESSAGES = "chat_top_messages";
    private static final int MAX_CHATS_RESOLVE_COUNT = 2100;
    private static final int DEFAULT_MAX_CHATS = 100;
    private static final CollectionParser<
        String,
        Set<String>,
        Exception>
        SET_PARSER = new CollectionParser<>(
            NonEmptyValidator.INSTANCE,
            LinkedHashSet::new);

    private static final boolean ALLOW_LAGGING_HOSTS = true;

    private final Moxy moxy;
    private final String messagesService;
    private final String chatsService;
    private INum[] globalShards = new INum[0];
    private int searchMapVersion = -1;

    public UserMessagesHandler(final Moxy moxy) {
        this.moxy = moxy;
        messagesService = moxy.config().messagesService();
        chatsService = moxy.config().chatsService();
        calcGlobalShards();
    }

    private void calcGlobalShards() {
        int searchMapVersion = moxy.searchMap().version();
        if (searchMapVersion == this.searchMapVersion) {
            return;
        }
        SearchMapRow row =
            moxy.searchMap().row(moxy.config().messagesService());
        if (row == null) {
            globalShards = new INum[0];
        } else {
            Map<Integer, INum> mergeMap = new HashMap<>();
            for (int i = 0; i < SearchMap.SHARDS_COUNT; i++) {
                SearchMapShard shard = row.get(i);
                if (shard != null) {
                    Prefix prefix = new ShardPrefix(i);
                    User user = new User(
                        moxy.config().messagesService(),
                        prefix);
                    mergeMap.putIfAbsent(shard.iNum(), new INum(shard, user));
                }
            }
            globalShards = mergeMap.values().toArray(new INum[0]);
        }
        this.searchMapVersion = searchMapVersion;
    }

    @Override
    public void handle(final ProxySession session) throws HttpException {
        final UserMessagesRequestContext topPostsContext =
            new UserMessagesRequestContext(session, moxy.config());
        final ChatsRequestContext chatsContext =
            new ChatsRequestContext(session, topPostsContext);

        if (topPostsContext.guid == null) {
            resolveGuid(chatsContext);
        } else {
            guidResolved(chatsContext);
        }
    }

    public void sendRequest(
        final RequestContext context,
        final ProxySession session,
        final FutureCallback<JsonObject> callback)
        throws HttpException
    {
        AsyncClient client = moxy.searchClient().adjust(session.context());
        UniversalSearchProxyRequestContext requestContext =
            new PlainUniversalSearchProxyRequestContext(
                context.user(),
                null,
                context.context.allowLaggingHosts,
                client,
                session.logger());
        QueryConstructor query = context.query();
        query.append(SERVICE, context.service());
        moxy.sequentialRequest(
            session,
            requestContext,
            new BasicAsyncRequestProducerGenerator(query.toString()),
            context.context.failoverDelay,
            context.context.localityShuffle,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.listener().createContextGeneratorFor(client),
            callback);
    }

    private void resolveGuid(final ChatsRequestContext chatsContext)
        throws HttpException
    {
        ResolveGuidRequestContext uidContext =
            new ResolveGuidRequestContext(
                chatsContext.session(),
                chatsContext.context());
        sendRequest(
            uidContext,
            chatsContext.session(),
            new GuidResolveCallback(chatsContext));
    }

    private void guidResolved(final ChatsRequestContext chatsContext)
        throws HttpException
    {
        final ProxySession session = chatsContext.session();
        if (chatsContext.context().chatIds.size() > 0) {
            session.logger().info(
                "chat-ids was specified, skipping chats resolve");
            resolveChatsInfo(session, chatsContext);
        } else {
            session.logger().info(
                "resolving chats");
            resolveChats(session, chatsContext);
        }
    }

    private void resolveChats(
        final ProxySession session,
        final ChatsRequestContext chatsContext)
        throws HttpException
    {
        MultiFutureCallback<List<String>> multiCallback =
            new MultiFutureCallback<>(
                new ChatsResolveCallback(chatsContext));
        for (INum iNum: globalShards) {
            session.logger().info("SubContext: global, user: "
                + iNum.user + ", shard: " + iNum.shard);
            sendRequest(
                new GlobalChatsSubRequestContext(
                    session,
                    chatsContext.context(),
                    iNum),
                session,
                new GlobalChatsSubSearchCallback(
                    multiCallback.newCallback(),
                    chatsContext));
        }
        multiCallback.done();
    }

    private void resolveChatsInfo(
        final ProxySession session,
        final ChatsRequestContext chatsContext)
        throws HttpException
    {
            sendRequest(
                chatsContext,
                session,
                new ChatsSearchCallback(chatsContext));
    }

    public void resolveMessages(
        final Map<String, Chat> chats,
        final ChatsRequestContext context)
        throws HttpException
    {
        final MessagesRequestContext messagesContext =
            new MessagesRequestContext(
                context.session(),
                context.context(),
                chats.keySet());
        MultiFutureCallback<JsonObject> multiCallback =
            new MultiFutureCallback<>(
                new MessagesSearchCallback(messagesContext, chats));
        for (RequestContext subContext: messagesContext.subContexts()) {
            sendRequest(
                subContext,
                context.session(),
                multiCallback.newCallback());
        }
        multiCallback.done();
    }

    @Override
    public String toString() {
        return "Messenger user messages handler: "
            + "https://wiki.yandex-team.ru/ps/documentation/"
            + "moxy#usermessages";
    }

    private static class UserMessagesRequestContext {
        private final int length;
        private final int maxChats;
        private final String get;
        private final String sort;
        private final String asc;
        private final boolean localityShuffle;
        private final boolean allowLaggingHosts;
        private final long failoverDelay;
        private final Collection<String> chatIds;
        private final Long uid;
        private String guid = null;
        private final boolean debug;
        private final boolean groupByChat;
        private final JsonType jsonType;
        private final String request;
        private final String namespace;
        private final boolean showPrivateChats;

        UserMessagesRequestContext(
            final ProxySession session,
            final ImmutableMoxyConfig config)
            throws HttpException
        {
            this.length = session.params().getInt("length", DEFAULT_LENGTH);
            String get =
                session.params().getString(GET, null);
            this.get = get;
            sort =
                session.params().getString("sort", "message_timestamp");
            asc = session.params().getString("asc", "false");
            this.maxChats =
                session.params().getInt("max-chats", DEFAULT_MAX_CHATS);
            this.request = session.params().getString("request", null);
            this.uid = session.params().getLong("uid", null);
            this.guid = session.params().getString("user-id", null);
            this.localityShuffle = session.params().getBoolean(
                "locality-shuffle",
                config.topPostsLocalityShuffle());
            this.failoverDelay = session.params().getLong(
                "failover-delay",
                config.topPostsFailoverDelay());
            this.allowLaggingHosts = session.params().getBoolean(
                "allow-lagging-hosts",
                ALLOW_LAGGING_HOSTS);
            this.chatIds = session.params().getAll(
                "chat-ids",
                new LinkedHashSet<>(),
                SET_PARSER);
            this.groupByChat = session.params().getBoolean(
                "group-by-chat",
                false);
            this.showPrivateChats = session.params().getBoolean(
                "show-private-chats",
                false);
            this.namespace = session.params().getString("namespace", null);
            this.debug = session.params().getBoolean("debug", false);
            this.jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
            if (uid == null && guid == null) {
                throw new BadRequestException("empty uid and guid");
            }
        }

        public boolean showPrivateChats() {
            return showPrivateChats;
        }

        public int length() {
            return length;
        }

        public String sort() {
            return sort;
        }

        public String get() {
            return get;
        }

        public String asc() {
            return asc;
        }
    }

    private static abstract class RequestContext {
        private UserMessagesRequestContext context;
        private final ProxySession session;

        RequestContext(
            final ProxySession session,
            final UserMessagesRequestContext context)
            throws HttpException
        {
            this.context = context;
            this.session = session;
        }

        public ProxySession session() {
            return session;
        }

        public UserMessagesRequestContext context() {
            return context;
        }

        protected abstract User user();

        protected abstract String service();

        protected QueryConstructor query() throws HttpException {
            return createQuery(context);
        }

        protected abstract QueryConstructor createQuery(
            final UserMessagesRequestContext context)
            throws HttpException;
    }

    private class MessagesRequestContext {
        private List<ShardMessagesRequestContext> subContexts;
        private final ProxySession session;
        private final UserMessagesRequestContext context;

        MessagesRequestContext(
            final ProxySession session,
            final UserMessagesRequestContext context,
            final Set<String> chatsIds)
            throws HttpException
        {
            this.session = session;
            this.context = context;
            Map<SearchMapShard, List<String>> shardMap = new HashMap<>();
            Map<SearchMapShard, User> userMap = new HashMap<>();
            for (String chatId: chatsIds) {
                Prefix prefix = new StringPrefix(chatId);
                User user = new User(messagesService, prefix);
                SearchMapShard shard = moxy.searchMap().apply(user);
                List<String> chats = shardMap.get(shard);
                if (chats == null) {
                    chats = new ArrayList<>(2);
                    shardMap.put(shard, chats);
                    userMap.put(shard, user);
                }
                chats.add(chatId);
            }
            subContexts = new ArrayList<>(shardMap.size());
            for (Map.Entry<SearchMapShard, List<String>> entry
                : shardMap.entrySet())
            {
                User user = userMap.get(entry.getKey());
                session.logger().info("SubContext: prefixed, user: "
                    + user + ", shard:  " + entry.getKey());
                subContexts.add(
                    new ShardMessagesRequestContext(
                        session,
                        context,
                        user,
//                        entry.getKey(),
                        entry.getValue()));
            }
        }

        public UserMessagesRequestContext context() {
            return context;
        }

        public ProxySession session() {
            return session;
        }

        public List<ShardMessagesRequestContext> subContexts() {
            return subContexts;
        }
    }

    private class ShardMessagesRequestContext extends RequestContext {
        private final User user;
        private final List<String> chatIds;

        //CSOFF: ParameterNumber
        ShardMessagesRequestContext(
            final ProxySession session,
            final UserMessagesRequestContext context,
            final User user,
            final List<String> chatIds)
            throws HttpException
        {
            super(session, context);
            this.user = user;
            this.chatIds = chatIds;
        }
        //CSON: ParameterNumber

        @Override
        protected String service() {
            return messagesService;
        }

        @Override
        protected QueryConstructor createQuery(
            final UserMessagesRequestContext context)
            throws HttpException
        {
            String get =
                MESSAGE_ID + ',' + MESSAGE_CHAT_ID
                + ',' + MESSAGE_DATA + ',' + MESSAGE_TIMESTAMP
                + ',' + MESSAGE_SEQ_NO + ',' + TYPE;
            if (context.get() != null) {
                get += ',' + context.get();
            }
            QueryConstructor query = new QueryConstructor(
                "/search?json-type=dollar&IO_PRIO=0&sort="
                    + context.sort()
                    + "&asc=" + context.asc()
                    + "&length="
                    + context.maxChats
                    + "&get=" + get);
            if (context.sort().equals(MESSAGE_TIMESTAMP)) {
                query.append("collector", "pruning(message_hour_p)");
            } else {
                query.append("collector", "sorted");
            }
            query.append(
                TEXT,
                searchText(context));
            for (String chatId: chatIds) {
                query.append(PREFIX, chatId);
            }
            query.append("group", MESSAGE_CHAT_ID);
            query.append("dp", "const(0 zero)");
            query.append("dp", "fallback(message_hid,zero message_filter)");
            if (!context.showPrivateChats()) {
                query.append(
                    "dp",
                    "contains_any(message_chat_id,0/0/,_ private_chat)");
                query.append(
                    "dp",
                    "sum(message_filter,private_chat message_filter)");
            }
            query.append("postfilter", "message_filter == 0");
            return query;
        }

        private String searchText(
            final UserMessagesRequestContext context)
        {
            final StringBuilder sb = new StringBuilder();
            sb.append("message_from_guid:");
            sb.append(context.guid);
            if (context.request != null) {
                sb.append(" AND ");
                sb.append("message_text_p:(");
                sb.append(context.request);
                sb.append(')');
            }
            return new String(sb);
        }

        @Override
        protected User user() {
            return user;
        }
    }

    private class ChatsRequestContext extends RequestContext {
        ChatsRequestContext(
            final ProxySession session,
            final UserMessagesRequestContext context)
            throws HttpException
        {
            super(session, context);
        }

        @Override
        protected String service() {
            return chatsService;
        }

        @Override
        protected QueryConstructor createQuery(
            final UserMessagesRequestContext context)
            throws HttpException
        {
            QueryConstructor query = new QueryConstructor(
                "/search?json-type=dollar&IO_PRIO=0"
                + "&get=chat_id,chat_name,chat_description");
            query.append(
                TEXT,
                searchText(context));
            query.append(PREFIX, ZERO);
            query.append("group", CHAT_ID);
            query.append("length", MAX_CHATS_RESOLVE_COUNT);
            query.append("sort", "chat_last_message_timestamp");
            if (context.uid != null) {
                query.append(DP, "const(" + context.uid + " user_uid)");
                query.append(
                    DP,
                    "left_join(user_uid,user_uid,,user_id user_id)");
                query.append(GET, USER_ID);
            }
            return query;
        }

        private String searchText(
            final UserMessagesRequestContext context)
        {
            final StringBuilder sb = new StringBuilder();
            if (context.chatIds.size() > 0) {
                sb.append("chat_id:(");
                sb.append(StringUtils.join(context.chatIds, ' '));
                sb.append(')');
            } else {
                sb.append("chat_members:(");
                sb.append(context.guid);
                sb.append(')');
            }
            if (context.namespace != null) {
                sb.append(" AND chat_namespace:" + context.namespace);
            }
            return new String(sb);
        }

        @Override
        protected User user() {
            return new User(chatsService, new LongPrefix(0));
        }
    }

    private class ResolveGuidRequestContext extends RequestContext {
        ResolveGuidRequestContext(
            final ProxySession session,
            final UserMessagesRequestContext context)
            throws HttpException
        {
            super(session, context);
        }

        @Override
        protected String service() {
            return chatsService;
        }

        @Override
        protected QueryConstructor createQuery(
            final UserMessagesRequestContext context)
            throws HttpException
        {
            QueryConstructor query = new QueryConstructor(
                "/search?json-type=dollar&IO_PRIO=1000"
                + "&get=user_id");
            query.append(
                TEXT,
                searchText(context));
            query.append(PREFIX, ZERO);
            query.append("length", 1);
            return query;
        }

        private String searchText(
            final UserMessagesRequestContext context)
        {
            final StringBuilder sb = new StringBuilder();
            sb.append("user_uid:(");
            sb.append(context.uid);
            sb.append(')');
            return new String(sb);
        }

        @Override
        protected User user() {
            return new User(chatsService, new LongPrefix(0));
        }
    }

    private class GlobalChatsSubRequestContext extends RequestContext {
        private final INum iNum;

        GlobalChatsSubRequestContext(
            final ProxySession session,
            final UserMessagesRequestContext context,
            final INum iNum)
            throws HttpException
        {
            super(session, context);
            this.iNum = iNum;
        }

        @Override
        protected String service() {
            return messagesService;
        }

        @Override
        protected QueryConstructor createQuery(
            final UserMessagesRequestContext context)
            throws HttpException
        {
            QueryConstructor query = new QueryConstructor(
                "/search?json-type=dollar&IO_PRIO=1000"
                + "&get=" + MESSAGE_CHAT_ID);
            query.append(
                TEXT,
                searchText(context));
            query.append("group", MESSAGE_CHAT_ID);
            query.append("merge_func", "none");
            query.append("length", MAX_CHATS_RESOLVE_COUNT);
            query.append("sort", "message_timestamp");
            if (!context.showPrivateChats()) {
                query.append(
                    "dp",
                    "contains_any(message_chat_id,0/0/,_ private_chat)");
                query.append("postfilter", "private_chat == 0");
            }
            return query;
        }

        private String searchText(
            final UserMessagesRequestContext context)
        {
            final StringBuilder sb = new StringBuilder();
            sb.append("message_from_guid:(");
            sb.append(context.guid);
            sb.append(')');
            return new String(sb);
        }

        @Override
        protected User user() {
            return iNum.user;
        }
    }

    private class GuidResolveCallback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final ChatsRequestContext context;

        GuidResolveCallback(final ChatsRequestContext context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(final JsonObject result) {
            try {
                JsonList hits = result.get("hitsArray").asList();
                String guid = null;
                if (hits.size() != 0) {
                    JsonMap hit = hits.get(0).asMap();
                    guid = hit.getString(USER_ID, null);
                }
                if (guid == null) {
                    context.session().response(
                        HttpStatus.SC_NOT_FOUND,
                        "Can't resolve GUID for UID: " + context.context().uid);
                } else {
                    context.context().guid = guid;
                    guidResolved(context);
                }
            } catch (HttpException | JsonException e) {
                failed(e);
            }
        }
    }

    private class ChatsSearchCallback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final ChatsRequestContext context;

        ChatsSearchCallback(final ChatsRequestContext context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(final JsonObject result) {
            try {
                JsonList hits = result.get("hitsArray").asList();
                if (context.context().debug) {
                    context.session().logger().fine(
                        "UserChatMembersSearch Lucene response: "
                            + JsonType.HUMAN_READABLE.toString(hits));
                }
                Map<String, Chat> chats;
                if (hits.size() == 0) {
                    chats = Collections.emptyMap();
                } else {
                    chats = new HashMap<>();
                    for (JsonObject item: hits) {
                        JsonMap hit = item.asMap();
                        String chatId = hit.getOrNull(CHAT_ID);
                        if (chatId != null) {
                            Chat chat = new Chat(hit);
                            chats.put(chatId, chat);
                        }
                    }
                }
                context.session().logger()
                    .info("Resolved chats filter size: " + chats.size());
                if (context.context().debug) {
                    context.session().logger().fine(
                        "ChatsFilter.set: " + chats);
                }
                resolveMessages(chats, context);
            } catch (HttpException | JsonException e) {
                failed(e);
            }
        }
    }

    private class GlobalChatsSubSearchCallback
        extends AbstractFilterFutureCallback<JsonObject, List<String>>
    {
        private final ChatsRequestContext context;

        GlobalChatsSubSearchCallback(
            final FutureCallback<List<String>> callback,
            final ChatsRequestContext context)
        {
            super(callback);
            this.context = context;
        }

        @Override
        public void completed(final JsonObject result) {
            try {
                JsonList hits = result.get("hitsArray").asList();
                if (context.context().debug) {
                    context.session().logger().fine(
                        "UserChatMembersSearch Lucene response: "
                            + JsonType.HUMAN_READABLE.toString(hits));
                }
                Set<String> chats;
                if (hits.size() == 0) {
                    chats = Collections.emptySet();
                } else {
                    chats = new HashSet<>();
                    for (JsonObject item: hits) {
                        JsonMap hit = item.asMap();
                        String chatId = hit.getOrNull(MESSAGE_CHAT_ID);
                        if (chatId != null) {
                            chats.add(chatId);
                        }
                    }
                }
                context.session().logger()
                    .info("Resolved chats filter size: " + chats.size());
                if (context.context().debug) {
                    context.session().logger().fine(
                        "ChatsFilter.set: " + chats);
                }
                callback.completed(new ArrayList<>(chats));
            } catch (JsonException e) {
                failed(e);
            }
        }
    }

    private class ChatsResolveCallback
        extends AbstractProxySessionCallback<List<List<String>>>
    {
        private final ChatsRequestContext context;

        ChatsResolveCallback(final ChatsRequestContext context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(final List<List<String>> results) {
            for (List<String> chats: results) {
                for (String chat: chats) {
                    context.context().chatIds.add(chat);
                }
            }
            context.session().logger().info("Resolved "
                + context.context().chatIds.size()
                + " through broadcast search");
            try {
                resolveChatsInfo(context.session(), context);
            } catch (HttpException e) {
                failed(e);
            }
        }
    }

    private static class MessagesSearchCallback
        extends AbstractProxySessionCallback<List<JsonObject>>
    {
        private final MessagesRequestContext context;
        private final Map<String, Chat> chats;

        MessagesSearchCallback(
            final MessagesRequestContext context,
            final Map<String, Chat> chats)
        {
            super(context.session());
            this.context = context;
            this.chats = chats;
        }

        @Override
        public void completed(final List<JsonObject> result) {
            try {
                for (JsonObject shardResult: result) {
                    JsonList shardHits =
                        shardResult.asMap().getList(HITS_ARRAY);
                    for (JsonObject obj: shardHits) {
                        JsonMap hit = obj.asMap();
                        String chatId = hit.getOrNull(MESSAGE_CHAT_ID);
                        if (chatId == null) {
                            continue;
                        }
                        Chat chat = chats.get(chatId);
                        if (chat == null) {
                            continue;
                        }
                        Message message = new Message(chat, hit);
                        chat.addMessage(message);
                        JsonList mergedList = hit.getListOrNull("merged_docs");
                        if (mergedList != null) {
                            for (JsonObject merged: mergedList) {
                                message = new Message(chat, merged.asMap());
                                chat.addMessage(message);
                            }
                        }
                    }
                }
                HttpEntity response;
//                if (context.proto()) {
//                    response = generateProtoResponse(chatMap);
//                } else {
                response = generateJsonResponse(chats);
//                }
                session.response(HttpStatus.SC_OK, response);
            } catch (Exception e) {
                failed(e);
            }
        }

        private HttpEntity generateJsonResponse(
            final Map<String, Chat> chatMap)
            throws IOException, JsonException
        {
            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = context.context().jsonType.create(sbw)) {
                writer.startArray();
                if (context.context().groupByChat) {
                    List<Chat> chatList = new ArrayList<>(chatMap.values());
                    Collections.sort(chatList);
                    int i = 0;
                    for (Chat chat: chatList) {
                        if (i++ >= context.context().maxChats) {
                            break;
                        }
                        if (chat.messages.size() > 0) {
                            chat.writeJson(writer, context.context().length);
                        }
                    }
                } else {
                    List<Message> allMessages = new ArrayList<>();
                    for (Chat chat: chatMap.values()) {
                        allMessages.addAll(chat.messages());
                        Collections.sort(allMessages);
                    }
                    int length = Math.min(
                        context.context().length,
                        allMessages.size());
                    for (int i = 0; i < length; i++) {
                        allMessages.get(i).writeJson(writer, true);
                    }
                }
                writer.endArray();
            }
            return new NStringEntity(
                sbw.toString(),
                ContentType.APPLICATION_JSON
                    .withCharset(session.acceptedCharset()));
        }
    }

    private static class Chat implements Comparable<Chat> {
        private final String chatId;
        private final String name;
        private final String description;
        private final List<Message> messages;
        private long maxTs = -1;

        Chat(final JsonMap json) throws JsonException {
            chatId = json.getOrNull(CHAT_ID);
            name = json.getOrNull(CHAT_NAME);
            description = json.getOrNull(CHAT_DESCRIPTION);
            messages = new ArrayList<>();
        }

        public void addMessage(final Message message) {
            if (maxTs < message.timestamp) {
                maxTs = message.timestamp;
            }
            messages.add(message);
        }

        public List<Message> messages() {
            return messages;
        }

        public void writeJson(final JsonWriter writer, final int len)
            throws JsonException, IOException
        {
            writer.startObject();

            writer.key(CHAT_ID);
            writer.value(chatId);

            writer.key(CHAT_NAME);
            writer.value(name);

            writer.key(CHAT_DESCRIPTION);
            writer.value(description);

            writer.key(TOP_MESSAGES);
            writer.startArray();
            Collections.sort(messages);
            int maxLen = Math.min(len, messages.size());
            for (int i = 0; i < maxLen; i++) {
                messages.get(i).writeJson(writer, false);
            }
            writer.endArray();
            writer.endObject();
        }

        public void writeInfoJson(final JsonWriter writer)
            throws JsonException, IOException
        {
            writer.startObject();

            writer.key(CHAT_ID);
            writer.value(chatId);

            writer.key(CHAT_NAME);
            writer.value(name);

            writer.key(CHAT_DESCRIPTION);
            writer.value(description);

            writer.endObject();
        }

        @Override
        public int hashCode() {
            return Long.hashCode(maxTs);
        }

        @Override
        public boolean equals(final Object o) {
            if (o instanceof Chat) {
                Chat other = (Chat) o;
                return other.maxTs == maxTs;
            }
            return false;
        }

        @Override
        public int compareTo(final Chat o) {
            return Long.compare(o.maxTs, maxTs);
        }
    }

    private static class Message implements Comparable<Message> {
//        private final String messageId;
        private TOutMessage message;
        private final long seqNo;
        private final long timestamp;
        private final Chat chat;

        Message(
            final Chat chat,
            final JsonMap json)
            throws JsonException
        {
            this.chat = chat;
//            messageId = json.getString(MESSAGE_ID);
            String seqNoStr = json.get(MESSAGE_SEQ_NO).asStringOrNull();
            long seqNo = 0;
            if (seqNoStr != null) {
                try {
                    seqNo = Long.parseLong(seqNoStr);
                } catch (Exception e) {
                    seqNo = 0;
                }
            }
            this.seqNo = seqNo;
            String tsNoStr = json.get(MESSAGE_TIMESTAMP).asStringOrNull();
            long timestamp = 0;
            if (tsNoStr != null) {
                try {
                    timestamp = Long.parseLong(tsNoStr);
                } catch (Exception e) {
                    timestamp = 0;
                }
            }
            this.timestamp = timestamp;
            String data = json.get(MESSAGE_DATA).asStringOrNull();
            if (data != null) {
                byte[] bytes = UnhexStrings.unhex(data);
                try {
                    message = TOutMessage.parseFrom(bytes);
                } catch (InvalidProtocolBufferException e) {
                    throw new JsonException(e);
                }
                String type = json.getString(TYPE, "");
                if (type.equals("delete_message")) {
                    TOutMessage.Builder builder = message.toBuilder();
                    builder
                        .getServerMessageBuilder()
                            .getServerMessageInfoBuilder()
                                .setDeleted(true);
                    message = builder.build();
                }
            } else {
                message = null;
            }
        }

        @Override
        public int hashCode() {
            return Long.hashCode(seqNo);
        }

        @Override
        public boolean equals(final Object o) {
            if (o instanceof Message) {
                Message other = (Message) o;
                return other.seqNo == seqNo;
            }
            return false;
        }

        @Override
        public int compareTo(final Message o) {
            return Long.compare(o.timestamp, timestamp);
        }

        public void writeJson(
            final JsonWriter writer,
            final boolean writeChatInfo)
            throws JsonException, IOException
        {
            if (writeChatInfo) {
                writer.startObject();
                writer.key("chat_info");
                chat.writeInfoJson(writer);
                writer.key(MESSAGE_DATA);
                ProtoUtils.protoToJson(message).writeValue(writer);
                writer.endObject();
            } else {
                ProtoUtils.protoToJson(message).writeValue(writer);
            }
        }
    }

    private static class INum {
        private SearchMapShard shard;
        private User user;

        INum(final SearchMapShard shard, final User user) {
            this.shard = shard;
            this.user = user;
        }
    }
}

