package ru.yandex.search.messenger.proxy.chatmedia;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import NMessengerProtocol.Message.TOutMessage;
import com.google.protobuf.InvalidProtocolBufferException;
import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.base64.Base64Encoder;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.FilterFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.logger.SearchProxyAccessLoggerConfigDefaults;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.ps.search.messenger.MessageFields;
import ru.yandex.ps.search.messenger.UserFields;
import ru.yandex.search.messenger.UserClearedChats;
import ru.yandex.search.messenger.proxy.Moxy;
import ru.yandex.search.messenger.proxy.SimpleJsonParser;
import ru.yandex.search.messenger.proxy.config.ImmutableMoxyConfig;
import ru.yandex.search.messenger.proxy.suggest.rules.ProtoUtils;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.proxy.universal.PlainUniversalSearchProxyRequestContext;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;
import ru.yandex.util.string.UnhexStrings;

public class ChatMediaHandler implements ProxyRequestHandler {
    private static final JsonObject EMPTY_RESULT;
    static {
        JsonMap map = new JsonMap(BasicContainerFactory.INSTANCE);
        map.put("hitsArray", new JsonList(BasicContainerFactory.INSTANCE));
        map.put("hitsCount", new JsonLong(0L));
        EMPTY_RESULT = map;
    }

    public static final int DEFAULT_LENGTH = 10;
    public static final String GET = "get";
    public static final String PREFIX = "prefix";
    public static final String SERVICE = "service";
    public static final String ZERO = "0";
    public static final String TEXT = "text";
    public static final String HITS_ARRAY = "hitsArray";
    public static final String CHAT_DATA = "chat_data";

    public static final CollectionParser<
        String,
        Set<String>,
        Exception>
        SET_PARSER = new CollectionParser<>(
            NonEmptyValidator.INSTANCE,
            LinkedHashSet::new);
    public static final Set<String> DEFAULT_MESSAGE_TYPES =
        Collections.unmodifiableSet(new LinkedHashSet<>(
            Arrays.asList(
                new String[]{
                    "voice_message",
                    "image_message",
                    "gallery_message"})));
    public static final Set<String> DEFAULT_EXCLUDE_MODERATION =
        Collections.unmodifiableSet(
            new LinkedHashSet<>(
                Arrays.asList(
                    new String[]{
                        "DELETE",
                        "HIDE"})));
    public static final boolean ALLOW_LAGGING_HOSTS = true;

    private final Moxy moxy;
    private final String messagesService;
    private final User chatsUser;
    private final SimpleJsonParser jsonParser;

    public ChatMediaHandler(final Moxy moxy) {
        this.moxy = moxy;
        messagesService = moxy.config().messagesService();
        this.chatsUser = new User(moxy.config().usersService(), new LongPrefix(0L));
        this.jsonParser = new SimpleJsonParser();
    }

    @Override
    public void handle(final ProxySession session) throws HttpException {
        ResolveUserSettingsContext userSettingsContext
            = new ResolveUserSettingsContext(session, moxy);

        if (userSettingsContext.guid() != null) {
            StringBuilder textSb = new StringBuilder();
            textSb.append("id:");
            textSb.append(ru.yandex.search.messenger.UserFields.id(userSettingsContext.guid(), "0"));
            textSb.append(" AND ");
            textSb.append(UserFields.CLEARED_CHATS.global());
            textSb.append(":");
            textSb.append(userSettingsContext.chatId());
            textSb.append("*");

            QueryConstructor qc = new QueryConstructor("/search?media_user_settings");
            qc.append("prefix", 0);
            qc.append("service", moxy.config().usersService());
            qc.append("length", "1");
            qc.append("text", textSb.toString());
            qc.append("get", UserFields.CLEARED_CHATS.stored());

            moxy.sequentialRequest(
                session,
                userSettingsContext,
                new BasicAsyncRequestProducerGenerator(qc.toString()),
                userSettingsContext.failoverDelay(),
                userSettingsContext.localityShuffle(),
                JsonAsyncTypesafeDomConsumerFactory.OK,
                session.listener().adjustContextGenerator(
                    userSettingsContext.client().httpClientContextGenerator()),
                new UserSettingsCallback(userSettingsContext));
        } else {
            handle(session, UserClearedChats.DEFAULT_VALUE);
        }
    }

    private class UserSettingsCallback extends AbstractProxySessionCallback<JsonObject> {
        private final ResolveUserSettingsContext context;

        public UserSettingsCallback(final ResolveUserSettingsContext context) {
            super(context.session());

            this.context = context;
        }

        @Override
        public void completed(final JsonObject jsonObject) {
            try {
                Long clearHistoryTs = UserClearedChats.DEFAULT_VALUE;
                JsonList hits = jsonObject.asMap().getList("hitsArray");
                if (hits.size() > 0) {
                    JsonMap map = hits.get(0).asMap();
                    Map<String, Long> cleared =
                        map.get(
                            UserFields.CLEARED_CHATS.stored(),
                            Collections.emptyMap(),
                            UserClearedChats.PARSER);
                    clearHistoryTs =
                        cleared.getOrDefault(context.chatId(), UserClearedChats.DEFAULT_VALUE);
                }

                handle(session, clearHistoryTs);
            } catch (JsonException | HttpException e) {
                failed(e);
            }
        }
    }
    private void handle(
        final ProxySession session,
        final Long clearTs)
        throws HttpException
    {
        final ChatMediaRequestContext mediaContext =
            new ChatMediaRequestContext(this, session, moxy.config(), clearTs);

        final PrevRequest prevContext = new PrevRequest(mediaContext);
        final NextRequest nextContext = new NextRequest(mediaContext);

        final DoubleFutureCallback<JsonObject, JsonObject> doubleCallback =
            new DoubleFutureCallback<>(
                new PrevNextMergeCallback(session, mediaContext));

        if (clearTs > 0 && clearTs >= mediaContext.near()){
            doubleCallback.first().completed(EMPTY_RESULT);
        } else {
            sendRequest(
                new PrevGuessRequest(mediaContext),
                session,
                new GuessCallback(
                    session,
                    doubleCallback.first(),
                    prevContext,
                    //new PrevAllFetchAndFilterRequest(mediaContext),
                    mediaContext.prev() + 1));
        }

        sendRequest(
            new NextGuessRequest(mediaContext),
            session,
            new GuessCallback(
                session,
                doubleCallback.second(),
                //new NextAllFetchAndFilterRequest(mediaContext),
                nextContext,
                mediaContext.next() + 1));
    }

    private class ResolveUserSettingsContext implements UniversalSearchProxyRequestContext {
        private final boolean localityShuffle;
        private final boolean allowLaggingHosts;
        private final long failoverDelay;
        private final AsyncClient client;
        private final ProxySession session;
        private final String guid;
        private final String chatId;

        public ResolveUserSettingsContext(
            final ProxySession session,
            final Moxy moxy)
            throws BadRequestException
        {
            this.session = session;
            this.client = moxy.searchClient().adjust(session.context());

            this.guid = session.params().getString("user-id", null);
            this.chatId = session.params().getString("chat-id", null);

            ImmutableMoxyConfig config = moxy.config();
            this.localityShuffle = session.params().getBoolean(
                "locality-shuffle",
                config.topPostsLocalityShuffle());
            this.failoverDelay = session.params().getLong(
                "failover-delay",
                config.topPostsFailoverDelay());
            this.allowLaggingHosts = session.params().getBoolean(
                "allow-lagging-hosts",
                ChatMediaHandler.ALLOW_LAGGING_HOSTS);
        }

        public String chatId() {
            return chatId;
        }

        public String guid() {
            return guid;
        }

        @Override
        public User user() {
            return chatsUser;
        }

        @Override
        public Long minPos() {
            return null;
        }

        @Override
        public AbstractAsyncClient<?> client() {
            return client;
        }

        @Override
        public Logger logger() {
            return session.logger();
        }

        @Override
        public long lagTolerance() {
            return allowLaggingHosts ? Long.MAX_VALUE : 0L;
        }

        public boolean localityShuffle() {
            return localityShuffle;
        }

        public long failoverDelay() {
            return failoverDelay;
        }

        public ProxySession session() {
            return session;
        }
    }

    private class GuessCallback extends FilterFutureCallback<JsonObject> {
        private final RequestContext nextContext;
        private final int limit;
        private final ProxySession session;
        private final FutureCallback<JsonObject> callback;

        public GuessCallback(
            final ProxySession session,
            final FutureCallback<JsonObject> callback,
            final RequestContext nextContext,
            final int limit)
        {
            super(callback);
            this.nextContext = nextContext;
            this.limit = limit;
            this.session = session;
            this.callback = callback;
        }

        @Override
        public void completed(final JsonObject result) {
            try {
                JsonList list = result.asMap().getList("hitsArray");
                if (list.size() < limit) {
                    session.logger().info(
                        "Last week hit missed, trying full scan " + list.size() + "/" + limit);
                    sendRequest(nextContext, session, callback);
                } else {
                    callback.completed(result);
                }
            } catch (JsonException | HttpException je) {
                failed(je);
            }
        }
    }

    public void sendRequest(
        final RequestContext context,
        final ProxySession session,
        final FutureCallback<JsonObject> callback)
        throws HttpException
    {
        AsyncClient client = moxy.searchClient().adjust(session.context());
        UniversalSearchProxyRequestContext requestContext =
            new PlainUniversalSearchProxyRequestContext(
                context.user(),
                null,
                context.allowLaggingHosts(),
                client,
                session.logger());
        QueryConstructor query = context.query();
        query.append(SERVICE, context.service());
        moxy.sequentialRequest(
            session,
            requestContext,
            new BasicAsyncRequestProducerGenerator(query.toString()),
            context.failoverDelay(),
            context.localityShuffle(),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.listener().createContextGeneratorFor(client),
            callback);
    }

    @Override
    public String toString() {
        return "Commentator chats info handler: "
            + "https://wiki.yandex-team.ru/ps/documentation/"
            + "moxy#cmntinfo";
    }

    private class PrevNextMergeCallback
        extends AbstractProxySessionCallback<Map.Entry<JsonObject, JsonObject>>
    {
        private final ChatMediaRequestContext mediaContext;

        PrevNextMergeCallback(
            final ProxySession session,
            final ChatMediaRequestContext mediaContext)
            throws HttpException
        {
            super(session);
            this.mediaContext = mediaContext;
        }

        @Override
        public void completed(final Map.Entry<JsonObject, JsonObject> results) {
            long startTime = System.currentTimeMillis();

            try {
                ArrayList<Message> messages = new ArrayList<>();
//                HashSet<Message> dedupSet = new HashSet<>();

                JsonMap prev = results.getKey().asMap();
                JsonList prevHits = prev.getList("hitsArray");
                boolean hasMorePrev = false;
                boolean hasMoreNext = false;
                if (prevHits.size() > mediaContext.prev()) {
                    hasMorePrev = true;
                }

                int maxPrev = Math.min(prevHits.size(), mediaContext.prev());
                int prevFound = 0;
                for (int i = maxPrev - 1; i >= 0; i--) {
                    JsonMap hit = prevHits.get(i).asMap();
                    Message msg = new Message(hit, jsonParser);
                    messages.add(msg);
                    prevFound++;
                }

                JsonMap next = results.getValue().asMap();
                JsonList nextHits = next.getList("hitsArray");

                int nextFound = 0;
                if (nextHits.size() > 0) {
                    int maxNext = Math.min(mediaContext.next(), nextHits.size());
                    int i;
                    for (i = 0; i < maxNext; i++) {
                        Message msg = new Message(nextHits.get(i).asMap(), jsonParser);
                        if (msg.timestamp == mediaContext.near()) {
                            maxNext = Math.min(mediaContext.next() + 1, nextHits.size());
                        } else {
                            nextFound++;
                        }
                        messages.add(msg);
                    }
                    if (nextHits.size() > maxNext) {
                        hasMoreNext = true;
                    }
                }

                session.connection().setSessionInfo(
                    SearchProxyAccessLoggerConfigDefaults.HITS_COUNT,
                    Long.toString(messages.size()));

                StringBuilderWriter sbw = new StringBuilderWriter();
                try (JsonWriter writer = mediaContext.jsonType().create(sbw)) {
                    writer.startObject();
                    writer.key("info");
                    writer.startObject();
                    writer.key("prev");
                    writer.value(prevFound);
                    writer.key("next");
                    writer.value(nextFound);
                    writer.key("has_prev");
                    writer.value(hasMorePrev);
                    writer.key("has_next");
                    writer.value(hasMoreNext);
                    writer.endObject();
                    writer.key("messages");
                    writer.startArray();
                    for (Message msg: messages) {
                        msg.writeJson(writer, mediaContext);
                    }
                    writer.endArray();
                    writer.endObject();
                }
                session.logger().info("Total chats: " + messages.size()
                    + ", time: " + (System.currentTimeMillis() - startTime));
                session.response(HttpStatus.SC_OK, sbw.toString());
            } catch (IOException | JsonException e) {
                session.logger().log(Level.SEVERE, "JsonParse exception", e);
                failed(e);
            } finally {
                session.logger().info("Callback time: "
                    + (System.currentTimeMillis() - startTime));
            }
        }
    }

    private static class Message {
        private final long timestamp;
        private final SimpleJsonParser jsonParser;
        private List<MessagePart> parts;

        Message(final JsonMap json, final SimpleJsonParser jsonParser) throws JsonException {
            this.timestamp = json.getLong("message_timestamp", 0L);
            this.jsonParser = jsonParser;
            parts = new ArrayList<>();
            parts.add(new MessagePart(json));
            JsonList mergedDocsList = json.getListOrNull("merged_docs");
            if (mergedDocsList != null) {
                for (JsonObject mergedDoc: mergedDocsList) {
                    JsonMap map = mergedDoc.asMapOrNull();
                    if (map != null) {
                        parts.add(new MessagePart(map));
                    }
                }
                Collections.sort(parts);
            }
        }

        public void writeJson(
            final JsonWriter writer,
            final ChatMediaRequestContext context)
            throws IOException, JsonException
        {
            if (parts.isEmpty()) {
                return;
            }
            JsonMap json = parts.get(0).json();
            writer.startObject();
            writer.key("timestamp");
            writer.value(timestamp);
            if (context.printData()) {
                writer.key("data");
                writer.value(
                    formatData(
                        json.getString("message_data"),
                        context));
            }

            for (String get: context.get()) {
                writer.key(get);
                if (MessageFields.LINKS.stored().equals(get)) {
                    writer.startArray();
                    for (MessagePart part: parts) {
                        String[] urls = part.json().getString(get, "").split("\n");
                        for (String url: urls) {
                            if (!url.isEmpty()) {
                                writer.value(url);
                            }
                        }
                    }
                    writer.endArray();
                } else if (MessageFields.RCA_DATA.stored().equals(get)) {

                    // temporarily for transition https://st.yandex-team.ru/PS-3816

                    JsonObject rcaData = JsonNull.INSTANCE;

                    String rcaDataStr = json.getString(get, null);

                    if (rcaDataStr != null && !rcaDataStr.isEmpty()) {
                        JsonObject rcaDataObj = jsonParser.parse(rcaDataStr);
                        if (rcaDataObj instanceof JsonList) {
                            JsonList rcaDataList = rcaDataObj.asList();
                            if (context.multipleLinks()) {
                                rcaData = rcaDataList;
                            } else if (!rcaDataList.isEmpty()) {
                                rcaData = rcaDataList.get(0);
                            }
                        } else if (rcaDataObj instanceof JsonMap) {
                            JsonMap rcaDataMap = rcaDataObj.asMap();
                            if (context.multipleLinks()) {
                                rcaData = new JsonList(BasicContainerFactory.INSTANCE);
                                rcaData.asList().add(rcaDataMap);
                            } else {
                                rcaData = rcaDataMap;
                            }
                        }
                    }
                    if (rcaData == JsonNull.INSTANCE) {
                        writer.value(rcaData);
                    } else {
                        writer.value(JsonType.NORMAL.toString(rcaData));
                    }
                } else {
                    writer.value(json.get(get));
                }
            }
            writer.endObject();
        }

        private JsonObject formatData(
            final String data,
            final ChatMediaRequestContext context)
        {
            if (data != null) {
                byte[] bytes = UnhexStrings.unhex(data);
                if (context.base64()) {
                    Base64Encoder encoder = new Base64Encoder();
                    encoder.process(bytes);
                    return new JsonString(encoder.toString());
                } else {
                    try {
                        TOutMessage message = TOutMessage.parseFrom(bytes);
                        return ProtoUtils.protoToJson(message);
                    } catch (InvalidProtocolBufferException e) {
                        return JsonNull.INSTANCE;
                    }
                }
            } else {
                return JsonNull.INSTANCE;
            }
        }
    }

    private static class MessagePart implements Comparable<MessagePart> {
        private final JsonMap json;
        private final int hid;

        MessagePart(final JsonMap json) throws JsonException {
            this.json = json;
            hid = json.getInt("message_hid", 0);
        }

        public JsonMap json() {
            return json;
        }

        public int hid() {
            return hid;
        }

        @Override
        public int compareTo(final MessagePart other) {
            return Integer.compare(hid, other.hid);
        }
    }
}

